
local assets=
{
    Asset("ATLAS", "images/pickleheart.xml"),
    Asset("ANIM", "anim/pickleheart.zip"),
}

local SPAWN_DIST = 30

local trace = function() end

local function GetSpawnPoint(pt)

    local theta = math.random() * 2 * PI
    local radius = SPAWN_DIST

	local offset = FindWalkableOffset(pt, theta, radius, 12, true)
	if offset then
		return pt+offset
	end
end

local function SpawnOichan(inst)
    trace("pickleheart - SpawnOichan")

    local pt = Vector3(inst.Transform:GetWorldPosition())
    trace("    near", pt)
        
    local spawn_pt = GetSpawnPoint(pt)
    if spawn_pt then
        trace("    at", spawn_pt)
        local oichan = SpawnPrefab("oichan")
        if oichan then
            oichan.Physics:Teleport(spawn_pt:Get())
            oichan:FacePoint(pt.x, pt.y, pt.z)

            return oichan
        end

    else
        -- this is not fatal, they can try again in a new location by picking up the bone again
        trace("pickleheart - SpawnOichan: Couldn't find a suitable spawn point for oichan")
    end
end


local function StopRespawn(inst)
    trace("pickleheart - StopRespawn")
    if inst.respawntask then
        inst.respawntask:Cancel()
        inst.respawntask = nil
        inst.respawntime = nil
    end
end

local function RebindOichan(inst, oichan)
    oichan = oichan or TheSim:FindFirstEntityWithTag("oichan")
    if oichan then

        inst.AnimState:PlayAnimation("idle", true)
        inst.components.inventoryitem:ChangeImageName(inst.openEye)
        inst:ListenForEvent("death", function() inst:OnOichanDeath() end, oichan)

        if oichan.components.follower.leader ~= inst then
            oichan.components.follower:SetLeader(inst)
        end
        return true
    end
end

local function RespawnOichan(inst)
    trace("pickleheart - RespawnOichan")

    StopRespawn(inst)

    local oichan = TheSim:FindFirstEntityWithTag("oichan")
    if not oichan then
        oichan = SpawnOichan(inst)
    end
    RebindOichan(inst, oichan)
end

local function StartRespawn(inst, time)
    StopRespawn(inst)

    local respawntime = time or 0
    if respawntime then
        inst.respawntask = inst:DoTaskInTime(respawntime, function() RespawnOichan(inst) end)
        inst.respawntime = GetTime() + respawntime
        inst.AnimState:PlayAnimation("dead", true)
        inst.components.inventoryitem:ChangeImageName(inst.closedEye)
    end
end

local function OnOichanDeath(inst)
    StartRespawn(inst, TUNING.CHESTER_RESPAWN_TIME)
end

local function FixOichan(inst)
	inst.fixtask = nil
	--take an existing oichan if there is one
	if not RebindOichan(inst) then
        inst.AnimState:PlayAnimation("dead", true)
        inst.components.inventoryitem:ChangeImageName(inst.closedEye)
		
		if inst.components.inventoryitem.owner then
			local time_remaining = 0
			local time = GetTime()
			if inst.respawntime and inst.respawntime > time then
				time_remaining = inst.respawntime - time		
			end
			StartRespawn(inst, time_remaining)
		end
	end
end

local function OnPutInInventory(inst)
	if not inst.fixtask then
		inst.fixtask = inst:DoTaskInTime(1, function() FixOichan(inst) end)	
	end
end

local function OnSave(inst, data)
    trace("pickleheart - OnSave")
    local time = GetTime()
    if inst.respawntime and inst.respawntime > time then
        data.respawntimeremaining = inst.respawntime - time
    end
end


local function OnLoad(inst, data)
    if data and data.respawntimeremaining then
		inst.respawntime = data.respawntimeremaining + GetTime()
	end
end

local function GetStatus(inst)
    trace("smallbird - GetStatus")
    if inst.respawntask then
        return "WAITING"
    end
end


local function fn(Sim)
    local inst = CreateEntity()
    inst.entity:AddTransform()
    inst.entity:AddAnimState()
	
    inst.entity:AddMiniMapEntity()
    inst.MiniMapEntity:SetIcon("pickleheart.tex")

    inst:AddTag("pickleheart")
    inst:AddTag("irreplaceable")
	inst:AddTag("nonpotatable")

    MakeInventoryPhysics(inst)
	
    inst.entity:AddDynamicShadow()
    inst.DynamicShadow:SetSize( 1, .5 )
    
    inst.AnimState:SetBank("pickleheart")
    inst.AnimState:SetBuild("pickleheart")
    inst.AnimState:PlayAnimation("idle", true)

    inst:AddComponent("inventoryitem")
    inst.components.inventoryitem:SetOnPutInInventoryFn(OnPutInInventory)
    inst.components.inventoryitem.atlasname = "images/pickleheart.xml"
    inst.openEye = "pickleheart"
    inst.closedEye = "pickleheart"

    inst.components.inventoryitem:ChangeImageName(inst.openEye) 
	
    inst:AddComponent("inspectable")
    inst.components.inspectable.getstatus = GetStatus
	inst.components.inspectable:RecordViews()

    inst:AddComponent("leader")

    inst.OnLoad = OnLoad
    inst.OnSave = OnSave
    inst.OnOichanDeath = OnOichanDeath
    local light = inst.entity:AddLight()
    inst.Light:Enable(true)
	inst.Light:SetRadius(3)
    inst.Light:SetFalloff(1)
    inst.Light:SetIntensity(.7)
    inst.Light:SetColour(255/255,50/255,50/255)

	inst.fixtask = inst:DoTaskInTime(1, function() FixOichan(inst) end)

    return inst
end

return Prefab( "common/inventory/pickleheart", fn, assets) 
