-----------------------------------------------------------------------------------------------
-- Client Lua Script for MinimalistPlates
-- Copyright (c) NCsoft. All rights reserved
-----------------------------------------------------------------------------------------------

require "Window"
require "ChallengesLib"
require "Unit"
require "GameLib"
require "Apollo"
require "PathMission"
require "Quest"
require "Episode"
require "math"
require "string"
require "DialogSys"
require "PublicEvent"
require "PublicEventObjective"
require "CommunicatorLib"
require "GroupLib"
require "PlayerPathLib"
require "GuildLib"
require "GuildTypeLib"

local MinimalistPlates = {}

-----------------------------------------------------------------------------------------------
-- Constants
-----------------------------------------------------------------------------------------------
local knTargetRange 		= 40000
local knNameplatePoolLimit 	= 100

local karSavedProperties =
{
	-- Visibility
	["nMaxRange"] = { default=70.0, nControlType=0 },
	["nFriendlyCutoff"] = { default=30.0, nControlType=0 },
	["nHostileCutoff"] = { default=30.0, nControlType=0 },
	["bReposition"] = { default=false, nControlType=1, strControlName="Button_Reposition" },
	["bShowIcons"] = { default=true, nControlType=1, strControlName="Button_ShowIcons", fnCallback="RequestUpdateAllNameplateRewards" },
	["bShowLevels"] = { default=false, nControlType=1, strControlName="Button_ShowLevels" },
	["bShowTitles"] = { default=false, nControlType=1, strControlName="Button_ShowTitles" },
	["bShowArmor"] = { default=true, nControlType=1, strControlName="Button_ShowArmor" },
	["bShowHealthNumbers"] = { default=false, nControlType=1, strControlName="Button_ShowHealthNumbers" },
	["bShowHealthPercent"] = { default=false, nControlType=1, strControlName="Button_ShowHealthPercent" },
	["bShowShieldNumbers"] = { default=false, nControlType=1, strControlName="Button_ShowShieldNumbers" },
	["bShowShieldPercent"] = { default=false, nControlType=1, strControlName="Button_ShowShieldPercent" },
	["bShowQuestIcons"] = { default=true, nControlType=1, strControlName="Button_ShowQuestIcons" },
	["bShowChallengeIcons"] = { default=true, nControlType=1, strControlName="Button_ShowChallengeIcons" },
	["bShowPathIcons"] = { default=true, nControlType=1, strControlName="Button_ShowPathIcons" },
	
	-- Tank options
	["bShowThreat"] = { default=false, nControlType=1, strControlName="Button_ShowThreat" },
	
	-- Healer options
	["bShowCleanseIcon"] = { default=false, nControlType=1, strControlName="Button_ShowCleanseIcon" },
	["bHideFriendlyBars"] = { default=false, nControlType=1, strControlName="Button_HideFriendlyBars" },
	["bHideFriendlyBars_Shield"] = { default=false, nControlType=1, strControlName="Button_HideFriendlyBars_Shield" },
	
	["bShowTargetMarker"] = { default=true, nControlType=1, strControlName="Button_ShowTargetMarker" },
	["bShowTargetNames"] = { default=true, nControlType=1, strControlName="Button_ShowTargetNames" },
	["bShowTargetGuilds"] = { default=true, nControlType=1, strControlName="Button_ShowTargetGuilds" },
	["bShowTargetBars"] = { default=true, nControlType=1, strControlName="Button_ShowTargetBars" },
	["bShowTargetCastBars"] = { default=true, nControlType=1, strControlName="Button_ShowTargetCastBars" },
	
	["bShowPlayerNames"] = { default=false, nControlType=1, strControlName="Button_ShowPlayerNames" },
	["bShowPlayerGuilds"] = { default=false, nControlType=1, strControlName="Button_ShowPlayerGuilds" },
	["bShowPlayerBars"] = { default=false, nControlType=1, strControlName="Button_ShowPlayerBars" },
	["bShowPlayerBars_Combat"] = { default=false, nControlType=1, strControlName="Button_ShowPlayerBars_Combat" },
	["bShowPlayerCastBars"] = { default=false, nControlType=1, strControlName="Button_ShowPlayerCastBars" },
	
	["bShowNeutralNames"] = { default=true, nControlType=1, strControlName="Button_ShowNeutralNames" },
	["bShowNeutralGuilds"] = { default=true, nControlType=1, strControlName="Button_ShowNeutralGuilds" },
	["bShowNeutralBars"] = { default=false, nControlType=1, strControlName="Button_ShowNeutralBars" },
	["bShowNeutralCastBars"] = { default=false, nControlType=1, strControlName="Button_ShowNeutralCastBars" },
	
	["bShowFriendlyNames"] = { default=false, nControlType=1, strControlName="Button_ShowFriendlyNames" },
	["bShowFriendlyGuilds"] = { default=false, nControlType=1, strControlName="Button_ShowFriendlyGuilds" },
	["bShowFriendlyBars"] = { default=false, nControlType=1, strControlName="Button_ShowFriendlyBars" },
	["bShowFriendlyCastBars"] = { default=false, nControlType=1, strControlName="Button_ShowFriendlyCastBars" },
	
	["bShowFriendlyPlayerNames"] = { default=true, nControlType=1, strControlName="Button_ShowFriendlyPlayerNames" },
	["bShowFriendlyPlayerGuilds"] = { default=true, nControlType=1, strControlName="Button_ShowFriendlyPlayerGuilds" },
	["bShowFriendlyPlayerBars"] = { default=false, nControlType=1, strControlName="Button_ShowFriendlyPlayerBars" },
	["bShowFriendlyPlayerBars_Combat"] = { default=false, nControlType=1, strControlName="Button_ShowFriendlyPlayerBars_Combat" },
	["bShowFriendlyPlayerCastBars"] = { default=false, nControlType=1, strControlName="Button_ShowFriendlyPlayerCastBars" },
	
	["bShowHostileNames"] = { default=true, nControlType=1, strControlName="Button_ShowHostileNames" },
	["bShowHostileGuilds"] = { default=true, nControlType=1, strControlName="Button_ShowHostileGuilds" },
	["bShowHostileBars"] = { default=false, nControlType=1, strControlName="Button_ShowHostileBars" },
	["bShowHostileBars_Combat"] = { default=true, nControlType=1, strControlName="Button_ShowHostileBars_Combat" },
	["bShowHostileCastBars"] = { default=true, nControlType=1, strControlName="Button_ShowHostileCastBars" },
	
	["bShowHostilePlayerNames"] = { default=true, nControlType=1, strControlName="Button_ShowHostilePlayerNames" },
	["bShowHostilePlayerGuilds"] = { default=true, nControlType=1, strControlName="Button_ShowHostilePlayerGuilds" },
	["bShowHostilePlayerBars"] = { default=false, nControlType=1, strControlName="Button_ShowHostilePlayerBars" },
	["bShowHostilePlayerBars_Combat"] = { default=true, nControlType=1, strControlName="Button_ShowHostilePlayerBars_Combat" },
	["bShowHostilePlayerCastBars"] = { default=true, nControlType=1, strControlName="Button_ShowHostilePlayerCastBars" },
	
	["bShowPlayerPetNames"] = { default=false, nControlType=1, strControlName="Button_ShowPlayerPet" },
	["bShowFriendlyPetNames"] = { default=false, nControlType=1, strControlName="Button_ShowFriendlyPet" },
	["bShowHostilePetNames"] = { default=false, nControlType=1, strControlName="Button_ShowHostilePet" },
	
	-- Colors
	["kcrOpacity"] = { default="FF", nControlType=0 },
	["kcrBackground"] = { default="8E8F8D", nControlType=3, strControlName="Edit_Background", fnCallback="UpdateAllNameplateStyles" },
	["kcrAbsorbBar"] = { default="FEB308", nControlType=3, strControlName="Edit_AbsorbBar", fnCallback="UpdateAllNameplateStyles" },
	["kcrShieldBar"] = { default="75E4EC", nControlType=3, strControlName="Edit_ShieldBar", fnCallback="UpdateAllNameplateStyles" },
	["kcrVulnerableBar"] = { default="7E00FF", nControlType=3, strControlName="Edit_VulnerableBar" },
	["kcrVulnerableTimerBar"] = { default="E583C3", nControlType=3, strControlName="Edit_VulnerableTimerBar", fnCallback="UpdateAllNameplateStyles" },
	["kcrFriendlyCutoff"] = { default="E4FF00", nControlType=3, strControlName="Edit_FriendlyCutoff" },
	["kcrHostileCutoff"] = { default="FF00FC", nControlType=3, strControlName="Edit_HostileCutoff" },
	["kcrCastBar"] = { default="FEB308", nControlType=3, strControlName="Edit_CastBar", fnCallback="UpdateAllNameplateStyles" },
	["kcrTargetIndicator"] = { default="FFFFFF", nControlType=3, strControlName="Edit_TargetIndicator", fnCallback="UpdateAllNameplateStyles" },
	["kcrNeutralName"] = { default="FFF569", nControlType=3, strControlName="Edit_NeutralName" },
	["kcrNeutralHealthBar"] = { default="F3D829", nControlType=3, strControlName="Edit_NeutralHealthBar" },
	["kcrFriendlyName"] = { default="76CD26", nControlType=3, strControlName="Edit_FriendlyName" },
	["kcrFriendlyHealthBar"] = { default="15B01A", nControlType=3, strControlName="Edit_FriendlyHealthBar" },
	["kcrFriendlyPlayerName"] = { default="FFFFFF", nControlType=3, strControlName="Edit_FriendlyPlayerName" },
	["kcrFriendlyPlayerHealthBar"] = { default="15B01A", nControlType=3, strControlName="Edit_FriendlyPlayerHealthBar" },
	["kcrPartyPlayerName"] = { default="43C8F3", nControlType=3, strControlName="Edit_PartyPlayerName" },
	["kcrPartyPlayerHealthBar"] = { default="15B01A", nControlType=3, strControlName="Edit_PartyPlayerHealthBar" },
	["kcrHostileName"] = { default="D9544D", nControlType=3, strControlName="Edit_HostileName" },
	["kcrHostileHealthBar"] = { default="E50000", nControlType=3, strControlName="Edit_HostileHealthBar" },
	["kcrHostileLowThreat"] = { default="F49500", nControlType=3, strControlName="Edit_HostileLowThreat" },
	["kcrHostileUnflaggedPlayerName"] = { default="FFFFFF", nControlType=3, strControlName="Edit_HostileUnflaggedPlayerName" },
	["kcrHostileUnflaggedPlayerHealthBar"] = { default="E50000", nControlType=3, strControlName="Edit_HostileUnflaggedPlayerHealthBar" },
	["kcrHostilePlayerName"] = { default="D9544D", nControlType=3, strControlName="Edit_HostilePlayerName" },
	["kcrHostilePlayerHealthBar"] = { default="E50000", nControlType=3, strControlName="Edit_HostilePlayerHealthBar" },
	
	["bShowHostileClassColors"] = { default=false, nControlType=2, strControlName="Button_ShowHostileClassColors" },
	["bShowFriendlyClassColors"] = { default=false, nControlType=2, strControlName="Button_ShowFriendlyClassColors" },
	
	["kcrEngineer"] = { default="EFAB48", nControlType=3, strControlName="Edit_Engineer" },
	["kcrEsper"] = { default="1591DB", nControlType=3, strControlName="Edit_Esper" },
	["kcrMedic"] = { default="FFE757", nControlType=3, strControlName="Edit_Medic" },
	["kcrSpellslinger"] = { default="98C723", nControlType=3, strControlName="Edit_Spellslinger" },
	["kcrStalker"] = { default="D23EF4", nControlType=3, strControlName="Edit_Stalker" },
	["kcrWarrior"] = { default="F54F4F", nControlType=3, strControlName="Edit_Warrior" },
	
	-- Style
	["kcrWidth"] = { default="90", nControlType=0 },
	["kcrShieldHeight"] = { default="4", nControlType=0 },
	["kcrHealthHeight"] = { default="11", nControlType=0 },
	
	["kcrNameFont"] = { default="CRB_Header10_O", nControlType=0 },
	["kcrGuildFont"] = { default="CRB_Header9_O", nControlType=0 },
}

local karClassId = {
	[1] = "Warrior",
	[2] = "Engineer",
	[3] = "Esper",
	[4] = "Medic",
	[5] = "Stalker",
	[7] = "Spellslinger",
}

local karDispositionId = {
	[0] = "Hostile",
	[1] = "Neutral",
	[2] = "Friendly",
	[3] = "Unknown",
}

local karPathId = {
	[0] = "Soldier",
	[1] = "Settler",
	[2] = "Scientist",
	[3] = "Explorer",
}

local karFonts = {
	["CRB_Header9_O"] = 13,
	["CRB_Header10_O"] = 14,
	["CRB_Header11_O"] = 15,
	["CRB_Pixel_O"] = 12,
	["CRB_FloaterSmall"] = 21,
	["CRB_InterfaceSmall_O"] = 14,
	["CRB_InterfaceMedium_O"] = 15,
	["CRB_InterfaceLarge_O"] = 17,
}

function MinimalistPlates:Init()
	Apollo.RegisterAddon(self, true)
end

-----------------------------------------------------------------------------------------------
-- OnLoad
-----------------------------------------------------------------------------------------------
function MinimalistPlates:OnLoad()

	self.arWindowPool = {}
	self.arUnit2Nameplate = {}
	self.arWnd2Nameplate = {}
	self.bRedrawRewardIcons = false
	self.tActiveChallenges = ChallengesLib.GetActiveChallengeList()
	
	-- Register events
	Apollo.RegisterEventHandler("UnitCreated", 					"OnUnitCreated", self)
	Apollo.RegisterEventHandler("UnitDestroyed", 				"OnUnitDestroyed", self)
	Apollo.RegisterEventHandler("VarChange_FrameCount", 		"OnFrame", self)
	
	Apollo.RegisterEventHandler("TargetUnitChanged", 			"OnTargetUnitChanged", self)
	Apollo.RegisterEventHandler("UnitGibbed",					"OnUnitGibbed", self)
	
	local tRewardUpdateEvents = {
		"QuestObjectiveUpdated", "QuestStateChanged", "ChallengeAbandon", "ChallengeLeftArea",
		"ChallengeFailTime", "ChallengeFailArea", "ChallengeActivate", "ChallengeCompleted",
		"ChallengeFailGeneric", "PublicEventObjectiveUpdate", "PublicEventUnitUpdate",
		"PlayerPathMissionUpdate", "FriendshipAdd", "FriendshipPostRemove", "FriendshipUpdate"
	}

	for _,str in ipairs(tRewardUpdateEvents) do
		Apollo.RegisterEventHandler(str, "RequestUpdateAllNameplateRewards", self)
	end
	
	-- Register timers
	Apollo.RegisterTimerHandler("BuffTimer", "OnBuffTimer", self)
	Apollo.CreateTimer("BuffTimer", 0.25, true)
	
	Apollo.RegisterTimerHandler("VisibilityTimer", "OnVisibilityTimer", self)
	Apollo.CreateTimer("VisibilityTimer", 1, true)
	
	-- Load sprites
	Apollo.LoadSprites("MinimalistPlates_Sprites.xml")

	-- Load forms
	self.xmlDoc = XmlDoc.CreateFromFile("MinimalistPlates.xml")
	
	self.wndMain = Apollo.LoadForm(self.xmlDoc, "Options", nil, self)
	self.wndOptionsMain = Apollo.LoadForm(self.xmlDoc, "OptionsMain", self.wndMain:FindChild("Content_Main"), self)
	self.wndOptionsColors = Apollo.LoadForm(self.xmlDoc, "OptionsColors", self.wndMain:FindChild("Content_Colors"), self)
	self.wndOptionsStyle = Apollo.LoadForm(self.xmlDoc, "OptionsStyle", self.wndMain:FindChild("Content_Style"), self)
	self.wndMain:Show(false)
	self.wndMain:FindChild("Content_Main"):Show(true)
	self.wndMain:FindChild("Content_Colors"):Show(false)
	self.wndMain:FindChild("Content_Style"):Show(false)
	self.wndMain:FindChild("Button_Main"):SetCheck(true)
	
	-- Set defaults
	for property,tData in pairs(karSavedProperties) do
		if self[property] == nil then
			self[property] = tData.default
		end
		if tData.nControlType == 1 then
			local wndControl = self.wndMain:FindChild(tData.strControlName)
			if wndControl ~= nil then
				wndControl:SetData(property)
			end
		end
		if tData.nControlType == 2 or tData.nControlType == 3 then
			local wndControl = self.wndOptionsColors:FindChild(tData.strControlName)
			if wndControl ~= nil then
				wndControl:SetData(property)
			end
		end
	end
	self.bUseOcclusion = Apollo.GetConsoleVariable("ui.occludeNameplatePositions")
	
	-- Perspective Plates
	self.bPerspectivePlates = false
	if Apollo.GetAddon("PerspectivePlates") ~= nil then
		self.bPerspectivePlates = true
	end
end

function MinimalistPlates:OnSave(eType)
	if eType ~= GameLib.CodeEnumAddonSaveLevel.Character then
		return
	end

	local tSave = {}
	for property,tData in pairs(karSavedProperties) do
		tSave[property] = self[property]
	end

	return tSave
end

function MinimalistPlates:OnRestore(eType, tSavedData)
	if eType ~= GameLib.CodeEnumAddonSaveLevel.Character then
		return
	end
	
	for property,tData in pairs(karSavedProperties) do
		if tSavedData[property] ~= nil then
			self[property] = tSavedData[property]
		end
	end
end

function MinimalistPlates:OnBuffTimer()
	if not self.bShowCleanseIcon then
		return
	end

	for idx, tNameplate in pairs(self.arUnit2Nameplate) do
		local unitOwner = tNameplate.unitOwner
		
		local bShow = false
		if tNameplate.bCheckBuffs and unitOwner ~= nil then
			for key, val in pairs(unitOwner:GetBuffs().arHarmful or {}) do
				if val["splEffect"]:GetClass() == 38 then
					bShow = true
					break
				end
			end
		end
		
		if tNameplate.wnd.cleanse:IsShown() ~= bShow then
			tNameplate.wnd.cleanse:Show(bShow)
			tNameplate.wnd.info:ArrangeChildrenHorz(0)
		end
	end
end

function MinimalistPlates:OnVisibilityTimer()
	self:UpdateAllNameplateVisibility()
end

function MinimalistPlates:RequestUpdateAllNameplateRewards()
	self.bRedrawRewardIcons = true
end

function MinimalistPlates:UpdateNameplateRewardInfo(tNameplate)
	local unitOwner = tNameplate.unitOwner
	local tRewardInfo = unitOwner:GetRewardInfo()
	
	local bShowQuest = false
	local bShowChallenge = false
	local bShowPath = false
	
	if tRewardInfo == nil or not self.bShowIcons then
		tNameplate.tRewards.bShowQuest = bShowQuest
		tNameplate.tRewards.bShowChallenge = bShowChallenge
		tNameplate.tRewards.bShowPath = bShowPath
		return
	end
	
	for i=1,#tRewardInfo do
		local ePathId = PlayerPathLib.GetPlayerPathType()
		local strPlayerPath = karPathId[ePathId]
		
		if tRewardInfo[i].strType == "Quest" then
			bShowQuest = true
		elseif tRewardInfo[i].strType == "Challenge" then
			if self.tActiveChallenges ~= nil then
				local tActiveChallenges = self.tActiveChallenges
				local nChallengeId = tRewardInfo[i].idChallenge
				
				local bIsActive = false
				if tActiveChallenges[nChallengeId] ~= nil then
					if tActiveChallenges[nChallengeId]:IsActivated() then
						bIsActive = true
					end
				end
				
				if bIsActive then
					bShowChallenge = true
				end
			end
		elseif tRewardInfo[i].strType == strPlayerPath then
			bShowPath = true
		end
	end
	
	tNameplate.tRewards.bShowQuest = bShowQuest
	tNameplate.tRewards.bShowChallenge = bShowChallenge
	tNameplate.tRewards.bShowPath = bShowPath
end

function MinimalistPlates:UpdateAllNameplateVisibility()
	for idx, tNameplate in pairs(self.arUnit2Nameplate) do
		self:UpdateNameplateVisibility(tNameplate)
		if self.bRedrawRewardIcons then
			self:UpdateNameplateRewardInfo(tNameplate)
		end
	end
	self.bRedrawRewardIcons = false
end

function MinimalistPlates:UpdateNameplateVisibility(tNameplate)
	local unitOwner = tNameplate.unitOwner
	local wndNameplate = tNameplate.wndNameplate
	local bIsMounted = unitOwner:IsMounted()
	local unitWindow = wndNameplate:GetUnit()
	
	if bIsMounted and unitWindow == unitOwner then
		wndNameplate:SetUnit(unitOwner:GetUnitMount(), 1)
	elseif not bIsMounted and unitWindow ~= unitOwner then
		wndNameplate:SetUnit(unitOwner, 1)
	end
	
	tNameplate.bOnScreen = wndNameplate:IsOnScreen()
	tNameplate.bOccluded = wndNameplate:IsOccluded()
	tNameplate.eDisposition = unitOwner:GetDispositionTo(self.unitPlayer)
	local bNewShow = self:HelperVerifyVisibility(tNameplate) and self:CheckDrawDistance(tNameplate)
	if bNewShow ~= tNameplate.bShow then
		wndNameplate:Show(bNewShow)
		tNameplate.bShow = bNewShow
	end
end

function MinimalistPlates:UpdateAllNameplateStyles()
	for idx, tNameplate in pairs(self.arUnit2Nameplate) do
		self:UpdateStyle(tNameplate)
	end
end

function MinimalistPlates:OnUnitCreated(unitNew)
	if unitNew == nil
		or not unitNew:IsValid()
		or (not unitNew:ShouldShowNamePlate() and not self:IsImportantNPC(unitNew))
		or unitNew:GetType() == "Collectible"
		or unitNew:GetType() == "PinataLoot" then
		-- Never have nameplates
		return
	end
	
	local idUnit = unitNew:GetId()
	if self.arUnit2Nameplate[idUnit] ~= nil and self.arUnit2Nameplate[idUnit].wndNameplate:IsValid() then
		return
	end
	
	local wnd = nil
	local wndReferences = nil
	if next(self.arWindowPool) ~= nil then
		local poolEntry = table.remove(self.arWindowPool)
		wnd = poolEntry[1]
		wndReferences = poolEntry[2]
	end
	
	if wnd == nil or not wnd:IsValid() then
		wnd = Apollo.LoadForm(self.xmlDoc, "NameplateNew", "InWorldHudStratum", self)
		wndReferences = nil
	end
	
	wnd:Show(false, true)
	wnd:SetUnit(unitNew, 1)
	
	local tNameplate =
	{
		unitOwner 		= unitNew,
		idUnit 			= idUnit,
		wndNameplate	= wnd,
		wndMeasure		= nil,
		bOnScreen 		= wnd:IsOnScreen(),
		bOccluded 		= wnd:IsOccluded(),
		--bSpeechBubble 	= false,
		bIsTarget 		= false,
		bGibbed			= false,
		nVulnerableTime = 0,
		eDisposition	= unitNew:GetDispositionTo(self.unitPlayer),
		bIsImportant	= self:IsImportantNPC(unitNew),
		bShow			= false,
		bCheckBuffs		= false,
		bPosCheck		= false,
		bIsOver 		= true,
		wnd				= wndReferences,
	}
	
	tNameplate.tRewards = {
		bShowQuest		= false,
		bShowChallenge	= false,
		bShowPath		= false,
	}
	
	if wndReferences == nil then
		tNameplate.wnd =
		{
			container = wnd:FindChild("NameplateNewContainer"),
			name = wnd:FindChild("Name"),
			nameContainer = wnd:FindChild("NameContainer"),
			guild = wnd:FindChild("Guild"),
			health = wnd:FindChild("HealthPlate"),
			healthBar = wnd:FindChild("HealthBar"),
			healthText = wnd:FindChild("HealthText"),
			shield = wnd:FindChild("ShieldPlate"),
			shieldBar = wnd:FindChild("ShieldBar"),
			shieldText = wnd:FindChild("ShieldText"),
			cast = wnd:FindChild("Cast"),
			castBar = wnd:FindChild("CastBar"),
			castName = wnd:FindChild("CastName"),
			absorb = wnd:FindChild("AbsorbBar"),
			vulnerable = wnd:FindChild("VulnerableBar"),
			rewards = wnd:FindChild("Rewards"),
			rewardQuest = wnd:FindChild("RewardQuest"),
			rewardChallenge = wnd:FindChild("RewardChallenge"),
			rewardPath = wnd:FindChild("RewardPath"),
			targetMarker = wnd:FindChild("TargetMarker"),
			info = wnd:FindChild("InfoPlate"),
			level = wnd:FindChild("LevelText"),
			armor = wnd:FindChild("Armor"),
			armorNumber = wnd:FindChild("ArmorNumber"),
			cleanse = wnd:FindChild("Cleanse"),
		}
	end
	
	self.arUnit2Nameplate[idUnit] = tNameplate
	self.arWnd2Nameplate[wnd:GetId()] = tNameplate
	
	self:UpdateNameplateRewardInfo(tNameplate)
	self:UpdateStyle(tNameplate)
	
	tNameplate.wnd.name:SetData(tNameplate.unitOwner)
	tNameplate.wnd.guild:SetData(tNameplate.unitOwner)
	tNameplate.wnd.absorb:SetData(tNameplate.unitOwner)
end

function MinimalistPlates:OnUnitDestroyed(unitOwner)
	local idUnit = unitOwner:GetId()
	if self.arUnit2Nameplate[idUnit] == nil then
		return
	end

	local tNameplate = self.arUnit2Nameplate[idUnit]
	local wndNameplate = tNameplate.wndNameplate

	self.arWnd2Nameplate[wndNameplate:GetId()] = nil
	if #self.arWindowPool < knNameplatePoolLimit then
		wndNameplate:Show(false, true)
		wndNameplate:SetUnit(nil)
		table.insert(self.arWindowPool, {wndNameplate, tNameplate.wnd})
	else
		wndNameplate:Destroy()
	end
	self.arUnit2Nameplate[idUnit] = nil
end

function MinimalistPlates:OnFrame()
	self.unitPlayer = GameLib.GetPlayerUnit()
	self.tActiveChallenges = ChallengesLib.GetActiveChallengeList()
	
	local fnDrawName = MinimalistPlates.DrawName
	local fnDrawGuild = MinimalistPlates.DrawGuild
	local fnDrawHealth = MinimalistPlates.DrawHealth
	local fnDrawShield = MinimalistPlates.DrawShield
	local fnDrawCast = MinimalistPlates.DrawCast
	local fnDrawVulnerable = MinimalistPlates.DrawVulnerable
	local fnDrawRewards = MinimalistPlates.DrawRewards
	local fnDrawTargeting = MinimalistPlates.DrawTargeting
	local fnDrawArmor = MinimalistPlates.DrawArmor
	local fnColorNameplate = MinimalistPlates.ColorNameplate
	
	for idx, tNameplate in pairs(self.arUnit2Nameplate) do
		if tNameplate.bShow then
			fnDrawName(self, tNameplate)
			fnDrawGuild(self, tNameplate)
		
			fnDrawHealth(self, tNameplate)
			fnDrawShield(self, tNameplate)
			fnDrawVulnerable(self, tNameplate)
			fnDrawRewards(self, tNameplate)
			fnDrawTargeting(self, tNameplate)
			fnDrawArmor(self, tNameplate)

			fnDrawCast(self, tNameplate)
			
			fnColorNameplate(self, tNameplate)
			
			tNameplate.wnd.container:ArrangeChildrenVert(2)
			tNameplate.wnd.nameContainer:ArrangeChildrenHorz(1)
			
			if self.bReposition then
				if tNameplate.bPosCheck then
					self:MoveForFatties(tNameplate)
				end
			end
			
			if self.bPerspectivePlates then
				Apollo.GetAddon("PerspectivePlates"):OnRequestedResize(tNameplate)
			end
		end
	end
end

function MinimalistPlates:DrawName(tNameplate)
	local wndNameplate = tNameplate.wndNameplate
	local unitOwner = tNameplate.unitOwner
	
	local wndName = tNameplate.wnd.name
	
	local bShow = false
	if self:HelperVerifyVisibilityOptions(tNameplate, "Names") then
		bShow = true
	end
	
	if wndName:IsShown() ~= bShow then
		wndName:Show(bShow)
	end
	
	if bShow then
		local strNewName
		if self.bShowTitles then
			strNewName = unitOwner:GetTitleOrName()
		else
			strNewName = unitOwner:GetName()
		end
		
		if wndName:GetText() ~= strNewName then
			wndName:SetText(strNewName)
		end
		
		-- Resize
		local nNameWidth = Apollo.GetTextWidth(self.kcrNameFont, strNewName .. " ")
		wndName:SetAnchorOffsets(0, 0, nNameWidth, karFonts[self.kcrNameFont])
		tNameplate.wnd.nameContainer:SetAnchorOffsets(-200, 0, 200, karFonts[self.kcrNameFont])
	end
end

function MinimalistPlates:DrawGuild(tNameplate)
	local wndNameplate = tNameplate.wndNameplate
	local unitOwner = tNameplate.unitOwner
	local strNewGuild = unitOwner:GetAffiliationName()
	
	local wndGuild = tNameplate.wnd.guild
	
	local bShow = false
	if strNewGuild ~= nil and strNewGuild ~= "" then
		if self:HelperVerifyVisibilityOptions(tNameplate, "Guilds") then
			bShow = true
		end
	end
	
	if wndGuild:IsShown() ~= bShow then
		wndGuild:Show(bShow)
	end
	
	if bShow then
		if unitOwner:GetType() == "Player" then
			strNewGuild = String_GetWeaselString(Apollo.GetString("Nameplates_GuildDisplay"), strNewGuild)
		end
		
		if wndGuild:GetText() ~= strNewGuild then
			wndGuild:SetTextRaw(strNewGuild)
		end
		
		-- Resize
		local nGuildWidth = Apollo.GetTextWidth(self.kcrGuildFont, strNewGuild .. " ")
		wndGuild:SetAnchorOffsets(-nGuildWidth/2, 0, nGuildWidth/2, karFonts[self.kcrGuildFont])
	end
end

function MinimalistPlates:DrawHealth(tNameplate)
	local wndNameplate = tNameplate.wndNameplate
	local unitOwner = tNameplate.unitOwner

	local wndHealth = tNameplate.wnd.health
	
	if unitOwner:GetHealth() == nil or unitOwner:GetHealth() == 0 then
		wndHealth:Show(false)
		return
	end

	local bShow = false
	if self:HelperVerifyVisibilityOptions(tNameplate, "Bars") then
		bShow = true
	end
	
	if wndHealth:IsShown() ~= bShow then
		wndHealth:Show(bShow)
	end
	
	if bShow then
		local nHealthCurr = unitOwner:GetHealth()
		local nHealthMax = unitOwner:GetMaxHealth()
		local nShieldCurr = unitOwner:GetShieldCapacity()
		local nAbsorbCurrent = unitOwner:GetAbsorptionValue()
		
		tNameplate.wnd.healthBar:SetMax(nHealthMax)
		tNameplate.wnd.healthBar:SetProgress(nHealthCurr)
		tNameplate.wnd.absorb:SetMax(nHealthMax)
		tNameplate.wnd.absorb:SetProgress(nAbsorbCurrent or 0)
	
		-- Health text
		local bShowNumber = self.bShowHealthNumbers
		if tNameplate.wnd.healthText:IsShown() ~= bShowNumber then
			tNameplate.wnd.healthText:Show(bShowNumber)
		end
		
		if bShowNumber then
			local nHealthText = self:HelperFormatBigNumber(nHealthCurr)
			
			if self.bShowHealthPercent then
				nHealthText = string.format("%.f%%", nHealthCurr / nHealthMax * 100)
			end
			
			if tNameplate.wnd.healthText:GetText() ~= nHealthText then
				tNameplate.wnd.healthText:SetText(nHealthText)
			end
		end
		
		-- Shield text
		local bShowShield = self.bShowShieldNumbers and nShieldCurr > 0
		if tNameplate.wnd.shieldText:IsShown() ~= bShowShield then
			tNameplate.wnd.shieldText:Show(bShowShield)
		end
		
		if bShowShield then
			local nShieldText = self:HelperFormatBigNumber(nShieldCurr)
			
			if self.bShowShieldPercent then
				nShieldText = string.format("%.f%%", nShieldCurr / unitOwner:GetShieldCapacityMax() * 100)
			end
			
			if tNameplate.wnd.shieldText:GetText() ~= nShieldText then
				tNameplate.wnd.shieldText:SetText(nShieldText)
			end
		end
	end
end

function MinimalistPlates:DrawShield(tNameplate)
	local wndNameplate = tNameplate.wndNameplate
	local unitOwner = tNameplate.unitOwner
	
	local wndShield = tNameplate.wnd.shield
	
	if unitOwner:GetShieldCapacity() == nil or unitOwner:GetShieldCapacity() == 0 then
		wndShield:Show(false)
		return
	end
	
	local bShow = false
	if self:HelperVerifyVisibilityOptions(tNameplate, "Bars") then
		bShow = true
	end
	
	if wndShield:IsShown() ~= bShow then
		wndShield:Show(bShow)
	end
	
	if bShow then
		local nShieldCurr = unitOwner:GetShieldCapacity()
		local nShieldMax = unitOwner:GetShieldCapacityMax()
		
		tNameplate.wnd.shieldBar:SetMax(nShieldMax)
		tNameplate.wnd.shieldBar:SetProgress(nShieldCurr)
	end
end

function MinimalistPlates:DrawCast(tNameplate)
	local wndNameplate = tNameplate.wndNameplate
	local unitOwner = tNameplate.unitOwner
	
	local wndCast = tNameplate.wnd.cast
	
	local bShow = false
	if unitOwner:ShouldShowCastBar() then
		if self:HelperVerifyVisibilityOptions(tNameplate, "CastBars") then
			bShow = true
		end
	end
	
	if wndCast:IsShown() ~= bShow then
		wndCast:Show(bShow)
	end
	
	if bShow then
		tNameplate.wnd.castBar:SetMax(unitOwner:GetCastDuration())
		tNameplate.wnd.castBar:SetProgress(unitOwner:GetCastElapsed())
		tNameplate.wnd.castName:SetText(unitOwner:GetCastName())
	end
end

function MinimalistPlates:DrawVulnerable(tNameplate)
	local wndNameplate = tNameplate.wndNameplate
	local unitOwner = tNameplate.unitOwner
	
	local wndVulnerable = tNameplate.wnd.vulnerable
	
	if tNameplate.wnd.health:IsShown() then
		local nVulnerable = unitOwner:GetCCStateTimeRemaining(Unit.CodeEnumCCState.Vulnerability)
		
		if nVulnerable ~= nil and nVulnerable > 0 then
			if nVulnerable ~= 0 and nVulnerable > tNameplate.nVulnerableTime then
				tNameplate.nVulnerableTime = nVulnerable
				wndVulnerable:Show(true)
			elseif nVulnerable ~= 0 and nVulnerable < tNameplate.nVulnerableTime then
				wndVulnerable:SetMax(tNameplate.nVulnerableTime)
				wndVulnerable:SetProgress(nVulnerable)
			end
		else
			tNameplate.nVulnerableTime = 0
			wndVulnerable:Show(false)
		end
	end
end

function MinimalistPlates:DrawRewards(tNameplate)

	-- Check individual icons
	local bShowQuest = tNameplate.tRewards.bShowQuest and self.bShowQuestIcons
	local bShowChallenge = tNameplate.tRewards.bShowChallenge and self.bShowChallengeIcons
	local bShowPath = tNameplate.tRewards.bShowPath and self.bShowPathIcons
	
	if tNameplate.wnd.rewardQuest:IsShown() ~= bShowQuest then
		tNameplate.wnd.rewardQuest:Show(bShowQuest)
	end
	if tNameplate.wnd.rewardChallenge:IsShown() ~= bShowChallenge then
		tNameplate.wnd.rewardChallenge:Show(bShowChallenge)
	end
	if tNameplate.wnd.rewardPath:IsShown() ~= bShowPath then
		tNameplate.wnd.rewardPath:Show(bShowPath)
	end
	
	-- Check reward container
	local bShowIcons = bShowQuest or bShowChallenge or bShowPath
	if tNameplate.wnd.rewards:IsShown() ~= bShowIcons then
		tNameplate.wnd.rewards:Show(bShowIcons)
	end
	
	if bShowIcons then
		tNameplate.wnd.rewards:SetAnchorOffsets(0, 0, tNameplate.wnd.rewards:ArrangeChildrenHorz(0), 15)
	end
end

function MinimalistPlates:DrawTargeting(tNameplate)
	local wndNameplate = tNameplate.wndNameplate
	local unitOwner = tNameplate.unitOwner
	
	local wndTarget = tNameplate.wnd.targetMarker
	
	local bShow = false
	if self.bShowTargetMarker then
		if tNameplate.bIsTarget then
			bShow = true
		end
	end
	
	if wndTarget:IsShown() ~= bShow then
		wndTarget:Show(bShow)
	end
	
	if bShow then
		if tNameplate.wnd.shield:IsShown() then
			wndTarget:SetAnchorOffsets(-2, -(2 + self.kcrShieldHeight), 2, 2)
		else
			wndTarget:SetAnchorOffsets(-2, -2, 2, 2)
		end
	end
end

function MinimalistPlates:DrawArmor(tNameplate)
	local wndNameplate = tNameplate.wndNameplate
	local unitOwner = tNameplate.unitOwner
	
	local wndArmor = tNameplate.wnd.armor
	local wndLevel = tNameplate.wnd.level
	
	local bShow = false
	if self.bShowArmor then
		if unitOwner:GetInterruptArmorValue() > 0 then
			bShow = true
		end
	end
	
	local bShowLevel = false
	if self.bShowLevels then
		if unitOwner:GetLevel() ~= nil then
			bShowLevel = true
		end
	end
	
	if wndArmor:IsShown() ~= bShow or wndLevel:IsShown() ~= bShowLevel then
		wndArmor:Show(bShow)
		wndLevel:Show(bShowLevel)
		tNameplate.wnd.info:ArrangeChildrenHorz(0)
	end
	
	if bShow then
		tNameplate.wnd.armorNumber:SetText(unitOwner:GetInterruptArmorValue())
	end
	
	if bShowLevel then
		tNameplate.wnd.level:SetText(unitOwner:GetLevel())
	end
end

function MinimalistPlates:ColorNameplate(tNameplate)
	local unitOwner = tNameplate.unitOwner
	local eDisposition = tNameplate.eDisposition 
	
	local tempNameColor = "ffffff"
	local tempHealthColor = "15b01a"
	
	local bShowClassColor = false
	local bShowHostileCutoff = false
	local bShowFriendlyCutoff = false
	
	if eDisposition == Unit.CodeEnumDisposition.Hostile then
		local nHealthCurr = unitOwner:GetHealth() or 0
		local nHealthMax = unitOwner:GetMaxHealth() or 0
		local nHealthPct = nHealthCurr / nHealthMax * 100
		
		if nHealthPct <= self.nHostileCutoff then
			bShowHostileCutoff = true
		end
		
		if unitOwner:GetType() == "Player" then
			if unitOwner:IsPvpFlagged() then
				tempNameColor = self.kcrHostilePlayerName
				tempHealthColor = self.kcrHostilePlayerHealthBar
			else
				tempNameColor = self.kcrHostileUnflaggedPlayerName
				tempHealthColor = self.kcrHostileUnflaggedPlayerHealthBar
			end
			
			if self.bShowHostileClassColors then
				bShowClassColor = true
			end
		else
			tempNameColor = self.kcrHostileName
			tempHealthColor = self.kcrHostileHealthBar
			
			if self.bShowThreat then
				local unitTarget = unitOwner:GetTarget()
				
				if unitTarget ~= nil then
					if not unitTarget:IsThePlayer() then
						tempHealthColor = self.kcrHostileLowThreat
					end
				end
			end
		end
	elseif eDisposition == Unit.CodeEnumDisposition.Neutral then
		tempNameColor = self.kcrNeutralName
		tempHealthColor = self.kcrNeutralHealthBar
	elseif eDisposition == Unit.CodeEnumDisposition.Friendly then
		if unitOwner:GetType() == "Player" then
			local nHealthCurr = unitOwner:GetHealth() or 0
			local nHealthMax = unitOwner:GetMaxHealth() or 0
			local nHealthPct = nHealthCurr / nHealthMax * 100
			
			if unitOwner:IsInYourGroup() then
				tempNameColor = self.kcrPartyPlayerName
				tempHealthColor = self.kcrPartyPlayerHealthBar
			else
				tempNameColor = self.kcrFriendlyPlayerName
				tempHealthColor = self.kcrFriendlyPlayerHealthBar
			end
			
			if nHealthPct <= self.nFriendlyCutoff then
				bShowFriendlyCutoff = true
			elseif self.bShowFriendlyClassColors then
				bShowClassColor = true
			end
		else
			tempNameColor = self.kcrFriendlyName
			tempHealthColor = self.kcrFriendlyHealthBar
		end
	end
	
	if tNameplate.nVulnerableTime > 0 then
		tempHealthColor = self.kcrVulnerableBar
	elseif bShowHostileCutoff then
		tempHealthColor = self.kcrHostileCutoff
	elseif bShowFriendlyCutoff then
		tempHealthColor = self.kcrFriendlyCutoff
	elseif bShowClassColor then
		local strClassColor = self["kcr"..karClassId[unitOwner:GetClassId()]]
		tempHealthColor = strClassColor
	end

	tNameplate.wnd.name:SetTextColor(self.kcrOpacity..tempNameColor)
	tNameplate.wnd.guild:SetTextColor(self.kcrOpacity..tempNameColor)
	tNameplate.wnd.healthBar:SetBarColor(self.kcrOpacity..tempHealthColor)
end

function MinimalistPlates:MoveForFatties(tNameplate)
	local unitOwner = tNameplate.unitOwner
	
	if unitOwner == nil then
		return
	end

	local posX, posY = self:GetRealPos(tNameplate.wnd.name)
	if posY < -5 and tNameplate.bIsOver then
		tNameplate.wndMeasure = Apollo.LoadForm(self.xmlDoc, "Measure", "InWorldHudStratum", self)
		tNameplate.wndMeasure:SetAnchorOffsets(0, 0, 0, 0)
		tNameplate.wndMeasure:SetUnit(unitOwner, 1)
		tNameplate.wndNameplate:SetUnit(unitOwner, 0)
		tNameplate.bIsOver = false
	elseif tNameplate.wndMeasure then
		local posX, posY = tNameplate.wndMeasure:GetPos()
		if posY - (15 + karFonts[self.kcrNameFont]) > 0 then
			tNameplate.wndMeasure:Destroy()
			tNameplate.wndMeasure = nil
			tNameplate.wndNameplate:SetUnit(unitOwner, 1)
			tNameplate.bIsOver = true
		end
	end
end

function MinimalistPlates:UpdateStyle(tNameplate)
	
	-- Temporarily hide
	tNameplate.wnd.name:Show(false)
	tNameplate.wnd.guild:Show(false)
	tNameplate.wnd.health:Show(false)
	tNameplate.wnd.shield:Show(false)
	
	-- Set style
	tNameplate.wnd.name:SetFont(self.kcrNameFont)
	tNameplate.wnd.guild:SetFont(self.kcrGuildFont)
	tNameplate.wnd.health:SetAnchorOffsets(-self.kcrWidth/2, 0, self.kcrWidth/2, self.kcrHealthHeight)
	tNameplate.wnd.shield:SetAnchorOffsets(-self.kcrWidth/2, 0, self.kcrWidth/2, self.kcrShieldHeight)
	tNameplate.wnd.healthBar:SetBGColor(self.kcrOpacity..self.kcrBackground)
	tNameplate.wnd.absorb:SetBarColor(self.kcrOpacity..self.kcrAbsorbBar)
	tNameplate.wnd.shieldBar:SetBarColor(self.kcrOpacity..self.kcrShieldBar)
	tNameplate.wnd.vulnerable:SetBarColor(self.kcrOpacity..self.kcrVulnerableTimerBar)
	tNameplate.wnd.castBar:SetBarColor(self.kcrOpacity..self.kcrCastBar)
	tNameplate.wnd.targetMarker:SetBGColor(self.kcrOpacity..self.kcrTargetIndicator)
end

function MinimalistPlates:CheckDrawDistance(tNameplate)
	local unitPlayer = self.unitPlayer
	local unitOwner = tNameplate.unitOwner

	if not unitOwner or not unitPlayer then
	    return false
	end

	local tPosTarget = unitOwner:GetPosition()
	local tPosPlayer = unitPlayer:GetPosition()

	if tPosTarget == nil or tPosPlayer == nil then
		return
	end

	local nDeltaX = tPosTarget.x - tPosPlayer.x
	local nDeltaY = tPosTarget.y - tPosPlayer.y
	local nDeltaZ = tPosTarget.z - tPosPlayer.z

	local nDistance = (nDeltaX * nDeltaX) + (nDeltaY * nDeltaY) + (nDeltaZ * nDeltaZ)

	if tNameplate.bIsTarget then
		bInRange = nDistance < knTargetRange
		return bInRange
	else
		bInRange = nDistance < (self.nMaxRange * self.nMaxRange) -- squaring for quick maths
		return bInRange
	end
end

function MinimalistPlates:HelperVerifyVisibility(tNameplate)
	local unitOwner = tNameplate.unitOwner
	local eDisposition = tNameplate.eDisposition

	local bHiddenUnit = not unitOwner:ShouldShowNamePlate() and not tNameplate.bIsImportant
	if bHiddenUnit and not tNameplate.bIsTarget then
		return false
	end
	
	if (self.bUseOcclusion and tNameplate.bOccluded) or not tNameplate.bOnScreen then
		return false
	end

	if tNameplate.bGibbed then
		return false
	end

	local bShowNameplate = true
	
	if unitOwner:IsDead() then
		bShowNameplate = false
	elseif unitOwner:GetHealth() == nil then
		bShowNameplate = false
	end

	return bShowNameplate or tNameplate.bIsTarget or tNameplate.bIsImportant
end

function MinimalistPlates:HelperVerifyVisibilityOptions(tNameplate, wnd)
	local unitOwner = tNameplate.unitOwner
	local eDisposition = tNameplate.eDisposition
	
	local unitType
	if tNameplate.bIsTarget then
		unitType = "Target"
	elseif unitOwner:IsThePlayer() then
		unitType = "Player"
	elseif unitOwner:GetType() == "Player" then
		if eDisposition == 0 then
			unitType = "HostilePlayer"
		else
			unitType = "FriendlyPlayer"
		end
	elseif unitOwner:GetType() == "Pet" then
		local petOwner = unitOwner:GetUnitOwner()
		
		if eDisposition == 0 then
			unitType = "HostilePet"
		elseif petOwner ~= nil and petOwner:IsThePlayer() then
			unitType = "PlayerPet"
		else
			unitType = "FriendlyPet"
		end
	else
		unitType = karDispositionId[eDisposition]
	end
	
	if unitType == "Hostile" then
		if not tNameplate.bPosCheck then
			tNameplate.bPosCheck = true
		end
	end
	
	-- Temporary buff code
	if unitType == "Player" or (unitType == "FriendlyPlayer" and unitOwner:IsInYourGroup()) then
		if not tNameplate.bCheckBuffs then
			tNameplate.bCheckBuffs = true
		end
	end
	
	if wnd == "Bars" and unitType == "FriendlyPlayer" then
		if self.bHideFriendlyBars then
			local nHealthCurr = unitOwner:GetHealth()
			local nHealthMax = unitOwner:GetMaxHealth()
			
			local nHealthPct
			if self.bHideFriendlyBars_Shield then
				nHealthPct = nHealthCurr / nHealthMax * 100
			else
				local nShieldCurr = unitOwner:GetShieldCapacity()
				local nShieldMax = unitOwner:GetShieldCapacityMax()
				nHealthPct = (nHealthCurr + nShieldCurr) / (nHealthMax + nShieldMax) * 100
			end
			
			if nHealthPct <= 95 then
				return true
			end
			return false
		end
	end
	
	if self["bShow"..unitType..wnd] then
		return true
	end
	
	if wnd == "Names" or wnd == "Guilds" then
		if tNameplate.bIsImportant then
			return true
		end
	end
	
	if wnd == "Bars" then
		if self["bShow"..unitType..wnd.."_Combat"] then
			if unitOwner:IsInCombat() then
				return true
			end
		end
	end
end

function MinimalistPlates:IsImportantNPC(unitOwner)
	local tActivation = unitOwner:GetActivationState()
	
	--Units without health
	if tActivation.Bank ~= nil then
		return true
	elseif tActivation.CREDDExchange then
		return true
	end

	--Flight paths
	if tActivation.FlightPathSettler ~= nil or tActivation.FlightPath ~= nil or tActivation.FlightPathNew then
		return true
	end
	
	--Quests
	if tActivation.QuestReward ~= nil then
		return true
	elseif tActivation.QuestNew ~= nil or tActivation.QuestNewMain ~= nil then
		return true
	elseif tActivation.QuestReceiving ~= nil then
		return true
	elseif tActivation.TalkTo ~= nil then
		return true
	end
	
	--Vendors
	if tActivation.CommodityMarketplace ~= nil then
		return true
	elseif tActivation.ItemAuctionhouse then
		return true
	elseif tActivation.Vendor then
		return true
	end
	
	--Trainers
	if tActivation.TradeskillTrainer then
		return true
	end
end

function MinimalistPlates:HelperFormatBigNumber(nArg)
	local strResult
	if nArg >= 1000000 then
		if math.floor(nArg%1000000/100000) == 0 then
			strResult = string.format("%sm", math.floor(nArg / 1000000))
		else
			strResult = string.format("%s.%sm", math.floor(nArg / 1000000), math.floor(nArg % 1000000 / 100000))
		end
	elseif nArg >= 1000 then
		if math.floor(nArg%1000/100) == 0 then
			strResult = string.format("%sk", math.floor(nArg / 1000))
		else
			strResult = string.format("%s.%sk", math.floor(nArg / 1000), math.floor(nArg % 1000 / 100))
		end
	else
		strResult = nArg
	end
	return strResult
end

function MinimalistPlates:GetRealPos(wnd)
	local posX, posY = wnd:GetPos()
	
	if wnd:GetParent() ~= nil then
		local currWnd = wnd:GetParent()
		repeat
			local parentX, parentY = currWnd:GetPos()
			posX = posX + parentX
			posY = posY + parentY
			currWnd = currWnd:GetParent()
		until currWnd == nil
	end
	
	return posX, posY
end

-----------------------------------------------------------------------------------------------
-- Nameplate Events
-----------------------------------------------------------------------------------------------
function MinimalistPlates:OnNameClick(wndHandler, wndControl, eMouseButton)
	local unitOwner = wndControl:GetData()

	if unitOwner ~= nil then
		if GameLib.GetTargetUnit() ~= unitOwner and eMouseButton == GameLib.CodeEnumInputMouse.Left then
			GameLib.SetTargetUnit(unitOwner)
			return true
		end
	end
end

function MinimalistPlates:OnWorldLocationOnScreen(wndHandler, wndControl, bOnScreen)
	local tNameplate = self.arWnd2Nameplate[wndHandler:GetId()]
	if tNameplate ~= nil then
		tNameplate.bOnScreen = bOnScreen
		self:UpdateNameplateVisibility(tNameplate)
	end
end

function MinimalistPlates:OnUnitOcclusionChanged(wndHandler, wndControl, bOccluded)
	local tNameplate = self.arWnd2Nameplate[wndHandler:GetId()]
	if tNameplate ~= nil then
		tNameplate.bOccluded = bOccluded
		self:UpdateNameplateVisibility(tNameplate)
	end
end

-----------------------------------------------------------------------------------------------
-- System Events
-----------------------------------------------------------------------------------------------
function MinimalistPlates:OnUnitGibbed(unitUpdated)
	local tNameplate = self.arUnit2Nameplate[unitUpdated:GetId()]
	if tNameplate ~= nil then
		tNameplate.bGibbed = true
		self:UpdateNameplateVisibility(tNameplate)
	end
end

function MinimalistPlates:OnTargetUnitChanged(unitOwner)
	for idx, tNameplateOther in pairs(self.arUnit2Nameplate) do
		tNameplateOther.bIsTarget = false
	end

	if unitOwner == nil then
		return
	end

	local tNameplate = self.arUnit2Nameplate[unitOwner:GetId()]
	if tNameplate == nil then
		return
	end

	if GameLib.GetTargetUnit() == unitOwner then
		tNameplate.bIsTarget = true
	end
end

-----------------------------------------------------------------------------------------------
-- Options
-----------------------------------------------------------------------------------------------
function MinimalistPlates:OnConfigure()
	self:OnMinimalistPlatesOn()
end

function MinimalistPlates:OnMinimalistPlatesOn()
	self.wndMain:Show(true)
	self:RefreshMinimalistPlatesConfigure()
end

function MinimalistPlates:RefreshMinimalistPlatesConfigure()
	
	-- Generic managed controls
	for property,tData in pairs(karSavedProperties) do
		if tData.nControlType == 1 and self[property] ~= nil then
			local wndControl = self.wndMain:FindChild(tData.strControlName)
			if wndControl ~= nil then
				wndControl:SetCheck(self[property])
			end
		end
		if tData.nControlType == 2 and self[property] ~= nil then
			local wndControl = self.wndOptionsColors:FindChild(tData.strControlName)
			if wndControl ~= nil then
				wndControl:SetCheck(self[property])
			end
		end
		if tData.nControlType == 3 and self[property] ~= nil then
			local wndControl = self.wndOptionsColors:FindChild(tData.strControlName)
			if wndControl ~= nil then
				wndControl:SetText(self[property])
				wndControl:SetTextColor(self.kcrOpacity..self[property])
			end
		end
	end
	
	-- Draw distance
	if self.nMaxRange ~= nil then
		self.wndOptionsMain:FindChild("Slider_DrawDistance"):SetValue(self.nMaxRange)
		self.wndOptionsMain:FindChild("Label_DrawDistance"):SetText("Draw distance ("..self.nMaxRange..")")
	end
	
	-- Health cutoff
	if self.nFriendlyCutoff ~= nil then
		self.wndOptionsMain:FindChild("Slider_FriendlyCutoff"):SetValue(self.nFriendlyCutoff)
		self.wndOptionsMain:FindChild("Label_FriendlyCutoff"):SetText("Friendly cutoff ("..self.nFriendlyCutoff.."%)")
	end
	
	if self.nHostileCutoff ~= nil then
		self.wndOptionsMain:FindChild("Slider_HostileCutoff"):SetValue(self.nHostileCutoff)
		self.wndOptionsMain:FindChild("Label_HostileCutoff"):SetText("Hostile cutoff ("..self.nHostileCutoff.."%)")
	end
	
	-- Occlusion setting
	self.wndOptionsMain:FindChild("Button_UseOcclusion"):SetCheck(Apollo.GetConsoleVariable("ui.occludeNameplatePositions"))
	
	-- Width
	if self.kcrWidth ~= nil then
		self.wndOptionsStyle:FindChild("Slider_Width"):SetValue(self.kcrWidth)
		self.wndOptionsStyle:FindChild("Label_Width"):SetText("Width ("..self.kcrWidth..")")
	end
	
	-- Shield height
	if self.kcrShieldHeight ~= nil then
		self.wndOptionsStyle:FindChild("Slider_ShieldHeight"):SetValue(self.kcrShieldHeight)
		self.wndOptionsStyle:FindChild("Label_ShieldHeight"):SetText("Shield Height ("..self.kcrShieldHeight..")")
	end
	
	-- Health height
	if self.kcrHealthHeight ~= nil then
		self.wndOptionsStyle:FindChild("Slider_HealthHeight"):SetValue(self.kcrHealthHeight)
		self.wndOptionsStyle:FindChild("Label_HealthHeight"):SetText("Health Height ("..self.kcrHealthHeight..")")
	end
	
	-- Fonts
	if self.kcrNameFont ~= nil then
		for key, value in pairs(karFonts) do
			self.wndOptionsStyle:FindChild("Button_Name_"..key):SetCheck(self.kcrNameFont == key)
			self.wndOptionsStyle:FindChild("Button_Name_"..key):SetData(key)
		end
	end
	
	if self.kcrGuildFont ~= nil then
		for key, value in pairs(karFonts) do
			self.wndOptionsStyle:FindChild("Button_Guild_"..key):SetCheck(self.kcrGuildFont == key)
			self.wndOptionsStyle:FindChild("Button_Guild_"..key):SetData(key)
		end
	end
end

function MinimalistPlates:Button_Close()
	self.wndMain:Show(false)
end

function MinimalistPlates:Button_Main()
	self.wndMain:FindChild("Content_Main"):Show(true)
	self.wndMain:FindChild("Content_Colors"):Show(false)
	self.wndMain:FindChild("Content_Style"):Show(false)
	
	self.wndMain:FindChild("Content_Main"):SetVScrollPos(0)
end

function MinimalistPlates:Button_Colors()
	self.wndMain:FindChild("Content_Main"):Show(false)
	self.wndMain:FindChild("Content_Colors"):Show(true)
	self.wndMain:FindChild("Content_Style"):Show(false)
	
	self.wndMain:FindChild("Content_Colors"):SetVScrollPos(0)
end

function MinimalistPlates:Button_Style()
	self.wndMain:FindChild("Content_Main"):Show(false)
	self.wndMain:FindChild("Content_Colors"):Show(false)
	self.wndMain:FindChild("Content_Style"):Show(true)
	
	self.wndMain:FindChild("Content_Style"):SetVScrollPos(0)
end

function MinimalistPlates:Button_Generic(wndHandler, wndControl, eMouseButton)
	local strSettingName = wndControl:GetData()
	if strSettingName ~= nil then
		self[strSettingName] = wndControl:IsChecked()
		local fnCallback = karSavedProperties[strSettingName].fnCallback
		if fnCallback ~= nil then
			self[fnCallback](self)
		end
	end
end

function MinimalistPlates:Edit_Generic(wndHandler, wndControl)
	local strSettingName = wndControl:GetData()
	if strSettingName ~= nil then
		local colorString = wndControl:GetText()
		
		if string.len(colorString) > 6 then
			wndControl:SetText(string.sub(colorString, 0, 6))
		elseif string.len(colorString) == 6 then
			wndControl:SetTextColor(self.kcrOpacity..colorString)
			self[strSettingName] = colorString
			local fnCallback = karSavedProperties[strSettingName].fnCallback
			if fnCallback ~= nil then
				self[fnCallback](self)
			end
		end
	end
end

function MinimalistPlates:Button_NameFont(wndHandler, wndControl)
	local strFont = wndControl:GetData()
	if strFont ~= nil then
		self.kcrNameFont = strFont
		
		self:UpdateAllNameplateStyles()
	end
end

function MinimalistPlates:Button_GuildFont(wndHandler, wndControl)
	local strFont = wndControl:GetData()
	if strFont ~= nil then
		self.kcrGuildFont = strFont
		
		self:UpdateAllNameplateStyles()
	end
end

function MinimalistPlates:Slider_DrawDistance(wndHandler, wndControl, nValue, nOldValue)
	if nValue ~= nOldValue then
		self.wndOptionsMain:FindChild("Label_DrawDistance"):SetText("Draw distance ("..nValue..")")
		self.nMaxRange = nValue
		
		self:UpdateAllNameplateVisibility()
	end
end

function MinimalistPlates:Slider_FriendlyCutoff(wndHandler, wndControl, nValue, nOldValue)
	if nValue ~= nOldValue then
		self.wndOptionsMain:FindChild("Label_FriendlyCutoff"):SetText("Friendly cutoff ("..nValue.."%)")
		self.nFriendlyCutoff = nValue
	end
end

function MinimalistPlates:Slider_HostileCutoff(wndHandler, wndControl, nValue, nOldValue)
	if nValue ~= nOldValue then
		self.wndOptionsMain:FindChild("Label_HostileCutoff"):SetText("Hostile cutoff ("..nValue.."%)")
		self.nHostileCutoff = nValue
	end
end

function MinimalistPlates:Button_UseOcclusion(wndHandler, wndControl, eMouseButton)
	local bUseOcclusion = wndControl:IsChecked()
	Apollo.SetConsoleVariable("ui.occludeNameplatePositions", bUseOcclusion)
	self.bUseOcclusion = bUseOcclusion
end

function MinimalistPlates:Slider_Width(wndHandler, wndControl, nValue, nOldValue)
	if nValue ~= nOldValue then
		self.wndOptionsStyle:FindChild("Label_Width"):SetText("Width ("..nValue..")")
		self.kcrWidth = nValue
		
		self:UpdateAllNameplateStyles()
	end
end

function MinimalistPlates:Slider_ShieldHeight(wndHandler, wndControl, nValue, nOldValue)
	if nValue ~= nOldValue then
		self.wndOptionsStyle:FindChild("Label_ShieldHeight"):SetText("Shield Height ("..nValue..")")
		self.kcrShieldHeight = nValue
		
		self:UpdateAllNameplateStyles()
	end
end

function MinimalistPlates:Slider_HealthHeight(wndHandler, wndControl, nValue, nOldValue)
	if nValue ~= nOldValue then
		self.wndOptionsStyle:FindChild("Label_HealthHeight"):SetText("Health Height ("..nValue..")")
		self.kcrHealthHeight = nValue
		
		self:UpdateAllNameplateStyles()
	end
end

-----------------------------------------------------------------------------------------------
-- MinimalistPlates
-----------------------------------------------------------------------------------------------
MinimalistPlates:Init()
