package com.verdantartifice.primalmagick.datagen.affinities;

import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.verdantartifice.primalmagick.common.affinities.AffinityType;
import com.verdantartifice.primalmagick.common.enchantments.EnchantmentsPM;
import com.verdantartifice.primalmagick.common.entities.EntityTypesPM;
import com.verdantartifice.primalmagick.common.items.ItemsPM;
import com.verdantartifice.primalmagick.common.sources.Source;
import com.verdantartifice.primalmagick.common.sources.SourceList;

import net.minecraft.data.DataGenerator;
import net.minecraft.data.DataProvider;
import net.minecraft.data.HashCache;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.item.Items;
import net.minecraft.world.item.alchemy.Potions;
import net.minecraft.world.item.enchantment.Enchantments;

public class AffinityProvider implements DataProvider {
    private static final Gson GSON = (new GsonBuilder()).setPrettyPrinting().create();
    private static final Logger LOGGER = LogManager.getLogger();
    protected final DataGenerator generator;
    
    public AffinityProvider(DataGenerator generator) {
        this.generator = generator;
    }

    @Override
    public void run(HashCache cache) throws IOException {
        Path path = this.generator.getOutputFolder();
        Map<AffinityType, Map<ResourceLocation, IFinishedAffinity>> map = new HashMap<>();
        this.registerAffinities((affinity) -> {
            if (map.computeIfAbsent(affinity.getType(), (type) -> { return new HashMap<>(); }).put(affinity.getId(), affinity) != null) {
                LOGGER.debug("Duplicate affinity in data generation: " + affinity.getId().toString());
            }
        });
        for (Map.Entry<AffinityType, Map<ResourceLocation, IFinishedAffinity>> entry1 : map.entrySet()) {
            for (Map.Entry<ResourceLocation, IFinishedAffinity> entry : entry1.getValue().entrySet()) {
                IFinishedAffinity affinity = entry.getValue();
                this.saveAffinity(cache, affinity.getAffinityJson(), path.resolve("data/" + entry.getKey().getNamespace() + "/affinities/" + affinity.getType().getFolder() + "/" + entry.getKey().getPath() + ".json"));
            }
        }
    }
    
    private void saveAffinity(HashCache cache, JsonObject json, Path path) {
        try {
            String jsonStr = GSON.toJson((JsonElement)json);
            String hash = SHA1.hashUnencodedChars(jsonStr).toString();
            if (!Objects.equals(cache.getHash(path), hash) || !Files.exists(path)) {
                Files.createDirectories(path.getParent());
                try (BufferedWriter writer = Files.newBufferedWriter(path)) {
                    writer.write(jsonStr);
                }
            }
            cache.putNew(path, hash);
        } catch (IOException e) {
            LOGGER.error("Couldn't save affinity {}", path, e);
        }
    }
    
    protected void registerAffinities(Consumer<IFinishedAffinity> consumer) {
        SourceList auraUnit = new SourceList().add(Source.EARTH, 1).add(Source.SEA, 1).add(Source.SKY, 1).add(Source.SUN, 1).add(Source.MOON, 1);
        
        // Define vanilla item affinities
        ItemAffinityBuilder.itemAffinity(Items.STONE).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRANITE).base(Items.STONE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POLISHED_GRANITE).base(Items.GRANITE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DIORITE).base(Items.STONE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POLISHED_DIORITE).base(Items.DIORITE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ANDESITE).base(Items.STONE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POLISHED_ANDESITE).base(Items.ANDESITE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEEPSLATE).base(Items.STONE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COBBLED_DEEPSLATE).base(Items.DEEPSLATE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POLISHED_DEEPSLATE).base(Items.DEEPSLATE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CALCITE).base(Items.STONE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TUFF).base(Items.OBSIDIAN).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DRIPSTONE_BLOCK).base(Items.STONE).add(Source.SEA, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRASS_BLOCK).base(Items.DIRT).add(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DIRT).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COARSE_DIRT).base(Items.DIRT).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PODZOL).base(Items.DIRT).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ROOTED_DIRT).base(Items.DIRT).add(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CRIMSON_NYLIUM).base(Items.NETHERRACK).add(Source.MOON, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WARPED_NYLIUM).base(Items.NETHERRACK).add(Source.MOON, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COBBLESTONE).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.OAK_PLANKS).set(Source.EARTH, 2).set(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPRUCE_PLANKS).base(Items.OAK_PLANKS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BIRCH_PLANKS).base(Items.OAK_PLANKS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.JUNGLE_PLANKS).base(Items.OAK_PLANKS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ACACIA_PLANKS).base(Items.OAK_PLANKS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DARK_OAK_PLANKS).base(Items.OAK_PLANKS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CRIMSON_PLANKS).base(Items.OAK_PLANKS).add(Source.MOON, 2).add(Source.INFERNAL, 2).remove(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WARPED_PLANKS).base(Items.CRIMSON_PLANKS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.OAK_SAPLING).set(Source.EARTH, 10).set(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPRUCE_SAPLING).base(Items.OAK_SAPLING).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BIRCH_SAPLING).base(Items.OAK_SAPLING).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.JUNGLE_SAPLING).base(Items.OAK_SAPLING).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ACACIA_SAPLING).base(Items.OAK_SAPLING).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DARK_OAK_SAPLING).base(Items.OAK_SAPLING).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BEDROCK).set(Source.EARTH, 20).set(Source.VOID, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SAND).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_SAND).base(Items.SAND).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRAVEL).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GOLD_ORE).base(Items.STONE).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEEPSLATE_GOLD_ORE).base(Items.DEEPSLATE).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COPPER_ORE).base(Items.STONE).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEEPSLATE_COPPER_ORE).base(Items.DEEPSLATE).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.IRON_ORE).base(Items.STONE).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEEPSLATE_IRON_ORE).base(Items.DEEPSLATE).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COAL_ORE).base(Items.STONE).add(Source.EARTH, 5).add(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEEPSLATE_COAL_ORE).base(Items.DEEPSLATE).add(Source.EARTH, 5).add(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.REDSTONE_ORE).base(Items.STONE).add(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEEPSLATE_REDSTONE_ORE).base(Items.DEEPSLATE).add(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.EMERALD_ORE).base(Items.STONE).add(Source.EARTH, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEEPSLATE_EMERALD_ORE).base(Items.DEEPSLATE).add(Source.EARTH, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LAPIS_ORE).base(Items.STONE).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEEPSLATE_LAPIS_ORE).base(Items.DEEPSLATE).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DIAMOND_ORE).base(Items.STONE).add(Source.EARTH, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEEPSLATE_DIAMOND_ORE).base(Items.DEEPSLATE).add(Source.EARTH, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHER_GOLD_ORE).base(Items.NETHERRACK).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHER_QUARTZ_ORE).base(Items.NETHERRACK).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ANCIENT_DEBRIS).base(Items.NETHERRACK).add(Source.EARTH, 5).add(Source.INFERNAL, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BUDDING_AMETHYST).base(Items.AMETHYST_BLOCK).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.EXPOSED_COPPER).base(Items.COPPER_BLOCK).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WEATHERED_COPPER).base(Items.COPPER_BLOCK).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.OXIDIZED_COPPER).base(Items.COPPER_BLOCK).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.OAK_LOG).set(Source.EARTH, 10).set(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPRUCE_LOG).base(Items.OAK_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BIRCH_LOG).base(Items.OAK_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.JUNGLE_LOG).base(Items.OAK_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ACACIA_LOG).base(Items.OAK_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DARK_OAK_LOG).base(Items.OAK_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CRIMSON_STEM).base(Items.OAK_LOG).add(Source.MOON, 10).add(Source.INFERNAL, 10).remove(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WARPED_STEM).base(Items.CRIMSON_STEM).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_OAK_LOG).base(Items.OAK_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_SPRUCE_LOG).base(Items.SPRUCE_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_BIRCH_LOG).base(Items.BIRCH_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_JUNGLE_LOG).base(Items.JUNGLE_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_ACACIA_LOG).base(Items.ACACIA_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_DARK_OAK_LOG).base(Items.DARK_OAK_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_CRIMSON_STEM).base(Items.CRIMSON_STEM).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_WARPED_STEM).base(Items.WARPED_STEM).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_OAK_WOOD).base(Items.OAK_WOOD).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_SPRUCE_WOOD).base(Items.SPRUCE_WOOD).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_BIRCH_WOOD).base(Items.BIRCH_WOOD).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_JUNGLE_WOOD).base(Items.JUNGLE_WOOD).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_ACACIA_WOOD).base(Items.ACACIA_WOOD).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_DARK_OAK_WOOD).base(Items.DARK_OAK_WOOD).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_CRIMSON_HYPHAE).base(Items.CRIMSON_HYPHAE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRIPPED_WARPED_HYPHAE).base(Items.WARPED_HYPHAE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.OAK_WOOD).base(Items.OAK_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPRUCE_WOOD).base(Items.SPRUCE_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BIRCH_WOOD).base(Items.BIRCH_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.JUNGLE_WOOD).base(Items.JUNGLE_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ACACIA_WOOD).base(Items.ACACIA_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DARK_OAK_WOOD).base(Items.DARK_OAK_LOG).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CRIMSON_HYPHAE).base(Items.CRIMSON_STEM).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WARPED_HYPHAE).base(Items.WARPED_STEM).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.OAK_LEAVES).set(Source.EARTH, 5).set(Source.SKY, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPRUCE_LEAVES).base(Items.OAK_LEAVES).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BIRCH_LEAVES).base(Items.OAK_LEAVES).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.JUNGLE_LEAVES).base(Items.OAK_LEAVES).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ACACIA_LEAVES).base(Items.OAK_LEAVES).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DARK_OAK_LEAVES).base(Items.OAK_LEAVES).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.AZALEA_LEAVES).base(Items.OAK_LEAVES).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FLOWERING_AZALEA_LEAVES).base(Items.AZALEA_LEAVES).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPONGE).set(Source.SEA, 10).set(Source.VOID, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WET_SPONGE).base(Items.SPONGE).add(Source.SEA, 10).remove(Source.VOID, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SANDSTONE).base(Items.STONE).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COBWEB).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRASS).set(Source.EARTH, 2).set(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FERN).base(Items.GRASS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.AZALEA).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FLOWERING_AZALEA).base(Items.AZALEA).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_BUSH).base(Items.GRASS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SEAGRASS).base(Items.GRASS).add(Source.SEA, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SEA_PICKLE).set(Source.EARTH, 5).set(Source.SEA, 5).set(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WHITE_WOOL).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ORANGE_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MAGENTA_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_BLUE_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.YELLOW_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIME_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PINK_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRAY_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_GRAY_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CYAN_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PURPLE_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLUE_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BROWN_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GREEN_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLACK_WOOL).base(Items.WHITE_WOOL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DANDELION).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POPPY).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLUE_ORCHID).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ALLIUM).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.AZURE_BLUET).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_TULIP).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ORANGE_TULIP).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WHITE_TULIP).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PINK_TULIP).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.OXEYE_DAISY).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CORNFLOWER).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LILY_OF_THE_VALLEY).base(Items.DANDELION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WITHER_ROSE).base(Items.DANDELION).add(Source.INFERNAL, 5).remove(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPORE_BLOSSOM).set(Source.EARTH, 5).set(Source.MOON, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BROWN_MUSHROOM).set(Source.EARTH, 5).set(Source.MOON, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_MUSHROOM).base(Items.BROWN_MUSHROOM).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CRIMSON_FUNGUS).base(Items.BROWN_MUSHROOM).add(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WARPED_FUNGUS).base(Items.CRIMSON_FUNGUS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CRIMSON_ROOTS).set(Source.EARTH, 2).set(Source.MOON, 2).set(Source.INFERNAL, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WARPED_ROOTS).base(Items.CRIMSON_ROOTS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHER_SPROUTS).base(Items.CRIMSON_ROOTS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WEEPING_VINES).base(Items.VINE).add(Source.MOON, 5).add(Source.INFERNAL, 5).remove(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TWISTING_VINES).base(Items.WEEPING_VINES).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SUGAR_CANE).set(Source.EARTH, 5).set(Source.SEA, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.KELP).set(Source.EARTH, 5).set(Source.SEA, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MOSS_BLOCK).set(Source.EARTH, 5).set(Source.MOON, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.HANGING_ROOTS).set(Source.EARTH, 2).set(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BIG_DRIPLEAF).set(Source.EARTH, 5).set(Source.MOON, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SMALL_DRIPLEAF).set(Source.EARTH, 2).set(Source.MOON, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BAMBOO).set(Source.EARTH, 5).set(Source.SKY, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.OAK_SLAB).set(Source.EARTH, 1).set(Source.SUN, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPRUCE_SLAB).base(Items.OAK_SLAB).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BIRCH_SLAB).base(Items.OAK_SLAB).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.JUNGLE_SLAB).base(Items.OAK_SLAB).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ACACIA_SLAB).base(Items.OAK_SLAB).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DARK_OAK_SLAB).base(Items.OAK_SLAB).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CRIMSON_SLAB).base(Items.OAK_SLAB).add(Source.MOON, 1).add(Source.INFERNAL, 1).remove(Source.SUN, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WARPED_SLAB).base(Items.CRIMSON_SLAB).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PETRIFIED_OAK_SLAB).base(Items.OAK_SLAB).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MOSSY_COBBLESTONE).base(Items.COBBLESTONE).add(Source.MOON, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.OBSIDIAN).set(Source.EARTH, 5).set(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TORCH).set(Source.EARTH, 2).set(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHORUS_PLANT).set(Source.EARTH, 5).set(Source.MOON, 5).set(Source.VOID, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHORUS_FLOWER).base(Items.CHORUS_PLANT).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PURPUR_BLOCK).set(Source.EARTH, 5).set(Source.VOID, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPAWNER).set(Source.BLOOD, 20).set(Source.INFERNAL, 10).set(Source.VOID, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FARMLAND).base(Items.DIRT).add(Source.SEA, 2).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FURNACE).set(Source.EARTH, 30).set(Source.INFERNAL, 5).build(consumer);  // Add burning affinity
        ItemAffinityBuilder.itemAffinity(Items.SNOW).set(Source.SEA, 2).set(Source.SKY, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ICE).set(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SNOW_BLOCK).set(Source.SEA, 5).set(Source.SKY, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CACTUS).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PUMPKIN).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CARVED_PUMPKIN).base(Items.PUMPKIN).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHERRACK).set(Source.EARTH, 5).set(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SOUL_SAND).base(Items.NETHERRACK).add(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SOUL_SOIL).base(Items.SOUL_SAND).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BASALT).base(Items.STONE).add(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SOUL_TORCH).base(Items.TORCH).add(Source.INFERNAL, 5).remove(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.INFESTED_STONE).base(Items.STONE).add(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.INFESTED_COBBLESTONE).base(Items.COBBLESTONE).add(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.INFESTED_STONE_BRICKS).base(Items.STONE_BRICKS).add(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.INFESTED_MOSSY_STONE_BRICKS).base(Items.MOSSY_STONE_BRICKS).add(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.INFESTED_CRACKED_STONE_BRICKS).base(Items.CRACKED_STONE_BRICKS).add(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.INFESTED_CHISELED_STONE_BRICKS).base(Items.CHISELED_STONE_BRICKS).add(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.INFESTED_DEEPSLATE).base(Items.DEEPSLATE).add(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STONE_BRICKS).set(Source.EARTH, 3).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MOSSY_STONE_BRICKS).base(Items.STONE_BRICKS).add(Source.MOON, 3).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CRACKED_STONE_BRICKS).base(Items.STONE_BRICKS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHISELED_STONE_BRICKS).base(Items.STONE_BRICKS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BROWN_MUSHROOM_BLOCK).base(Items.MUSHROOM_STEM).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_MUSHROOM_BLOCK).base(Items.MUSHROOM_STEM).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSHROOM_STEM).set(Source.EARTH, 5).set(Source.MOON, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MELON).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.VINE).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GLOW_LICHEN).set(Source.EARTH, 2).set(Source.SUN, 2).set(Source.MOON, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MYCELIUM).set(Source.EARTH, 5).set(Source.MOON, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LILY_PAD).set(Source.EARTH, 2).set(Source.SEA, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.END_PORTAL_FRAME).set(Source.EARTH, 5).set(Source.VOID, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.END_STONE).set(Source.EARTH, 5).set(Source.VOID, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DRAGON_EGG).set(Source.BLOOD, 20).set(Source.VOID, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHIPPED_ANVIL).base(Items.ANVIL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DAMAGED_ANVIL).base(Items.ANVIL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DIRT_PATH).base(Items.GRASS_BLOCK).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SUNFLOWER).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LILAC).base(Items.SUNFLOWER).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ROSE_BUSH).base(Items.SUNFLOWER).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PEONY).base(Items.SUNFLOWER).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TALL_GRASS).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LARGE_FERN).base(Items.TALL_GRASS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SEA_LANTERN).set(Source.EARTH, 24).set(Source.SEA, 52).set(Source.SUN, 15).build(consumer);  // Add lighting affinity
        ItemAffinityBuilder.itemAffinity(Items.RED_SANDSTONE).base(Items.SANDSTONE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MAGMA_BLOCK).set(Source.INFERNAL, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BONE_BLOCK).set(Source.MOON, 10).set(Source.BLOOD, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WARPED_WART_BLOCK).base(Items.NETHER_WART_BLOCK).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WHITE_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ORANGE_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MAGENTA_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_BLUE_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.YELLOW_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIME_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PINK_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRAY_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_GRAY_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CYAN_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PURPLE_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLUE_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BROWN_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GREEN_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLACK_SHULKER_BOX).base(Items.SHULKER_BOX).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WHITE_CONCRETE).base(Items.WHITE_CONCRETE_POWDER).add(Source.SEA, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ORANGE_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MAGENTA_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_BLUE_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.YELLOW_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIME_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PINK_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRAY_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_GRAY_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CYAN_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PURPLE_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLUE_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BROWN_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GREEN_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLACK_CONCRETE).base(Items.WHITE_CONCRETE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TURTLE_EGG).set(Source.SEA, 5).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_TUBE_CORAL_BLOCK).base(Items.DEAD_TUBE_CORAL).add(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_BRAIN_CORAL_BLOCK).base(Items.DEAD_BRAIN_CORAL).add(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_BUBBLE_CORAL_BLOCK).base(Items.DEAD_BUBBLE_CORAL).add(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_FIRE_CORAL_BLOCK).base(Items.DEAD_FIRE_CORAL).add(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_HORN_CORAL_BLOCK).base(Items.DEAD_HORN_CORAL).add(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TUBE_CORAL_BLOCK).base(Items.DEAD_TUBE_CORAL_BLOCK).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BRAIN_CORAL_BLOCK).base(Items.DEAD_BRAIN_CORAL_BLOCK).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BUBBLE_CORAL_BLOCK).base(Items.DEAD_BUBBLE_CORAL_BLOCK).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FIRE_CORAL_BLOCK).base(Items.DEAD_FIRE_CORAL_BLOCK).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.HORN_CORAL_BLOCK).base(Items.DEAD_HORN_CORAL_BLOCK).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TUBE_CORAL).base(Items.DEAD_TUBE_CORAL).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BRAIN_CORAL).base(Items.DEAD_BRAIN_CORAL).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BUBBLE_CORAL).base(Items.DEAD_BUBBLE_CORAL).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FIRE_CORAL).base(Items.DEAD_FIRE_CORAL).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.HORN_CORAL).base(Items.DEAD_HORN_CORAL).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_TUBE_CORAL).set(Source.EARTH, 5).set(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_BRAIN_CORAL).base(Items.DEAD_TUBE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_BUBBLE_CORAL).base(Items.DEAD_TUBE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_FIRE_CORAL).base(Items.DEAD_TUBE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_HORN_CORAL).base(Items.DEAD_TUBE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TUBE_CORAL_FAN).base(Items.TUBE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BRAIN_CORAL_FAN).base(Items.BRAIN_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BUBBLE_CORAL_FAN).base(Items.BUBBLE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FIRE_CORAL_FAN).base(Items.FIRE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.HORN_CORAL_FAN).base(Items.HORN_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_TUBE_CORAL_FAN).base(Items.DEAD_TUBE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_BRAIN_CORAL_FAN).base(Items.DEAD_BRAIN_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_BUBBLE_CORAL_FAN).base(Items.DEAD_BUBBLE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_FIRE_CORAL_FAN).base(Items.DEAD_FIRE_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DEAD_HORN_CORAL_FAN).base(Items.DEAD_HORN_CORAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CONDUIT).set(Source.SEA, 78).set(Source.SUN, 15).set(Source.BLOOD, 12).build(consumer);  // Add lighting affinity
        ItemAffinityBuilder.itemAffinity(Items.SCULK_SENSOR).set(Source.EARTH, 5).set(Source.VOID, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ELYTRA).set(Source.SKY, 25).set(Source.VOID, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SCUTE).set(Source.SEA, 2).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.APPLE).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COAL).set(Source.EARTH, 5).set(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHARCOAL).base(Items.COAL).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DIAMOND).set(Source.EARTH, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.EMERALD).set(Source.EARTH, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LAPIS_LAZULI).set(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.QUARTZ).set(Source.EARTH, 10).set(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.AMETHYST_SHARD).set(Source.EARTH, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RAW_IRON).set(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.IRON_INGOT).set(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RAW_COPPER).set(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COPPER_INGOT).set(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RAW_GOLD).set(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GOLD_INGOT).set(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_SCRAP).set(Source.EARTH, 5).set(Source.INFERNAL, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_SWORD).set(Source.EARTH, 56).set(Source.INFERNAL, 22).build(consumer);     // Smithing recipes can't be auto-scanned
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_SHOVEL).set(Source.EARTH, 45).set(Source.INFERNAL, 22).build(consumer);    // Smithing recipes can't be auto-scanned
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_PICKAXE).set(Source.EARTH, 68).set(Source.INFERNAL, 22).build(consumer);   // Smithing recipes can't be auto-scanned
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_AXE).set(Source.EARTH, 68).set(Source.INFERNAL, 22).build(consumer);       // Smithing recipes can't be auto-scanned
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_HOE).set(Source.EARTH, 57).set(Source.INFERNAL, 22).build(consumer);       // Smithing recipes can't be auto-scanned
        ItemAffinityBuilder.itemAffinity(Items.STICK).set(Source.EARTH, 1).set(Source.SUN, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.STRING).set(Source.SKY, 5).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FEATHER).set(Source.SKY, 10).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GUNPOWDER).set(Source.EARTH, 5).set(Source.INFERNAL, 15).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WHEAT_SEEDS).set(Source.EARTH, 2).set(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WHEAT).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHAINMAIL_HELMET).base(Items.IRON_HELMET).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHAINMAIL_CHESTPLATE).base(Items.IRON_CHESTPLATE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHAINMAIL_LEGGINGS).base(Items.IRON_LEGGINGS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHAINMAIL_BOOTS).base(Items.IRON_BOOTS).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_HELMET).set(Source.EARTH, 90).set(Source.INFERNAL, 22).build(consumer);        // Smithing recipes can't be auto-scanned
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_CHESTPLATE).set(Source.EARTH, 123).set(Source.INFERNAL, 22).build(consumer);   // Smithing recipes can't be auto-scanned
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_LEGGINGS).set(Source.EARTH, 112).set(Source.INFERNAL, 22).build(consumer);     // Smithing recipes can't be auto-scanned
        ItemAffinityBuilder.itemAffinity(Items.NETHERITE_BOOTS).set(Source.EARTH, 78).set(Source.INFERNAL, 22).build(consumer);         // Smithing recipes can't be auto-scanned
        ItemAffinityBuilder.itemAffinity(Items.FLINT).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PORKCHOP).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ENCHANTED_GOLDEN_APPLE).base(Items.GOLDEN_APPLE).add(Source.HALLOWED, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WATER_BUCKET).base(Items.BUCKET).add(Source.SEA, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LAVA_BUCKET).base(Items.BUCKET).add(Source.INFERNAL, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POWDER_SNOW_BUCKET).base(Items.BUCKET).add(Source.SEA, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SADDLE).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.REDSTONE).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SNOWBALL).set(Source.SEA, 2).set(Source.SKY, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LEATHER).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MILK_BUCKET).base(Items.BUCKET).add(Source.BLOOD, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PUFFERFISH_BUCKET).base(Items.BUCKET).add(Source.SEA, 5).add(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SALMON_BUCKET).base(Items.BUCKET).add(Source.SEA, 5).add(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COD_BUCKET).base(Items.BUCKET).add(Source.SEA, 5).add(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TROPICAL_FISH_BUCKET).base(Items.BUCKET).add(Source.SEA, 5).add(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.AXOLOTL_BUCKET).base(Items.BUCKET).add(Source.SEA, 5).add(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BRICK).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CLAY_BALL).set(Source.EARTH, 2).set(Source.SEA, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SLIME_BALL).set(Source.SEA, 5).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.EGG).set(Source.SKY, 2).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GLOWSTONE_DUST).set(Source.SUN, 10).set(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COD).set(Source.SEA, 5).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SALMON).set(Source.SEA, 5).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TROPICAL_FISH).set(Source.SEA, 5).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PUFFERFISH).set(Source.SEA, 5).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.INK_SAC).set(Source.SEA, 5).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GLOW_INK_SAC).base(Items.INK_SAC).add(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.COCOA_BEANS).set(Source.EARTH, 2).set(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WHITE_DYE).set(Source.SEA, 1).set(Source.SUN, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ORANGE_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MAGENTA_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_BLUE_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.YELLOW_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIME_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PINK_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRAY_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_GRAY_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CYAN_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PURPLE_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLUE_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BROWN_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GREEN_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLACK_DYE).base(Items.WHITE_DYE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BONE_MEAL).set(Source.MOON, 1).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BONE).set(Source.MOON, 5).set(Source.BLOOD, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WHITE_BED).set(Source.EARTH, 4).set(Source.SUN, 4).set(Source.BLOOD, 11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ORANGE_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MAGENTA_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_BLUE_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.YELLOW_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIME_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PINK_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRAY_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_GRAY_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CYAN_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PURPLE_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLUE_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BROWN_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GREEN_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLACK_BED).base(Items.WHITE_BED).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FILLED_MAP).base(Items.MAP).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MELON_SLICE).set(Source.EARTH, 2).set(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DRIED_KELP).base(Items.KELP).remove(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PUMPKIN_SEEDS).set(Source.EARTH, 2).set(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MELON_SEEDS).set(Source.EARTH, 2).set(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BEEF).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHICKEN).set(Source.SKY, 2).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ROTTEN_FLESH).set(Source.MOON, 5).set(Source.BLOOD, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ENDER_PEARL).set(Source.VOID, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLAZE_ROD).set(Source.INFERNAL, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GHAST_TEAR).set(Source.SKY, 5).set(Source.BLOOD, 5).set(Source.INFERNAL, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHER_WART).set(Source.EARTH, 5).set(Source.MOON, 5).set(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POTION).set(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPIDER_EYE).set(Source.MOON, 5).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.EXPERIENCE_BOTTLE).set(auraUnit.copy().multiply(5)).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WRITTEN_BOOK).base(Items.WRITABLE_BOOK).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FLOWER_POT).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CARROT).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POTATO).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POISONOUS_POTATO).base(Items.POTATO).add(Source.MOON, 5).add(Source.VOID, 2).remove(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MAP).set(Source.EARTH, 42).set(Source.SKY, 18).set(Source.SUN, 18).build(consumer);  // Recipe recurses
        ItemAffinityBuilder.itemAffinity(Items.SKELETON_SKULL).set(Source.MOON, 5).set(Source.BLOOD, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WITHER_SKELETON_SKULL).base(Items.SKELETON_SKULL).add(Source.INFERNAL, 10).remove(Source.MOON, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PLAYER_HEAD).set(Source.BLOOD, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ZOMBIE_HEAD).set(Source.MOON, 5).set(Source.BLOOD, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CREEPER_HEAD).set(Source.BLOOD, 20).set(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DRAGON_HEAD).set(Source.BLOOD, 20).set(Source.VOID, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHER_STAR).set(Source.EARTH, 25).set(Source.SEA, 25).set(Source.SKY, 25).set(Source.SUN, 25).set(Source.MOON, 25).set(Source.BLOOD, 25).set(Source.INFERNAL, 25).set(Source.VOID, 25).set(Source.HALLOWED, 25).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FIREWORK_ROCKET).set(Source.EARTH, 5).set(Source.SKY, 5).set(Source.SUN, 15).set(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.FIREWORK_STAR).base(Items.GUNPOWDER).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ENCHANTED_BOOK).set(auraUnit.copy().multiply(5)).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NETHER_BRICK).base(Items.BRICK).add(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PRISMARINE_SHARD).set(Source.EARTH, 2).set(Source.SEA, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PRISMARINE_CRYSTALS).set(Source.EARTH, 5).set(Source.SEA, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RABBIT).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RABBIT_FOOT).set(Source.MOON, 20).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RABBIT_HIDE).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.IRON_HORSE_ARMOR).set(Source.EARTH, 52).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GOLDEN_HORSE_ARMOR).set(Source.EARTH, 52).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DIAMOND_HORSE_ARMOR).set(Source.EARTH, 105).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NAME_TAG).set(Source.MOON, 10).set(Source.BLOOD, 10).set(Source.INFERNAL, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUTTON).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CHORUS_FRUIT).set(Source.EARTH, 5).set(Source.MOON, 5).set(Source.VOID, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BEETROOT).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BEETROOT_SEEDS).set(Source.EARTH, 2).set(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.DRAGON_BREATH).set(Source.SKY, 10).set(Source.VOID, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SPLASH_POTION).base(Items.POTION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TIPPED_ARROW).base(Items.ARROW).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LINGERING_POTION).base(Items.POTION).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TOTEM_OF_UNDYING).set(Source.HALLOWED, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SHULKER_SHELL).set(Source.BLOOD, 5).set(Source.VOID, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_11).set(Source.SKY, 20).set(Source.MOON, 20).set(Source.HALLOWED, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_13).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_BLOCKS).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_CAT).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_CHIRP).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_FAR).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_MALL).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_MELLOHI).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_OTHERSIDE).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_PIGSTEP).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_STAL).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_STRAD).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_WAIT).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MUSIC_DISC_WARD).base(Items.MUSIC_DISC_11).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.TRIDENT).set(Source.EARTH, 15).set(Source.SEA, 15).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PHANTOM_MEMBRANE).set(Source.SKY, 10).set(Source.BLOOD, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.NAUTILUS_SHELL).set(Source.SEA, 10).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.HEART_OF_THE_SEA).set(Source.SEA, 25).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SUSPICIOUS_STEW).set(Source.EARTH, 12).set(Source.SUN, 4).set(Source.MOON, 7).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GLOBE_BANNER_PATTERN).base(Items.PAPER).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PIGLIN_BANNER_PATTERN).base(Items.PAPER).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BELL).set(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LANTERN).set(Source.EARTH, 7).set(Source.SUN, 22).build(consumer);   // Add lighting affinity
        ItemAffinityBuilder.itemAffinity(Items.SOUL_LANTERN).set(Source.EARTH, 7).set(Source.SUN, 18).set(Source.INFERNAL, 3).build(consumer);  // Add lighting affinity
        ItemAffinityBuilder.itemAffinity(Items.SWEET_BERRIES).set(Source.EARTH, 5).set(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GLOW_BERRIES).set(Source.EARTH, 5).set(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SHROOMLIGHT).set(Source.EARTH, 5).set(Source.SUN, 15).set(Source.MOON, 10).set(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.HONEYCOMB).set(Source.SEA, 5).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BEE_NEST).base(Items.BEEHIVE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.HONEY_BOTTLE).base(Items.GLASS_BOTTLE).add(Source.SEA, 5).add(Source.BLOOD, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CRYING_OBSIDIAN).base(Items.OBSIDIAN).add(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLACKSTONE).base(Items.COBBLESTONE).add(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GILDED_BLACKSTONE).base(Items.BLACKSTONE).add(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.WHITE_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.ORANGE_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MAGENTA_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_BLUE_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.YELLOW_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIME_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PINK_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GRAY_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LIGHT_GRAY_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.CYAN_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.PURPLE_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLUE_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BROWN_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.GREEN_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.RED_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.BLACK_CANDLE).base(Items.CANDLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.SMALL_AMETHYST_BUD).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.MEDIUM_AMETHYST_BUD).set(Source.EARTH, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.LARGE_AMETHYST_BUD).set(Source.EARTH, 15).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.AMETHYST_CLUSTER).set(Source.EARTH, 20).build(consumer);
        ItemAffinityBuilder.itemAffinity(Items.POINTED_DRIPSTONE).set(Source.EARTH, 2).set(Source.SEA, 2).build(consumer);
        
        // Define mod affinities
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_RAW.get()).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED.get()).base(ItemsPM.MARBLE_RAW.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_SLAB.get()).base(ItemsPM.MARBLE_SLAB.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_STAIRS.get()).base(ItemsPM.MARBLE_STAIRS.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_WALL.get()).base(ItemsPM.MARBLE_WALL.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_BRICKS.get()).base(ItemsPM.MARBLE_BRICKS.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_BRICK_SLAB.get()).base(ItemsPM.MARBLE_BRICK_SLAB.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_BRICK_STAIRS.get()).base(ItemsPM.MARBLE_BRICK_STAIRS.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_BRICK_WALL.get()).base(ItemsPM.MARBLE_BRICK_WALL.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_PILLAR.get()).base(ItemsPM.MARBLE_PILLAR.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_CHISELED.get()).base(ItemsPM.MARBLE_CHISELED.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_ENCHANTED_RUNED.get()).base(ItemsPM.MARBLE_RUNED.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_SMOKED.get()).base(ItemsPM.MARBLE_RAW.get()).add(Source.MOON, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MARBLE_HALLOWED.get()).base(ItemsPM.MARBLE_RAW.get()).add(Source.HALLOWED, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MOONWOOD_LOG.get()).base(Items.OAK_LOG).add(Source.MOON, 20).remove(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STRIPPED_MOONWOOD_LOG.get()).base(ItemsPM.MOONWOOD_LOG.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MOONWOOD_WOOD.get()).base(ItemsPM.MOONWOOD_LOG.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STRIPPED_MOONWOOD_WOOD.get()).base(ItemsPM.MOONWOOD_LOG.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MOONWOOD_LEAVES.get()).base(Items.OAK_LEAVES).add(Source.MOON, 10).remove(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MOONWOOD_SAPLING.get()).base(Items.OAK_SAPLING).add(Source.MOON, 20).remove(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MOONWOOD_PLANKS.get()).base(Items.OAK_PLANKS).add(Source.MOON, 4).remove(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MOONWOOD_SLAB.get()).base(Items.OAK_SLAB).add(Source.MOON, 2).remove(Source.SUN, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.SUNWOOD_LOG.get()).base(Items.OAK_LOG).add(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STRIPPED_SUNWOOD_LOG.get()).base(ItemsPM.SUNWOOD_LOG.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.SUNWOOD_WOOD.get()).base(ItemsPM.SUNWOOD_LOG.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STRIPPED_SUNWOOD_WOOD.get()).base(ItemsPM.SUNWOOD_LOG.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.SUNWOOD_LEAVES.get()).base(Items.OAK_LEAVES).add(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.SUNWOOD_SAPLING.get()).base(Items.OAK_SAPLING).add(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.SUNWOOD_PLANKS.get()).base(Items.OAK_PLANKS).add(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.SUNWOOD_SLAB.get()).base(Items.OAK_SLAB).add(Source.SUN, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HALLOWOOD_LOG.get()).base(Items.OAK_LOG).add(Source.HALLOWED, 10).remove(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STRIPPED_HALLOWOOD_LOG.get()).base(ItemsPM.HALLOWOOD_LOG.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HALLOWOOD_WOOD.get()).base(ItemsPM.HALLOWOOD_LOG.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STRIPPED_HALLOWOOD_WOOD.get()).base(ItemsPM.HALLOWOOD_LOG.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HALLOWOOD_LEAVES.get()).base(Items.OAK_LEAVES).add(Source.HALLOWED, 5).remove(Source.SUN, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HALLOWOOD_SAPLING.get()).base(Items.OAK_SAPLING).add(Source.HALLOWED, 10).remove(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HALLOWOOD_PLANKS.get()).base(Items.OAK_PLANKS).add(Source.HALLOWED, 2).remove(Source.SUN, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HALLOWOOD_SLAB.get()).base(Items.OAK_SLAB).add(Source.HALLOWED, 1).remove(Source.SUN, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.INFUSED_STONE_EARTH.get()).base(Items.STONE).add(Source.EARTH, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.INFUSED_STONE_SEA.get()).base(Items.STONE).add(Source.SEA, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.INFUSED_STONE_SKY.get()).base(Items.STONE).add(Source.SKY, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.INFUSED_STONE_SUN.get()).base(Items.STONE).add(Source.SUN, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.INFUSED_STONE_MOON.get()).base(Items.STONE).add(Source.MOON, 1).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_BLACK.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_BLUE.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_BROWN.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_CYAN.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_GRAY.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_GREEN.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_LIGHT_BLUE.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_LIGHT_GRAY.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_LIME.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_MAGENTA.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_ORANGE.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PINK.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PURPLE.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_RED.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_WHITE.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_YELLOW.get()).base(ItemsPM.SKYGLASS.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.SKYGLASS_PANE.get()).base(Items.GLASS_PANE).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_BLACK.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_BLUE.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_BROWN.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_CYAN.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_GRAY.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_GREEN.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_LIGHT_BLUE.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_LIGHT_GRAY.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_LIME.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_MAGENTA.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_ORANGE.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_PINK.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_PURPLE.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_RED.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_WHITE.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.STAINED_SKYGLASS_PANE_YELLOW.get()).base(ItemsPM.SKYGLASS_PANE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ANCIENT_FONT_EARTH.get()).set(Source.EARTH, 25).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ANCIENT_FONT_SEA.get()).set(Source.SEA, 25).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ANCIENT_FONT_SKY.get()).set(Source.SKY, 25).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ANCIENT_FONT_SUN.get()).set(Source.SUN, 25).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ANCIENT_FONT_MOON.get()).set(Source.MOON, 25).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ROCK_SALT_ORE.get()).base(Items.STONE).add(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.QUARTZ_ORE.get()).base(Items.STONE).add(Source.EARTH, 10).add(Source.INFERNAL, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ARCANE_WORKBENCH.get()).base(Items.CRAFTING_TABLE).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ESSENCE_FURNACE.get()).base(Items.FURNACE).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.CALCINATOR_BASIC.get()).set(auraUnit.copy().multiply(5)).set(Source.EARTH, 41).set(Source.INFERNAL, 3).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.CALCINATOR_ENCHANTED.get()).set(auraUnit.copy().multiply(26)).set(Source.EARTH, 51).set(Source.INFERNAL, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_WHITE.get()).set(Source.SEA, 3).set(Source.SKY, 3).set(Source.SUN, 1).set(Source.BLOOD, 4).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_ORANGE.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_MAGENTA.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_LIGHT_BLUE.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_YELLOW.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_LIME.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_PINK.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_GRAY.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_LIGHT_GRAY.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_CYAN.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_PURPLE.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_BLUE.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_BROWN.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_GREEN.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_RED.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.RITUAL_CANDLE_BLACK.get()).base(ItemsPM.RITUAL_CANDLE_WHITE.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.GRIMOIRE.get()).base(Items.ENCHANTED_BOOK).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MUNDANE_WAND.get()).set(new SourceList()).build(consumer); // Wands have no affinities
        ItemAffinityBuilder.itemAffinity(ItemsPM.BLOODY_FLESH.get()).set(Source.BLOOD, 25).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HALLOWED_ORB.get()).set(Source.HALLOWED, 25).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ENCHANTED_INK.get()).set(Source.SEA, 3).set(Source.MOON, 3).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ROCK_SALT.get()).set(Source.EARTH, 5).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.REFINED_SALT.get()).base(ItemsPM.ROCK_SALT.get()).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MANA_PRISM.get()).set(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.MANA_SALTS.get()).set(auraUnit.copy().multiply(5)).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.BEESWAX.get()).set(Source.SEA, 2).set(Source.BLOOD, 2).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.PRIMALITE_INGOT.get()).set(auraUnit.copy().multiply(3)).set(Source.EARTH, 8).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HEXIUM_INGOT.get()).set(auraUnit.copy().multiply(2)).set(Source.EARTH, 6).set(Source.BLOOD, 15).set(Source.INFERNAL, 15).set(Source.VOID, 15).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HALLOWSTEEL_INGOT.get()).set(auraUnit).set(Source.EARTH, 5).set(Source.BLOOD, 11).set(Source.INFERNAL, 11).set(Source.VOID, 11).set(Source.HALLOWED, 37).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.SPELLCLOTH.get()).set(Source.EARTH, 4).set(Source.SEA, 4).set(Source.SKY, 19).set(Source.SUN, 4).set(Source.MOON, 4).set(Source.BLOOD, 9).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HEXWEAVE.get()).set(Source.EARTH, 3).set(Source.SEA, 3).set(Source.SKY, 29).set(Source.SUN, 3).set(Source.MOON, 3).set(Source.BLOOD, 30).set(Source.INFERNAL, 27).set(Source.VOID, 27).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.HEARTWOOD.get()).set(auraUnit.copy().multiply(5)).set(Source.EARTH, 10).set(Source.SUN, 10).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.INCENSE_STICK.get()).set(Source.EARTH, 4).set(Source.SKY, 5).set(Source.SUN, 4).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.DOWSING_ROD.get()).set(Source.EARTH, 3).set(Source.SUN, 3).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.APPRENTICE_WAND_GEM_ITEM.get()).base(Items.DIAMOND).add(auraUnit.copy().multiply(2)).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ADEPT_WAND_GEM_ITEM.get()).base(ItemsPM.APPRENTICE_WAND_GEM_ITEM.get()).add(auraUnit).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.WIZARD_WAND_GEM_ITEM.get()).base(ItemsPM.ADEPT_WAND_GEM_ITEM.get()).add(auraUnit.copy().multiply(2)).build(consumer);
        ItemAffinityBuilder.itemAffinity(ItemsPM.ARCHMAGE_WAND_GEM_ITEM.get()).base(ItemsPM.WIZARD_WAND_GEM_ITEM.get()).add(auraUnit.copy().multiply(2)).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_DUST_EARTH.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_DUST_SEA.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_DUST_SKY.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_DUST_SUN.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_DUST_MOON.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_DUST_BLOOD.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_DUST_INFERNAL.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_DUST_VOID.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_DUST_HALLOWED.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_SHARD_EARTH.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_SHARD_SEA.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_SHARD_SKY.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_SHARD_SUN.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_SHARD_MOON.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_SHARD_BLOOD.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_SHARD_INFERNAL.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_SHARD_VOID.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_SHARD_HALLOWED.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CRYSTAL_EARTH.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CRYSTAL_SEA.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CRYSTAL_SKY.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CRYSTAL_SUN.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CRYSTAL_MOON.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CRYSTAL_BLOOD.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CRYSTAL_INFERNAL.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CRYSTAL_VOID.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CRYSTAL_HALLOWED.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CLUSTER_EARTH.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CLUSTER_SEA.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CLUSTER_SKY.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CLUSTER_SUN.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CLUSTER_MOON.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CLUSTER_BLOOD.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CLUSTER_INFERNAL.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CLUSTER_VOID.get()).build(consumer);
        ItemAffinityBuilder.essenceAffinity(ItemsPM.ESSENCE_CLUSTER_HALLOWED.get()).build(consumer);
        
        // Define vanilla entity type affinities
        EntityTypeAffinityBuilder.entityAffinity(EntityType.AXOLOTL).value(Source.SEA, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.BAT).value(Source.SKY, 10).value(Source.MOON, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.BEE).value(Source.SKY, 10).value(Source.SUN, 5).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.BLAZE).value(Source.SKY, 10).value(Source.INFERNAL, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.CAT).value(Source.MOON, 5).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.CAVE_SPIDER).value(Source.EARTH, 10).value(Source.MOON, 5).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.CHICKEN).value(Source.SKY, 5).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.COD).value(Source.SEA, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.COW).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.CREEPER).value(Source.EARTH, 5).value(Source.SUN, 5).value(Source.INFERNAL, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.DOLPHIN).value(Source.SEA, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.DONKEY).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.DROWNED).value(Source.SEA, 5).value(Source.MOON, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ELDER_GUARDIAN).value(Source.SEA, 20).value(Source.BLOOD, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ENDER_DRAGON).value(Source.SKY, 20).value(Source.BLOOD, 50).value(Source.VOID, 50).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ENDERMAN).value(Source.BLOOD, 10).value(Source.VOID, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ENDERMITE).value(Source.BLOOD, 5).value(Source.VOID, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.EVOKER).value(Source.BLOOD, 10).value(Source.INFERNAL, 5).value(Source.VOID, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.FOX).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.GHAST).value(Source.SKY, 10).value(Source.BLOOD, 10).value(Source.INFERNAL, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.GLOW_SQUID).value(Source.SEA, 10).value(Source.SUN, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.GOAT).value(Source.SKY, 5).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.GUARDIAN).value(Source.SEA, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.HOGLIN).value(Source.BLOOD, 10).value(Source.INFERNAL, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.HORSE).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.HUSK).value(Source.SUN, 5).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ILLUSIONER).value(Source.BLOOD, 10).value(Source.INFERNAL, 5).value(Source.VOID, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.IRON_GOLEM).value(Source.EARTH, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.LLAMA).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.MAGMA_CUBE).value(Source.INFERNAL, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.MULE).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.MOOSHROOM).value(Source.MOON, 5).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.OCELOT).value(Source.MOON, 5).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.PANDA).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.PARROT).value(Source.SKY, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.PHANTOM).value(Source.SKY, 10).value(Source.MOON, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.PIG).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.PIGLIN).value(Source.BLOOD, 10).value(Source.INFERNAL, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.PIGLIN_BRUTE).value(Source.BLOOD, 10).value(Source.INFERNAL, 5).build(consumer); // Piglin brute
        EntityTypeAffinityBuilder.entityAffinity(EntityType.PILLAGER).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.POLAR_BEAR).value(Source.SEA, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.PUFFERFISH).value(Source.SEA, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.RABBIT).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.RAVAGER).value(Source.BLOOD, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SALMON).value(Source.SEA, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SHEEP).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SHULKER).value(Source.BLOOD, 10).value(Source.VOID, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SILVERFISH).value(Source.EARTH, 5).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SKELETON).value(Source.MOON, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SKELETON_HORSE).value(Source.MOON, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SLIME).value(Source.SEA, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SNOW_GOLEM).value(Source.SEA, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SPIDER).value(Source.MOON, 5).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.SQUID).value(Source.SEA, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.STRAY).value(Source.SEA, 5).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.STRIDER).value(Source.BLOOD, 10).value(Source.INFERNAL, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.TRADER_LLAMA).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.TROPICAL_FISH).value(Source.SEA, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.TURTLE).value(Source.SEA, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.VEX).value(Source.SKY, 10).value(Source.VOID, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.VILLAGER).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.VINDICATOR).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.WANDERING_TRADER).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.WITCH).value(Source.BLOOD, 10).value(Source.INFERNAL, 5).value(Source.VOID, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.WITHER).value(Source.SKY, 20).value(Source.INFERNAL, 50).value(Source.VOID, 50).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.WITHER_SKELETON).value(Source.BLOOD, 10).value(Source.INFERNAL, 10).value(Source.VOID, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.WOLF).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ZOGLIN).value(Source.MOON, 10).value(Source.BLOOD, 10).value(Source.INFERNAL, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ZOMBIE).value(Source.MOON, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ZOMBIE_HORSE).value(Source.MOON, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ZOMBIE_VILLAGER).value(Source.MOON, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityType.ZOMBIFIED_PIGLIN).value(Source.MOON, 10).value(Source.BLOOD, 10).value(Source.INFERNAL, 5).build(consumer);
        
        // Define mod entity type affinities
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.TREEFOLK.get()).value(Source.EARTH, 10).value(Source.SUN, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.INNER_DEMON.get()).value(Source.BLOOD, 50).value(Source.INFERNAL, 50).value(Source.VOID, 50).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.PRIMALITE_GOLEM.get()).value(Source.EARTH, 30).value(Source.SEA, 10).value(Source.SKY, 10).value(Source.SUN, 10).value(Source.MOON, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.HEXIUM_GOLEM.get()).value(Source.EARTH, 20).value(Source.BLOOD, 10).value(Source.INFERNAL, 10).value(Source.VOID, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.HALLOWSTEEL_GOLEM.get()).value(Source.EARTH, 20).value(Source.HALLOWED, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.BASIC_EARTH_PIXIE.get()).value(Source.EARTH, 5).value(Source.SKY, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.GRAND_EARTH_PIXIE.get()).value(Source.EARTH, 10).value(Source.SKY, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.MAJESTIC_EARTH_PIXIE.get()).value(Source.EARTH, 20).value(Source.SKY, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.BASIC_SEA_PIXIE.get()).value(Source.SEA, 5).value(Source.SKY, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.GRAND_SEA_PIXIE.get()).value(Source.SEA, 10).value(Source.SKY, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.MAJESTIC_SEA_PIXIE.get()).value(Source.SEA, 20).value(Source.SKY, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.BASIC_SKY_PIXIE.get()).value(Source.SKY, 15).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.GRAND_SKY_PIXIE.get()).value(Source.SKY, 20).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.MAJESTIC_SKY_PIXIE.get()).value(Source.SKY, 30).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.BASIC_SUN_PIXIE.get()).value(Source.SKY, 10).value(Source.SUN, 5).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.GRAND_SUN_PIXIE.get()).value(Source.SKY, 10).value(Source.SUN, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.MAJESTIC_SUN_PIXIE.get()).value(Source.SKY, 10).value(Source.SUN, 20).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.BASIC_MOON_PIXIE.get()).value(Source.SKY, 10).value(Source.MOON, 5).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.GRAND_MOON_PIXIE.get()).value(Source.SKY, 10).value(Source.MOON, 10).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.MAJESTIC_MOON_PIXIE.get()).value(Source.SKY, 10).value(Source.MOON, 20).value(Source.BLOOD, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.BASIC_BLOOD_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.GRAND_BLOOD_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 15).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.MAJESTIC_BLOOD_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 25).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.BASIC_INFERNAL_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 5).value(Source.INFERNAL, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.GRAND_INFERNAL_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 5).value(Source.INFERNAL, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.MAJESTIC_INFERNAL_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 5).value(Source.INFERNAL, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.BASIC_VOID_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 5).value(Source.VOID, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.GRAND_VOID_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 5).value(Source.VOID, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.MAJESTIC_VOID_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 5).value(Source.VOID, 20).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.BASIC_HALLOWED_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 5).value(Source.HALLOWED, 5).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.GRAND_HALLOWED_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 5).value(Source.HALLOWED, 10).build(consumer);
        EntityTypeAffinityBuilder.entityAffinity(EntityTypesPM.MAJESTIC_HALLOWED_PIXIE.get()).value(Source.SKY, 10).value(Source.BLOOD, 5).value(Source.HALLOWED, 20).build(consumer);
        
        // Define potion bonuses
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.NIGHT_VISION).bonus(Source.MOON, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_NIGHT_VISION).bonus(Source.MOON, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.INVISIBILITY).bonus(Source.MOON, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_INVISIBILITY).bonus(Source.MOON, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LEAPING).bonus(Source.SKY, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_LEAPING).bonus(Source.SKY, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRONG_LEAPING).bonus(Source.SKY, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.FIRE_RESISTANCE).bonus(Source.INFERNAL, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_FIRE_RESISTANCE).bonus(Source.INFERNAL, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.SWIFTNESS).bonus(Source.SKY, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_SWIFTNESS).bonus(Source.SKY, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRONG_SWIFTNESS).bonus(Source.SKY, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.SLOWNESS).bonus(Source.SKY, 1).bonus(Source.VOID, 1).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_SLOWNESS).bonus(Source.SKY, 2).bonus(Source.VOID, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRONG_SLOWNESS).bonus(Source.SKY, 2).bonus(Source.VOID, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.TURTLE_MASTER).bonus(Source.SEA, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_TURTLE_MASTER).bonus(Source.SEA, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRONG_TURTLE_MASTER).bonus(Source.SEA, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.WATER_BREATHING).bonus(Source.SEA, 1).bonus(Source.SKY, 1).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_WATER_BREATHING).bonus(Source.SEA, 2).bonus(Source.SKY, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.HEALING).bonus(Source.SUN, 1).bonus(Source.BLOOD, 1).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRONG_HEALING).bonus(Source.SUN, 2).bonus(Source.BLOOD, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.HARMING).bonus(Source.BLOOD, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRONG_HARMING).bonus(Source.BLOOD, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.POISON).bonus(Source.EARTH, 1).bonus(Source.BLOOD, 1).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_POISON).bonus(Source.EARTH, 2).bonus(Source.BLOOD, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRONG_POISON).bonus(Source.EARTH, 2).bonus(Source.BLOOD, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.REGENERATION).bonus(Source.SUN, 1).bonus(Source.BLOOD, 1).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_REGENERATION).bonus(Source.SUN, 2).bonus(Source.BLOOD, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRONG_REGENERATION).bonus(Source.SUN, 2).bonus(Source.BLOOD, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRENGTH).bonus(Source.EARTH, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_STRENGTH).bonus(Source.EARTH, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.STRONG_STRENGTH).bonus(Source.EARTH, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.WEAKNESS).bonus(Source.EARTH, 1).bonus(Source.VOID, 1).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_WEAKNESS).bonus(Source.EARTH, 2).bonus(Source.VOID, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LUCK).bonus(Source.MOON, 5).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.SLOW_FALLING).bonus(Source.SKY, 2).build(consumer);
        PotionBonusAffinityBuilder.potionBonusAffinity(Potions.LONG_SLOW_FALLING).bonus(Source.SKY, 5).build(consumer);
        
        // Define enchantment bonuses
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.ALL_DAMAGE_PROTECTION).multiplier(Source.EARTH).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.FIRE_PROTECTION).multiplier(Source.INFERNAL).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.FALL_PROTECTION).multiplier(Source.SKY).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.BLAST_PROTECTION).multiplier(Source.MOON).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.PROJECTILE_PROTECTION).multiplier(Source.SKY).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.RESPIRATION).multiplier(Source.SEA).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.AQUA_AFFINITY).multiplier(Source.SEA).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.THORNS).multiplier(Source.BLOOD).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.DEPTH_STRIDER).multiplier(Source.SEA).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.FROST_WALKER).multiplier(Source.SEA).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.BINDING_CURSE).multiplier(Source.VOID).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.SHARPNESS).multiplier(Source.BLOOD).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.SMITE).multiplier(Source.HALLOWED).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.BANE_OF_ARTHROPODS).multiplier(Source.BLOOD).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.KNOCKBACK).multiplier(Source.EARTH).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.FIRE_ASPECT).multiplier(Source.INFERNAL).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.MOB_LOOTING).multiplier(Source.MOON).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.SWEEPING_EDGE).multiplier(Source.SKY).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.BLOCK_EFFICIENCY).multiplier(Source.EARTH).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.SILK_TOUCH).multiplier(Source.MOON).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.UNBREAKING).multiplier(Source.EARTH).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.BLOCK_FORTUNE).multiplier(Source.MOON).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.POWER_ARROWS).multiplier(Source.BLOOD).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.PUNCH_ARROWS).multiplier(Source.BLOOD).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.FLAMING_ARROWS).multiplier(Source.INFERNAL).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.INFINITY_ARROWS).multiplier(Source.VOID).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.FISHING_LUCK).multiplier(Source.MOON).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.FISHING_SPEED).multiplier(Source.SEA).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.LOYALTY).multiplier(Source.SKY).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.IMPALING).multiplier(Source.BLOOD).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.RIPTIDE).multiplier(Source.SEA).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.CHANNELING).multiplier(Source.SKY).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.MULTISHOT).multiplier(Source.SKY).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.QUICK_CHARGE).multiplier(Source.SKY).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.PIERCING).multiplier(Source.BLOOD).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.MENDING).multiplier(Source.SUN).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(Enchantments.VANISHING_CURSE).multiplier(Source.VOID).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.LIFESTEAL.get()).multiplier(Source.BLOOD).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.ENDERLOCK.get()).multiplier(Source.VOID).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.JUDGMENT.get()).multiplier(Source.HALLOWED).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.ENDERPORT.get()).multiplier(Source.VOID).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.REGROWTH.get()).multiplier(Source.HALLOWED).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.AEGIS.get()).multiplier(Source.HALLOWED).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.MANA_EFFICIENCY.get()).multiplier(Source.VOID).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.SPELL_POWER.get()).multiplier(Source.INFERNAL).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.TREASURE.get()).multiplier(Source.MOON).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.BLUDGEONING.get()).multiplier(Source.EARTH).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.REVERBERATION.get()).multiplier(Source.EARTH).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.BOUNTY.get()).multiplier(Source.SEA).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.DISINTEGRATION.get()).multiplier(Source.SKY).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.VERDANT.get()).multiplier(Source.SUN).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.LUCKY_STRIKE.get()).multiplier(Source.MOON).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.RENDING.get()).multiplier(Source.BLOOD).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.SOULPIERCING.get()).multiplier(Source.INFERNAL).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.ESSENCE_THIEF.get()).multiplier(Source.VOID).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.BULWARK.get()).multiplier(Source.HALLOWED).build(consumer);
        EnchantmentBonusAffinityBuilder.enchantmentBonusAffinity(EnchantmentsPM.MAGICK_PROTECTION.get()).multiplier(Source.VOID).build(consumer);
    }

    @Override
    public String getName() {
        return "Primal Magick Affinities";
    }
}
