import AINetrunnersUtils.*
import AINetrunnersConfig.*

// possible enemy Netrunner hacks
enum AINetrunnerHack {
  Undefined = 0,
  CrippleMovement = 1,
  WeaponGlitch = 2,
  CyberwareMalfunction = 3,
  RebootOptics = 4,
  ShortCircuit = 5,
  SynapseBurnout = 6,
  Overheat = 7,
  Burning = 8,
  Contagion = 9,
  ImminentSystemFailure = 10
}

@addField(UploadFromNPCToPlayerListener)
let AI_Netrunners_lastNetrunnerHealth: Float;

@addField(AIQuickHackAction)
let AI_Netrunners_customHack: AINetrunnerHack;

// player being hacked
@addField(ScriptedPuppet)
let AI_Netrunners_hasBeenHackedBefore: Bool;

@addField(ScriptedPuppet)
let AI_Netrunners_lastHackApplied: AINetrunnerHack;

// hacking NPC
@addField(ScriptedPuppet)
let AI_Netrunners_knownHacks: array<AINetrunnerHack>;

@replaceMethod(ScriptedPuppet)
protected cb func OnHackPlayerEvent(evt: ref<HackPlayerEvent>) -> Bool {
  let action: ref<AIQuickHackAction>;
  let attackAttemptEvent: ref<AIAttackAttemptEvent>;
  let target: wref<GameObject>;
  // let netrunner: wref<ScriptedPuppet>;
  let netrunner: ref<ScriptedPuppet>;
  let isBeingHacked: Bool;
  target = (GameInstance.FindEntityByID(this.GetGame(), evt.targetID) as GameObject);
  if target == null {
    return false;
  };
  netrunner = (GameInstance.FindEntityByID(this.GetGame(), evt.netrunnerID) as ScriptedPuppet);
  if netrunner == null {
    return false;
  };

  isBeingHacked = GameInstance.GetStatusEffectSystem(this.GetGame()).HasStatusEffect(evt.targetID, t"AIQuickHackStatusEffect.BeingHacked");
  
  // remove bugged starting status effect that says player is being hacked
  if !this.AI_Netrunners_hasBeenHackedBefore {
    this.AI_Netrunners_hasBeenHackedBefore = true;
    GameInstance.GetStatusEffectSystem(this.GetGame()).RemoveStatusEffect(this.GetEntityID(), t"AIQuickHackStatusEffect.BeingHacked");
    isBeingHacked = false;
  }

  // Enemy netrunners can always hack if the player isn't currently being hacked
  // if !isBeingHacked && !EntityID.IsDefined(this.m_linkedStatusEffect.targetID) {
  if !isBeingHacked {   
    action = new AIQuickHackAction();
    action.RegisterAsRequester(evt.targetID);
    action.SetExecutor(netrunner);
    
    // Alternatively can check for eaual to <TDBID:C7854CFD:18> / AIQuickHack.HackOverheat
    if evt.objectRecord.GetID() != t"AIQuickHack.HackRevealPosition" {
      let level: Float = GameInstance.GetStatsSystem(netrunner.GetGame()).GetStatValue(Cast(evt.netrunnerID), gamedataStatType.PowerLevel);
      if ArraySize(netrunner.AI_Netrunners_knownHacks) == 0 {
        netrunner.AI_Netrunners_knownHacks = generateKnownHacks(level);
      };

      let id: TweakDBID;
      let hackToApply: AINetrunnerHack = chooseFromKnownHacks(netrunner.AI_Netrunners_knownHacks);
      action.AI_Netrunners_customHack = hackToApply;
      this.AI_Netrunners_lastHackApplied = hackToApply;
      id = getIDForCustomAINetrunnerHack(action.AI_Netrunners_customHack);
      action.SetObjectActionID(id);
    } else {
      action.SetObjectActionID(evt.objectRecord.GetID());
    };
        
    action.m_target = target;
    action.SetUp(target.GetPS());
    if !action.IsPossible(target) {
      return false;
    };
    action.ProcessRPGAction(this.GetGame());
    if evt.showDirectionalIndicator {
      attackAttemptEvent = new AIAttackAttemptEvent();
      attackAttemptEvent.instigator = netrunner;
      attackAttemptEvent.target = target;
      attackAttemptEvent.isWindUp = false;
      attackAttemptEvent.continuousMode = gameEContinuousMode.Start;
      attackAttemptEvent.minimumOpacity = 0.50;
      target.QueueEvent(attackAttemptEvent);
      this.GetOwner().QueueEvent(attackAttemptEvent);
    };
    StatusEffectHelper.ApplyStatusEffect(this, t"AIQuickHackStatusEffect.BeingHacked", evt.netrunnerID);
    this.ProcessEnemyNetrunnerTutorialFact();
  } else {
    if !isBeingHacked && EntityID.IsDefined(this.m_linkedStatusEffect.targetID) {
      // this block is now unreachable. linked status effect feature is removed
      if !ArrayContains(this.m_linkedStatusEffect.netrunnerIDs, evt.netrunnerID) {
        ArrayPush(this.m_linkedStatusEffect.netrunnerIDs, evt.netrunnerID);
        AIActionHelper.UpdateLinkedStatusEffects(this, this.m_linkedStatusEffect);
        netrunner.AddLinkedStatusEffect(evt.netrunnerID, evt.targetID);
      };
    } else {
      GameInstance.GetDelaySystem(this.GetGame()).DelayEvent(GameInstance.FindEntityByID(this.GetGame(), evt.targetID), evt, 2.00);
    };
  };
}

@replaceMethod(AIQuickHackAction)
protected func SetRegenBehavior(gameInstance: GameInstance) -> Void {
  let activationTime: Float;
  let regenMod: StatPoolModifier;
  let statValue: Float;
  let distanceToTarget: Float;
  let multiplier: Float;
  GameInstance.GetStatPoolsSystem(gameInstance).GetModifier(Cast(this.GetRequesterID()), gamedataStatPoolType.QuickHackUpload, gameStatPoolModificationTypes.Regeneration, regenMod);
  statValue = GameInstance.GetStatsSystem(this.m_target.GetGame()).GetStatValue(Cast(this.m_target.GetEntityID()), gamedataStatType.NPCUploadTime);
  distanceToTarget = Vector4.Distance(this.m_executor.GetWorldPosition(), GameInstance.FindEntityByID(gameInstance, this.m_requesterID).GetWorldPosition());
  multiplier = PowF(1.02, distanceToTarget);
  activationTime = this.GetActivationTime();

  if NotEquals(this.AI_Netrunners_customHack, AINetrunnerHack.Undefined) {
    activationTime = getBaseUploadTimeFromAINetrunnerHack(this.AI_Netrunners_customHack);
  };

  let maxSeconds: Float = 30;
  let totalTime: Float = multiplier * activationTime * statValue;
  regenMod.valuePerSec = 100.00 / MinF(totalTime, maxSeconds);
  
  regenMod.enabled = true;
  // regenMod.valuePerSec = 100.00 / (multiplier * activationTime * statValue);
  regenMod.rangeEnd = 100.00;
  GameInstance.GetStatPoolsSystem(gameInstance).RequestSettingModifier(Cast(this.GetRequesterID()), gamedataStatPoolType.QuickHackUpload, gameStatPoolModificationTypes.Regeneration, regenMod);
}

@replaceMethod(UploadFromNPCToPlayerListener)
protected func SendUploadStartedEvent(action: ref<ScriptableDeviceAction>) -> Void {
  this.m_variantHud.active = true;

  // custom caption
  let hack: ref<AIQuickHackAction> = action as AIQuickHackAction;
  if IsDefined(hack) {
    switch hack.AI_Netrunners_customHack {
      case AINetrunnerHack.Contagion:
        this.m_variantHud.header = "LocKey#51313"; // "Contagion" in English
        break;
      case AINetrunnerHack.RebootOptics:
        this.m_variantHud.header = "LocKey#350"; // "Reboot Optics" in English
        break;
      case AINetrunnerHack.SynapseBurnout:
        this.m_variantHud.header = "LocKey#54070"; // "Synapse Burnout" in English
        break;
      default:
        this.m_variantHud.header = LocKeyToString(action.GetObjectActionRecord().ObjectActionUI().Caption()); // original brhaviour
        break;
    };
  }

  // reset NPC interrupted status if they just started a new hack
  GameInstance.GetStatusEffectSystem(this.m_npcPuppet.GetGame()).RemoveStatusEffect(this.m_npcPuppet.GetEntityID(), t"AIQuickHackStatusEffect.HackingInterrupted");

  this.m_hudBlackboard = GameInstance.GetBlackboardSystem(this.m_gameInstance).Get(GetAllBlackboardDefs().UI_HUDProgressBar);
  this.m_hudBlackboard.SetBool(GetAllBlackboardDefs().UI_HUDProgressBar.Active, this.m_variantHud.active);
  this.m_hudBlackboard.SetString(GetAllBlackboardDefs().UI_HUDProgressBar.Header, this.m_variantHud.header);
}

@replaceMethod(UploadFromNPCToPlayerListener)
protected cb func OnStatPoolMaxValueReached(value: Float) -> Bool {
  let actionEffects: array<wref<ObjectActionEffect_Record>>;
  let targetTracker: wref<TargetTrackingExtension>;
  super.OnStatPoolMaxValueReached(value);
  this.RemoveLink(this.m_npcPuppet);
  ScriptedPuppet.SendActionSignal(this.m_npcPuppet, n"HackingCompleted", 1.00);
  this.m_variantHud.active = false;
  this.m_hudBlackboard.SetBool(GetAllBlackboardDefs().UI_HUDProgressBar.Active, this.m_variantHud.active);
  this.m_action.GetObjectActionRecord().CompletionEffects(actionEffects);
  
  // Removing the linked status effect feature
  // this.m_npcPuppet.AddLinkedStatusEffect(this.m_npcPuppet.GetEntityID(), this.m_playerPuppet.GetEntityID(), actionEffects);
  // this.m_playerPuppet.AddLinkedStatusEffect(this.m_npcPuppet.GetEntityID(), this.m_playerPuppet.GetEntityID(), actionEffects);
  if this.m_ssAction {
    if !TargetTrackingExtension.Get(this.m_npcPuppet, targetTracker) {
      return false;
    };
    if AIActionHelper.TryChangingAttitudeToHostile(this.m_npcPuppet, this.m_playerPuppet) {
      targetTracker.AddThreat(this.m_playerPuppet, true, this.m_playerPuppet.GetWorldPosition(), 1.00, -1.00, false);
      this.m_npcPuppet.TriggerSecuritySystemNotification(this.m_playerPuppet.GetWorldPosition(), this.m_playerPuppet, ESecurityNotificationType.COMBAT);
    };
  };
}

@replaceMethod(UploadFromNPCToPlayerListener)
public func OnStatPoolValueChanged(oldValue: Float, newValue: Float, percToPoints: Float) -> Void {
  let evt: ref<UploadProgramProgressEvent>;
  let statValue: Float;
  let immune: Bool;
  statValue = GameInstance.GetStatsSystem(this.m_playerPuppet.GetGame()).GetStatValue(Cast(this.m_playerPuppet.GetEntityID()), gamedataStatType.QuickhackShield);
  this.m_variantHud.progress = newValue / 100.00;
  this.m_hudBlackboard.SetFloat(GetAllBlackboardDefs().UI_HUDProgressBar.Progress, this.m_variantHud.progress);
  if newValue > 15.00 &&
  statValue > 0.00 && 
  !StatusEffectHelper.HasStatusEffect(this.m_playerPuppet, t"BaseStatusEffect.AntiVirusCooldown") {
    // lasts 45 seconds
    StatusEffectHelper.ApplyStatusEffect(this.m_playerPuppet, t"BaseStatusEffect.AntiVirusCooldown");
    immune = true;
  };

  // previously could never quit an in-progress reveal position hack
  // if this.m_ssAction {
  //   return ;
  // };

  // optionally stop hack if netrunner has been damaged
  let statPoolsSystem: ref<StatPoolsSystem> = GameInstance.GetStatPoolsSystem(this.m_npcPuppet.GetGame());
  let targetID: StatsObjectID = Cast(this.m_npcPuppet.GetEntityID());
  let currentHealth: Float = statPoolsSystem.GetStatPoolValue(targetID, gamedataStatPoolType.Health);
  let damaged: Bool;
  if currentHealth < this.AI_Netrunners_lastNetrunnerHealth {
    damaged = true;
  }
  this.AI_Netrunners_lastNetrunnerHealth = currentHealth;

  let shouldStopHack: Bool;
  if this.m_ssAction {
    // reveal position hack should be stopped if NPC is dead/unconscious or already in combat
      if !ScriptedPuppet.IsActive(this.m_npcPuppet) || NPCPuppet.IsInCombat(this.m_npcPuppet) {
      shouldStopHack = true;
    };
  } else {
    // conditions for stopping hacks
    if !ScriptedPuppet.IsActive(this.m_npcPuppet) ||
    !ScriptedPuppet.IsActive(this.m_playerPuppet) ||
    (StatusEffectHelper.HasStatusEffect(this.m_npcPuppet, t"AIQuickHackStatusEffect.HackingInterrupted") && allowAINetrunnersToBeInterrupted()) ||
    StatusEffectHelper.HasStatusEffect(this.m_npcPuppet, t"BaseStatusEffect.CyberwareMalfunction") ||
    StatusEffectHelper.HasStatusEffect(this.m_npcPuppet, t"BaseStatusEffect.CyberwareMalfunctionLvl2") ||
    StatusEffectHelper.HasStatusEffect(this.m_npcPuppet, t"BaseStatusEffect.CyberwareMalfunctionLvl3") ||
    immune ||
    (damaged && stopHacksWhenAINetrunnerIsDamaged())
    {
      shouldStopHack = true;
    };
  };
  
  if shouldStopHack {
    //this.RemoveLinkedStatusEffects(this.m_npcPuppet);
    ScriptedPuppet.SendActionSignal(this.m_npcPuppet, n"HackingCompleted", 1.00);
    
    this.m_variantHud.active = false;
    this.m_hudBlackboard.SetBool(GetAllBlackboardDefs().UI_HUDProgressBar.Active, this.m_variantHud.active);
    GameInstance.GetStatPoolsSystem(this.m_gameInstance).RequestRemovingStatPool(Cast(this.m_action.GetRequesterID()), gamedataStatPoolType.QuickHackUpload);
    GameInstance.GetStatPoolsSystem(this.m_gameInstance).RequestUnregisteringListener(Cast(this.m_action.GetRequesterID()), gamedataStatPoolType.QuickHackUpload, this);
    evt = new UploadProgramProgressEvent();
    evt.state = EUploadProgramState.COMPLETED;
    GameInstance.GetPersistencySystem(this.m_gameInstance).QueueEntityEvent(this.m_action.GetRequesterID(), evt);

    // Added because hack effect is never removed otherwise
    GameInstance.GetStatusEffectSystem(this.m_playerPuppet.GetGame()).RemoveStatusEffect(this.m_playerPuppet.GetEntityID(), t"AIQuickHackStatusEffect.BeingHacked");
  };
}
