/* Values in this file are the first ones to modify if you want to change how this mod works
   Some parts require general knowledge of programming.
*/
module AINetrunnersConfig

import AINetrunnersUtils.*

// change to true if you want hacks to be interrupted the they can be in the unmodded game (I find it to be too often and not for obvious reasons)
public func allowAINetrunnersToBeInterrupted() -> Bool { return false; }

// change to true to stop hacks any time the Netrunner is damaged. This can be used instead of or in addition to the interruption effect.
public func stopHacksWhenAINetrunnerIsDamaged() -> Bool { return false; }

// This is the upload time in seconds, but it gets multiplied by an exponential distance multiplier (1.02^distance)
// which is typically around 1.5-2.0 in the ranges you're usually fighting enemies in. That's not from the mod, it's vanilla behaviour.
public func getBaseUploadTimeFromAINetrunnerHack(hack: AINetrunnerHack) -> Float {
  switch hack {
    case AINetrunnerHack.RebootOptics:
      return 3;
    case AINetrunnerHack.CyberwareMalfunction:
    case AINetrunnerHack.Overheat:
      return 5;
    case AINetrunnerHack.ShortCircuit:
    case AINetrunnerHack.Contagion:
    case AINetrunnerHack.CrippleMovement:
      return 6;
    case AINetrunnerHack.WeaponGlitch:
    case AINetrunnerHack.Burning:
    case AINetrunnerHack.SynapseBurnout:
      return 7;
    case AINetrunnerHack.ImminentSystemFailure:
      return 15;
    default:
      Log("getBaseUploadTimeFromAINetrunnerHack hack not found");
      return 5;
  };
}

// these are relative weights that get translated to probabilites when a Netrunner knows a subset of the hacks
// they determine how often each hack is chosen out of a netrunner's catalogue of hacks (fixed at 3 hacks)
public func getHackWeight(hack: AINetrunnerHack) -> Float {
  switch hack {
    case AINetrunnerHack.RebootOptics:
    case AINetrunnerHack.CyberwareMalfunction:
    case AINetrunnerHack.CrippleMovement:
    case AINetrunnerHack.WeaponGlitch:
      return 2;
    case AINetrunnerHack.Burning:
    case AINetrunnerHack.SynapseBurnout:
    case AINetrunnerHack.Overheat:
    case AINetrunnerHack.ShortCircuit:
    case AINetrunnerHack.Contagion:
      return 3;
    case AINetrunnerHack.ImminentSystemFailure:
      return 1;
    default:
      Log("getHackWeight hack not found");
      return 0;
  };
}

// works similar to how the player's hack works
// does small amount of damage (in percentage of player's health) at high health
// damage increases as player health decreases.
public func getSynapseBurnoutDamagePerc(currentHealthPerc: Float) -> Float {
  let inputHealthPerc: array<Float> = [35.0, 50.0, 90.0];
  let damageHealthPerc: array<Float> = [35.0, 25.0, 20.0];
  return GetValueFromCurve(inputHealthPerc, damageHealthPerc, currentHealthPerc);
}

public func generateKnownHacks(netrunnerLevel: Float) -> array<AINetrunnerHack> {
  // Overheat/Burning are very similar except burning does twice the damage,
  // so we want to see it more at higher levels and we don't want the Netrunner to have both
  let chanceOfBurningInsteadOfOverheat: Float = GetValueFromCurve([0.0, 50.0], [0.25, 0.75], netrunnerLevel);
  // death hack more likely at higher levels since it's so powerful
  let chanceOfUltimate: Float = GetValueFromCurve([0.0, 50.0], [0.1, 0.3], netrunnerLevel);
  // increase the chance of Synapse Burnout with higher levels. motivation is that players have lots of ways to get
  // burning/poison/shock immunity as their level increases, but Synapse Burnout will always damage them
  let baseSynapseBurnoutChance: Float = GetValueFromCurve([20.0, 50.0], [0.25, 0.5], netrunnerLevel);

  let knownHacks: array<AINetrunnerHack>;
  let choice: Int32;

  let damageHacks: array<AINetrunnerHack>;
  damageHacks = [
    AINetrunnerHack.ShortCircuit,
    AINetrunnerHack.Contagion
  ];
  if RandRangeF(0.0, 1.0) < chanceOfBurningInsteadOfOverheat {
    ArrayPush(damageHacks, AINetrunnerHack.Burning);
  } else {
    ArrayPush(damageHacks, AINetrunnerHack.Overheat);
  };

  // choose one damage hack  
  if RandRangeF(0.0, 1.0) < baseSynapseBurnoutChance {
    ArrayPush(knownHacks, AINetrunnerHack.SynapseBurnout);
    //ArrayRemove(damageHacks, AINetrunnerHack.SynapseBurnout);
  } else {
    choice = RandRange(0, ArraySize(damageHacks) - 1);
    ArrayPush(knownHacks, damageHacks[choice]);
    ArrayErase(damageHacks, choice);
    // put Sunapse Burnout in the list for possible third choice
    ArrayPush(damageHacks, AINetrunnerHack.SynapseBurnout);
  };

  let controlHacks: array<AINetrunnerHack>;
  controlHacks = [
    AINetrunnerHack.RebootOptics,
    AINetrunnerHack.CyberwareMalfunction,
    AINetrunnerHack.CrippleMovement,
    AINetrunnerHack.WeaponGlitch
  ];

  // choose one control hack
  choice = RandRange(0, ArraySize(controlHacks) - 1);
  ArrayPush(knownHacks, controlHacks[choice]);
  ArrayErase(controlHacks, choice);

  // add either the death hack or one other control/damage
  if RandRangeF(0.0, 1.0) < chanceOfUltimate {
    ArrayPush(knownHacks, AINetrunnerHack.ImminentSystemFailure);
  } else {
    if RandRangeF(0.0, 1.0) < 0.5 {
      choice = RandRange(0, ArraySize(controlHacks) - 1);
      ArrayPush(knownHacks, controlHacks[choice]);
    } else {
      choice = RandRange(0, ArraySize(damageHacks) - 1);
      ArrayPush(knownHacks, damageHacks[choice]);
    };
  };

  return knownHacks;
}
