/*
 * Decompiled with CFR 0.152.
 */
package org.firebirdsql.pool;

import java.sql.SQLException;
import java.util.HashMap;
import java.util.HashSet;
import org.firebirdsql.jdbc.FBSQLException;
import org.firebirdsql.logging.Logger;
import org.firebirdsql.pool.BlockingStack;
import org.firebirdsql.pool.ConnectionPoolConfiguration;
import org.firebirdsql.pool.PoolDebugConfiguration;
import org.firebirdsql.pool.PooledConnectionManager;
import org.firebirdsql.pool.PooledObject;

class PooledConnectionQueue {
    private static final boolean LOG_DEBUG_INFO = PoolDebugConfiguration.LOG_DEBUG_INFO;
    private static final boolean SHOW_STACK_ON_BLOCK = PoolDebugConfiguration.SHOW_TRACE;
    private static final boolean SHOW_STACK_ON_ALLOCATION = PoolDebugConfiguration.SHOW_TRACE;
    private PooledConnectionManager connectionManager;
    private Logger logger;
    private ConnectionPoolConfiguration configuration;
    private Object key;
    private String queueName;
    private int blockingTimeout;
    private int size;
    private BlockingStack stack = new BlockingStack();
    private boolean blocked;
    private Object takeMutex = new Object();
    private IdleRemover idleRemover;
    private int totalConnections;
    private HashSet workingConnections = new HashSet();
    private HashSet workingConnectionsToClose = new HashSet();
    private HashMap connectionIdleTime = new HashMap();

    private PooledConnectionQueue(PooledConnectionManager connectionManager, Logger logger, ConnectionPoolConfiguration configuration, String queueName) {
        this.connectionManager = connectionManager;
        this.logger = logger;
        this.configuration = configuration;
        this.queueName = queueName;
        this.blockingTimeout = configuration.getBlockingTimeout();
    }

    public PooledConnectionQueue(PooledConnectionManager connectionManager, Logger logger, ConnectionPoolConfiguration configuration, String queueName, Object key) {
        this(connectionManager, logger, configuration, queueName);
        this.key = key;
    }

    protected Logger getLogger() {
        return this.logger;
    }

    private ConnectionPoolConfiguration getConfiguration() {
        return this.configuration;
    }

    public int size() {
        return this.size;
    }

    public int totalSize() {
        return this.totalConnections;
    }

    public int workingSize() {
        return this.workingConnections.size();
    }

    public void start() throws SQLException {
        for (int i = 0; i < this.getConfiguration().getMinPoolSize(); ++i) {
            try {
                this.addConnection(this.stack);
                continue;
            }
            catch (InterruptedException iex) {
                throw new SQLException("Could not start connection queue.");
            }
        }
        this.idleRemover = new IdleRemover();
        Thread t = new Thread((Runnable)this.idleRemover, "Pool " + this.queueName + " idleRemover");
        t.setDaemon(true);
        t.start();
    }

    public synchronized void restart() {
        this.workingConnectionsToClose.addAll(this.workingConnections);
        while (this.size() > 0) {
            try {
                PooledObject connection = this.take();
                if (!connection.isValid()) continue;
                connection.deallocate();
                this.physicalConnectionDeallocated(connection);
            }
            catch (SQLException ex) {
                if (this.getLogger() == null) continue;
                this.getLogger().warn("Could not close connection.", ex);
            }
        }
        while (this.totalSize() < this.getConfiguration().getMinPoolSize()) {
            try {
                this.addConnection(this.stack);
            }
            catch (Exception e) {
                if (this.getLogger() == null) continue;
                this.getLogger().warn("Could not add connection.", e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        try {
            for (PooledObject item : this.workingConnections) {
                if (!item.isValid()) continue;
                item.deallocate();
            }
            while (this.size() > 0) {
                try {
                    PooledObject item;
                    item = this.take();
                    if (!item.isValid()) continue;
                    item.deallocate();
                }
                catch (SQLException ex) {
                    if (this.getLogger() == null) continue;
                    this.getLogger().warn("Could not close connection.", ex);
                }
            }
            this.totalConnections = 0;
            this.workingConnections.clear();
            this.workingConnectionsToClose.clear();
        }
        finally {
            if (this.idleRemover != null) {
                this.idleRemover.stop();
            }
            this.idleRemover = null;
        }
    }

    private boolean keepBlocking(long startTime) {
        return System.currentTimeMillis() - startTime < (long)this.blockingTimeout;
    }

    public void destroyConnection(PooledObject connection) {
        connection.deallocate();
        --this.totalConnections;
    }

    public synchronized void physicalConnectionDeallocated(PooledObject connection) {
        --this.totalConnections;
        this.connectionIdleTime.remove(connection);
        this.workingConnections.remove(connection);
    }

    public void put(PooledObject connection) throws SQLException {
        try {
            if (this.blocked && this.getLogger() != null) {
                this.getLogger().warn("Pool " + this.queueName + " will be unblocked");
            }
            if (this.getConfiguration().isPooling()) {
                if (this.workingConnectionsToClose.remove(connection)) {
                    connection.deallocate();
                    this.physicalConnectionDeallocated(connection);
                    this.addConnection(this.stack);
                } else {
                    this.stack.push(connection);
                    connection.setInPool(true);
                    this.connectionIdleTime.put(connection, new Long(System.currentTimeMillis()));
                    ++this.size;
                }
            } else {
                this.connectionIdleTime.remove(connection);
            }
            this.blocked = false;
            this.workingConnections.remove(connection);
            if (!this.getConfiguration().isPooling()) {
                this.destroyConnection(connection);
            }
            if (LOG_DEBUG_INFO && this.getLogger() != null) {
                this.getLogger().debug("Thread " + Thread.currentThread().getName() + " released connection.");
            }
        }
        catch (InterruptedException iex) {
            if (this.getLogger() != null) {
                this.getLogger().warn("Thread " + Thread.currentThread().getName() + " was interrupted.", iex);
            }
            connection.deallocate();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PooledObject take() throws SQLException {
        long startTime = System.currentTimeMillis();
        if (LOG_DEBUG_INFO && this.getLogger() != null) {
            this.getLogger().debug("Thread " + Thread.currentThread().getName() + " wants to take connection.");
        }
        PooledObject result = null;
        SQLException pendingExceptions = null;
        try {
            Object object = this.takeMutex;
            synchronized (object) {
                if (this.stack.isEmpty()) {
                    while (result == null) {
                        boolean connectionAdded;
                        block23: {
                            if (!this.keepBlocking(startTime)) {
                                String message = "Could not obtain connection during blocking timeout (" + this.blockingTimeout + " ms)";
                                FBSQLException ex = pendingExceptions != null ? new FBSQLException(message, pendingExceptions) : new FBSQLException(message);
                                throw ex;
                            }
                            connectionAdded = false;
                            try {
                                connectionAdded = this.addConnection(this.stack);
                            }
                            catch (SQLException sqlex) {
                                if (this.getLogger() != null) {
                                    this.getLogger().warn("Could not create connection." + sqlex.getMessage());
                                }
                                if (pendingExceptions == null) {
                                    pendingExceptions = sqlex;
                                }
                                if (pendingExceptions.getErrorCode() == sqlex.getErrorCode()) break block23;
                                pendingExceptions.setNextException(sqlex);
                            }
                        }
                        if (!connectionAdded) {
                            String message = "Pool " + this.queueName + " is empty and will block here." + " Thread " + Thread.currentThread().getName();
                            if (SHOW_STACK_ON_BLOCK) {
                                if (this.getLogger() != null) {
                                    this.getLogger().warn(message, new Exception());
                                }
                            } else if (this.getLogger() != null) {
                                this.getLogger().warn(message);
                            }
                            this.blocked = true;
                        }
                        if ((result = (PooledObject)this.stack.pop(this.getConfiguration().getRetryInterval())) == null && this.getLogger() != null) {
                            this.getLogger().warn("No connection in pool. Thread " + Thread.currentThread().getName());
                            continue;
                        }
                        if (result == null || connectionAdded || this.getLogger() == null) continue;
                        this.getLogger().info("Obtained connection. Thread " + Thread.currentThread().getName());
                    }
                } else {
                    result = (PooledObject)this.stack.pop();
                }
            }
        }
        catch (InterruptedException iex) {
            throw new SQLException("No free connection was available and waiting thread was interrupted.");
        }
        --this.size;
        this.workingConnections.add(result);
        if (LOG_DEBUG_INFO && this.getLogger() != null) {
            if (SHOW_STACK_ON_ALLOCATION) {
                this.getLogger().debug("Thread " + Thread.currentThread().getName() + " got connection.", new Exception());
            } else {
                this.getLogger().debug("Thread " + Thread.currentThread().getName() + " got connection.");
            }
        }
        result.setInPool(false);
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean addConnection(BlockingStack queue) throws SQLException, InterruptedException {
        PooledConnectionQueue pooledConnectionQueue = this;
        synchronized (pooledConnectionQueue) {
            boolean maximumCapacityReached;
            if (LOG_DEBUG_INFO && this.getLogger() != null) {
                this.getLogger().debug("Trying to create connection, total connections " + this.totalConnections + ", max allowed " + this.getConfiguration().getMaxPoolSize());
            }
            boolean bl = maximumCapacityReached = this.getConfiguration().getMaxPoolSize() <= this.totalConnections && this.getConfiguration().getMaxPoolSize() != 0 && this.getConfiguration().isPooling();
            if (maximumCapacityReached) {
                if (LOG_DEBUG_INFO && this.getLogger() != null) {
                    this.getLogger().debug("Was not able to add more connections.");
                }
                return false;
            }
            PooledObject pooledConnection = this.connectionManager.allocateConnection(this.key);
            if (LOG_DEBUG_INFO && this.getLogger() != null) {
                this.getLogger().debug("Thread " + Thread.currentThread().getName() + " created connection.");
            }
            queue.push(pooledConnection);
            ++this.size;
            ++this.totalConnections;
            return true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean releaseNextIdleConnection() throws SQLException {
        Object object = this.takeMutex;
        synchronized (object) {
            if (this.totalSize() <= this.getConfiguration().getMinPoolSize()) {
                return false;
            }
            PooledObject candidate = (PooledObject)this.stack.peek();
            if (candidate == null) {
                return false;
            }
            Long lastUsageTime = (Long)this.connectionIdleTime.get(candidate);
            if (lastUsageTime == null) {
                return false;
            }
            long idleTime = System.currentTimeMillis() - lastUsageTime;
            if (idleTime < (long)this.getConfiguration().getMaxIdleTime()) {
                return false;
            }
            try {
                this.take().deallocate();
            }
            finally {
                this.workingConnections.remove(candidate);
                this.connectionIdleTime.remove(candidate);
                --this.totalConnections;
            }
            return true;
        }
    }

    private class IdleRemover
    implements Runnable {
        private boolean running = true;

        public void stop() {
            this.running = false;
        }

        @Override
        public void run() {
            while (this.running) {
                try {
                    while (PooledConnectionQueue.this.releaseNextIdleConnection()) {
                    }
                }
                catch (SQLException ex) {
                    // empty catch block
                }
                try {
                    int idleTimeout = PooledConnectionQueue.this.getConfiguration().getMaxIdleTime();
                    int maxConnections = PooledConnectionQueue.this.getConfiguration().getMaxPoolSize();
                    if (maxConnections < 1) {
                        maxConnections = 1;
                    }
                    Thread.sleep(idleTimeout / maxConnections);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }
}

