package pl.mpak.sky.gui.swing.syntax.structure;

import java.util.HashMap;
import java.util.List;

import pl.mpak.sky.gui.swing.syntax.SyntaxDocument;
import pl.mpak.sky.gui.swing.syntax.SyntaxEditor.TokenRef;
import pl.mpak.sky.gui.swing.syntax.TokenCursor;

public abstract class StructureParser {

  private TokenCursor tc;
  private int[] skipBlanks;
  protected HashMap<String, Integer> keyWordList;
  private boolean ignoreCase = true;
  
  public StructureParser(int[] skipBlanks) {
    this.skipBlanks = skipBlanks;
    this.keyWordList = new HashMap<String, Integer>(); 
  }
  
  public void setTokenList(List<TokenRef> list) {
    tc = new TokenCursor(list, skipBlanks);
    tc.setIgnoreCase(ignoreCase);
  }
  
  public void setIgnoreCase(boolean ignoreCase) {
    if (tc != null) {
      tc.setIgnoreCase(ignoreCase);
    }
    else {
      this.ignoreCase = ignoreCase;
    }
  }
  
  protected TokenRef checkNull(TokenRef token) throws ParserException {
    if (token == null) {
      throw new ParserException("Nieoczekiwane zakoczenie kodu!");
    }
    return token;
  }
  
  /**
   * <p>Zwraca nastpny element lub null jeli koniec
   * @return
   * @throws ParserException 
   * @see getToken().prev
   * @see getToken().next
   * @see getToken()
   */
  protected TokenRef nextToken() throws ParserException {
    return checkNull(tc.nextToken(false));
  }
  
  /**
   * <p>Zwraca nastpny element i przy okazji pomija wszystko co zostao zdefiniowane w skipBlank()
   * @param skipBlanks
   * @return
   * @throws ParserException
   * @see skipBlank()
   */
  protected TokenRef nextToken(boolean skipBlanks) throws ParserException {
    return checkNull(tc.nextToken(skipBlanks));
  }
  
  /**
   * <p>Domylnie pomija elementy SyntaxDocument.NONE ale mona funkcj przej i pomija inne elementy
   * @return
   * @throws ParserException
   * @see nextToken(boolean skipBlanks)
   */
  protected TokenRef skipBlank() throws ParserException {
    return checkNull(tc.skipBlank());
  }
  
  /**
   * <p>Zwraca poprzednio uzyskany token.<br>
   * Nie myli z getToken().prev
   * @return
   * @throws ParserException
   */
  protected TokenRef getLastToken() throws ParserException {
    return checkNull(tc.lastToken());
  }
  
  /**
   * <p>Zwraca offset dla okrelenia pocztku bloku
   * @return
   * @throws ParserException
   */
  protected int getStartOffset() throws ParserException {
    return tc.getStartOffset();
  }
  
  /**
   * <p>Zwraca offset dla okrelenia koca bloku
   * @return
   * @throws ParserException
   */
  protected int getEndOffset() throws ParserException {
    checkNull(tc.token());
    return tc.getEndOffset();
  }
  
  /**
   * <p>Sprawdzenie czy token jest identyczny co text
   * @param text
   * @return
   * @throws ParserException
   * @see Parser(Iterator&lt;TokenRef&gt; iterator, boolean <b>ignoreCase</b>)
   */
  protected boolean isToken(String text) throws ParserException {
    checkNull(tc.token());
    return tc.isToken(text);
  }
  
  /**
   * <p>Zwraca cig znakw aktualnego tokenu
   * @return
   * @throws ParserException
   */
  protected String getTokenString() throws ParserException {
    checkNull(tc.token());
    return tc.getString();
  }
  
  /**
   * <p>Pozwla zmieni styleId elementu zalenego (w edytorze)
   * @param styleId
   * @throws ParserException
   */
  protected void setStyle(int styleId) throws ParserException {
    checkNull(tc.token());
    getToken().ref.styleId = styleId;
  }
  
  /**
   * <p>Zwraca bierzcy token
   * @return
   * @throws ParserException
   */
  protected TokenRef getToken() throws ParserException {
    return checkNull(tc.token());
  }
  
  /**
   * <p>Umieszcza sowo na specjalnej licie, ktra na kocu parsowania moe by uyta do ustawienia styli tokenw
   * @param keyWord
   * @param styleId
   * @throws ParserException
   * @see updateKeyWordStyles
   */
  protected void putKeyWord(String keyWord, int styleId) throws ParserException {
    keyWordList.put((tc.isIgnoreCase() ? keyWord.toUpperCase() : keyWord), styleId);
    if (isToken(keyWord)) {
      setStyle(styleId);
    }
  }
  
  protected boolean updateStyle(int orygStyleId) {
    return orygStyleId == SyntaxDocument.NONE;
  }
  
  /**
   * <p>Aktualizuje tokeny edytora stylami okrelonymi na licie sw kluczowych
   * @throws ParserException
   * @see putKeyWord
   */
  protected void updateKeyWordStyles() {
    tc.reset();
    try {
      while (nextToken() != null) {
        Integer styleId = keyWordList.get((tc.isIgnoreCase() ? getToken().token.toUpperCase() : getToken().token));
        if (styleId != null && updateStyle(getToken().styleId)) {
          setStyle(styleId);
        }
      }
    } catch (ParserException e) {
    }
  }

  /**
   * <p>Gwna funkcja do parsowania
   * @return
   * @throws ParserException
   */
  public abstract BlockElement parse() throws ParserException;
  
}
