#!/bin/bash
# OssImport Console for Linux

#######################################################
# Define function
#######################################################

function usage()
{
    echo "deploy [import_home]"
    echo "    Deploy import service. 'import_home' is optional, the default is the directory where 'console.sh' is located."
    echo "submit [job_cfg_file]"
    echo "    Submit job. 'job_cfg_file' is optional, the default is \${import_home}/conf/job.cfg."
    echo "start"
    echo "    Start import service."
    echo "stop"
    echo "    Stop import service."
    echo "clean job_name"
    echo "    Clean the configuration and environment of the job. The import service must be stopped before clean."
    echo "stat"
    echo "    View the status of all jobs."
    echo "retry [job_name]"
    echo "    Retry the failed tasks of the specified job or all jobs. 'job_name' is optional."
    echo "version"
    echo "    Print the version of ossimport."
}

function check_command_args() 
{
    num_of_args="$#"
        
    # check the number of args
    if [ ${num_of_args} -gt 2 -o ${num_of_args} -eq 0 ]; then
       usage
       exit 1
    fi
    
    # check command
    command=$1
    case ${command} in
    "deploy" | "Deploy" )
        ;;
    "start" | "Start")
        if [ ${num_of_args} -ne 1 ]; then
            usage
            exit 1
        fi
        ;;
    "stop" | "Stop" )
        if [ ${num_of_args} -ne 1 ]; then
            usage
            exit 1
        fi
        ;;
    "submit" | "Submit" )
        ;;
    "clean" | "Clean" )
        if [ ${num_of_args} -ne 2 ]; then
            usage
            exit
        fi
        ;;
    "stat" | "Stat" )
        if [ ${num_of_args} -ne 1 ]; then
            usage
            exit 1
        fi
        ;;
    "retry" | "Retry" )
        ;;
    "version" | "Version" )
        ;;
    * ) 
        usage
        exit 1
        ;;
    esac
}

function set_import_env()
{
    # set import home, priority `read from file` > `environment variable` > `default`
    if [ -e "${HOME}/.oss_import_home" ]; then
        import_home_from_file=$(head -n 1 examples.desktop)
        if [ -z "$import_home_from_file" ]; then
            import_home=${import_home_from_file}
        fi
    fi
    
    if [ -z "$import_home" ]; then
        osname=$(uname)
        if [ "$osname" = "Linux" ]; then
            import_home=$( dirname $(readlink -f $0) )
        else
            import_home=$( cd "$( dirname "$0" )" && pwd )
        fi
    fi
    
    # set work dir
    if [ -e ${import_home}/conf/sys.properties ]; then
        work_dir_line=$(grep "workingDir=" ${import_home}/conf/sys.properties | tr -d '\r')
        if [ -z "$work_dir_line" ]; then
            echo "Error: configuration item 'workingDir' not found in ${import_home}/conf/sys.properties."
            exit 2
        fi
        
        work_dir=${work_dir_line#workingDir*=}
        if [ -z "$work_dir" ]; then
            echo "Error: ossimport system configuration:${import_home}/conf/sys.properties not found."
            exit 2
        fi
    else 
        echo "Error: ossimport system configuration:${import_home}/conf/sys.properties not found."
        exit 2
    fi
}

function deploy_import_service()
{
    if [ -n "$specified_import_home" ]; then
        import_home=${specified_import_home}
        echo "${specified_import_home}" > ${HOME}/.oss_import_home
    fi
    
    java -jar ${import_home}/bin/console.jar deploy ${import_home}
    # TODO Add check deploy, jar/conf/logs and so on 
    echo "Info: deploy import service completed."
}

function start_import_service()
{
    echo "java -jar ${import_home}/bin/console.jar start ${work_dir}"
    java -jar ${import_home}/bin/console.jar start ${work_dir}
    sleep 1s
    mkdir -p ${import_home}/job_bak
    # TODO Add check the process master/worker/tracker
    echo "Info: start import service completed."
}

function stop_import_service()
{
    java -jar ${import_home}/bin/console.jar stop ${work_dir}
    sleep 1s
    echo "Info: stop import service completed."
}

function submit_job()
{
    # use default job cfg if not specified the job cfg
    if [ -z "$job_cfg" ]; then
        job_cfg=${import_home}/conf/job.cfg
    fi
    
    if [ ! -e "$job_cfg" ]; then
        echo "Error: job cfg file '${job_cfg}' not exist."
        exit 3
    fi

    java -jar ${import_home}/bin/console.jar submit ${work_dir} ${job_cfg} > ${import_home}/logs/submit.log 2>&1
    submit_result=$(grep "succceed!" ${import_home}/logs/submit.log)
    if [ -n "$submit_result" ]; then
        # successful
        job_name_line=$(grep "jobName=" ${job_cfg} | tr -d '\r')
        job_name=${job_name_line#jobName*=}
        echo "Info: submitted job:${job_name}, cfg:${job_cfg}"
    else
        # failed
        cat ${import_home}/logs/submit.log
    fi
    
}

function clean_job()
{   
    if [ -z "$job_name" ]; then
        echo "Error: job_name:${job_name} invalid."
        exit 3
    fi
    
    # first stop service
    stop_import_service
    # clean the job files in master and workers    
    java -jar ${import_home}/bin/console.jar clean ${work_dir} ${job_name} 
    echo "Clean job:${job_name} completed."
    # last start service
    start_import_service
}

function stat_job()
{
    # process status
    echo "================ Process Status ================="
    jps | grep -v Jps
    echo "=========================================="
    
    # job status
    num_of_job=$(ls ${work_dir}/master/jobs/ | wc -l)
    outlines=$(expr ${num_of_job} \* 8 + 1)
    sed -n '/Jobs Status/,/=========/p' ${work_dir}/logs/import.log | grep INFO | tail -n ${outlines} 
    
    # failed tasks
    echo "================ Failed Tasks ================="
    for jobs_name in $(ls ${work_dir}/master/jobs/); do
        # the failed task directory has been created
        if [ -d ${work_dir}/master/jobs/${jobs_name}/failed_tasks ]; then
            # show failed tasks
            num_of_failed_tasks=$(ls ${work_dir}/master/jobs/${jobs_name}/failed_tasks | wc -l)
            if [ ${num_of_failed_tasks} -gt 0 ]; then
                echo "Failed Job: ${jobs_name}"
                for task_name in $(ls ${work_dir}/master/jobs/${jobs_name}/failed_tasks); do
                    echo "    Failed Task: ${task_name}"
            done
            fi
        fi
    done
    echo "==========================================" 
}

function retry_failed_tasks()
{
    # stop service
    stop_import_service
    
    # retry 
    if [ -z "$job_name" ]; then
        # retry all failed tasks
        echo "Retry all failed tasks."
        for job_name in $(ls ${work_dir}/master/jobs/); do
            java -jar ${import_home}/bin/console.jar retry ${work_dir} ${job_name} > ${import_home}/logs/retry.log 2>&1
            echo "    Failed Job: ${job_name}, Retry Committed"
        done
    else
        # retry the failed tasks of the specified job
        echo "Retry job:${job_name}"
        java -jar ${import_home}/bin/console.jar retry ${work_dir} ${job_name} > ${import_home}/logs/retry.log 2>&1
        echo "    Failed Job: ${job_name}, Retry Committed"
    fi
    
    # start service
    start_import_service
}

function ver_import()
{
    java -jar ${import_home}/bin/console.jar version
}

#######################################################
# Start of main
#######################################################

import_home=${OSS_IMPORT_HOME}
work_dir=
command=
job_name=
job_cfg=
args="$*"

# check args
check_command_args ${args}

# set env
set_import_env

#echo "OSS_IMPORT_HOME:[${import_home}]."
#echo "OSS_IMPORT_WORK_DIR:[${work_dir}]."

# execute command
command=$1
case ${command} in
    "deploy" | "Deploy" )
        specified_import_home=$2
        deploy_import_service
        ;;
    "start" | "Start" )
        start_import_service
        ;;
    "stop" | "Stop" )
        stop_import_service
        ;;
    "submit" | "Submit" )
        job_cfg=$2
        submit_job
        ;;
    "clean" | "Clean" )
        job_name=$2
        clean_job
        ;;
    "stat" | "Stat" )
        stat_job
        ;;
    "retry" | "Retry" )
        job_name=$2
        retry_failed_tasks
        ;;
    "version" | "Version" )
        ver_import
        ;;
    * )
        usage
        exit 1
        ;;
esac

exit 0

