<?php
Wind::import('ADMIN:service.dm.AdminUserDm');
/**
 * 后台用户服务类
 *
 * 后台用户服务类,职责:<ol>
 * <li>login,用户登录</li>
 * <li>logout,用户退出</li>
 * <li>isLogin,用户是否已登录</li>
 * </ol>
 *
 * @author Qiong Wu <papa0924@gmail.com> 2011-10-17
 * @copyright ©2003-2103 phpwind.com
 * @license http://www.windframework.com
 * @version $Id: AdminUserService.php 18021 2012-09-11 02:48:16Z long.shi $
 * @package admin
 * @subpackage library.service
 */
class AdminUserService {
	protected $cookieName = 'AdminUser';
	const FOUNDER = 'founder';
	const USER = 'user';

	/**
	 * 根据用户名来判断用户的合法性
	 *
	 * 合法用户返回true，非法用户返回false
	 * @param string $username
	 * @return boolean
	 */
	public function verifyUserByUsername($username) {
		return $this->loadUser()->getUserByName($username);
	}

	/**
	 * 验证用户是否有访问菜单的权限
	 *
	 * @param AdminUserDm $dm 用户ID
	 * @param string $m 路由信息Module
	 * @param string $c 路由信息Controller
	 * @param string $a 路由信息Action
	 * @return true|Error
	 */
	public function verifyUserMenuAuth($dm, $m, $c, $a) {
		$_menus = $this->getAuths($dm);
		if ($_menus === '-1') return true;
		if (empty($_menus) || !is_array($_menus)) return new PwError('ADMIN:menu.fail.allow');
		/* @var $menuService AdminMenuService */
		$menuService = Wekit::load('ADMIN:service.srv.AdminMenuService');
		$authStruts = $menuService->getMenuAuthStruts();
		$authKeys = array();
		if (isset($authStruts[$m][$c]['_all'])) $authKeys += $authStruts[$m][$c]['_all'];
		if (isset($authStruts[$m][$c][$a])) $authKeys += $authStruts[$m][$c][$a];
		foreach ($authKeys as $_key)
			if (in_array($_key, $_menus)) return true;
		return new PwError('ADMIN:menu.fail.allow');
	}

	/**
	 * 根据用户ID,获取这个用户的全部后台权限菜单.
	 *
	 * 返回值定义:<pre>
	 * 1. -1 			所有权限
	 * 2. array()		没有任何权限
	 * 3. array('home') 只有home菜单权限
	 * </pre>
	 * @param AdminUserDm $dm
	 * @return array|PwError|-1
	 */
	public function getAuths($dm) {
		list($uid, $username) = array($dm->getUid(), $dm->getUsername());
		if ($this->isFounder($username)) return '-1';
		
		/* @var $authDS AdminAuth */
		$authService = Wekit::load('ADMIN:service.AdminAuth');
		$userAuths = $authService->findByUid($uid);
		if (empty($userAuths['roles'])) return array();
		
		$roles = explode(',', $userAuths['roles']);
		/* @var $roleService AdminRole */
		$roleService = Wekit::load('ADMIN:service.AdminRole');
		$roles = $roleService->findRolesByNames($roles);
		if ($roles instanceof PwError) return new PwError('ADMIN:fail');
		
		$_tmp = '';
		foreach ($roles as $role) {
			$_tmp .= $role['auths'] . ',';
		}
		return empty($_tmp) ? array() : explode(',', trim($_tmp, ','));
	}

	/**
	 * 判断用户是否登录
	 *
	 * 判断用户是否登录,并返回用户对象.登录则返回<i>true</i>,没有登录则返回<i>false</i>.
	 * 用户登录信息保存在<i>cookie</i>中,用户后台管理平台的cookie信息保存数据前缀为<i>admin</i>.
	 *
	 * @return AdminUserDm|boolean
	 */
	public function isLogin() {
		/* @var $user AdminUserDm */
		$user = new AdminUserDm();
		$cookie = Pw::getCookie($this->cookieName);
		if (!$cookie) return false;
		list($type, $u, $pwd) = explode("\t", Pw::decrypt($cookie));
		if ($type == self::FOUNDER) {
			$founders = $this->_getFounders();
			if (!isset($founders[$u])) return false;
			list($md5pwd, $salt) = explode('|', $founders[$u], 2);
			if (Pw::getPwdCode($md5pwd) != $pwd) return false;
			$user->setUsername($u);
		} else {
			$r = Wekit::load('user.PwUser')->getUserByUid($u);
			if (!$r) return false;
			if (Pw::getPwdCode($r['password']) != $pwd) return false;
			$user->setUid($u);
			$user->setUsername($r['username']);
			$user->setEmail($r['email']);
		}
		
		return $user;
	}

	/**
	 * 后台用户登录服务
	 *
	 * 后台用户登录服务,并返回用户对象.参数信息:<code>
	 * $loginInfo: AdminUser
	 * </code>
	 *
	 * @param AdminUserDm $loginInfo 用户信息,
	 * @return boolean|array
	 */
	public function login($loginInfo) {
		$conf = $this->_getFounders();
		if ($conf instanceof PwError) return $conf;
		list($username, $password) = array($loginInfo->getUsername(), $loginInfo->getPassword());
		if (isset($conf[$username])) {
			list($md5pwd, $salt) = explode('|', $conf[$username], 2);
			if (md5($password . $salt) != $md5pwd) return new PwError('ADMIN:login.fail.user.illegal');
			$loginInfo->setUid('');
			$loginInfo->setPassword('');
			$loginInfo->setUsername($loginInfo->getUsername());
			$loginInfo->setEmail('');
			$cookie = Pw::encrypt(self::FOUNDER . "\t" . $username . "\t" . Pw::getPwdCode($md5pwd));
		} else {
			$user = $this->loadUserService()->verifyUser($loginInfo->getUsername(), $loginInfo->getPassword());
			if ($user instanceof PwError) return new PwError('ADMIN:login.fail.user.illegal');
			/* @var $auth AdminAuth */
			$auth = Wekit::load('ADMIN:service.AdminAuth');
			if (!$auth->findByUid($user['uid'])) return new PwError('ADMIN:login.fail.allow');
			
			$loginInfo->setUid($user['uid']);
			$loginInfo->setPassword('');
			$loginInfo->setUsername($user['username']);
			$loginInfo->setEmail($user['email']);
			$u = Wekit::load('user.PwUser')->getUserByUid($user['uid']);
			$cookie = Pw::encrypt(self::USER . "\t" . $user['uid'] . "\t" . Pw::getPwdCode($u['password']));
		}

		Pw::setCookie($this->cookieName, $cookie, 1800);
		return $loginInfo;
	}

	/**
	 * 后台用户退出服务
	 *
	 * @return boolean
	 */
	public function logout() {
		return Pw::setCookie($this->cookieName, '', -1);
	}

	/**
	 * @return PwUserService
	 */
	private function loadUserService() {
		try {
			return Wekit::load('user.srv.PwUserService');
		} catch (Exception $e) {
			throw new PwDependanceException('EXCEPTION:admin.userservice',
				array('{service}' => __CLASS__, '{userservice}' => Wind::getRealPath('user.srv.PwUserService')));
		}
	}

	/**
	 * @return PwUser
	 */
	private function loadUser() {
		try {
			return Wekit::load('user.PwUser');
		} catch (Exception $e) {
			throw new PwDependanceException('admin.userservice',
				array('{service}' => __CLASS__, '{userservice}' => Wind::getRealPath('SRV:user.PwUser')));
		}
	}

	/**
	 * 读取创始人配置文件
	 *
	 * @return PwError|array
	 */
	private function _getFounders() {
		$file = Wind::getRealPath('CONF:founder.php', true);
		if (!WindFile::isFile($file)) return new PwError('ADMIN:founder.file.exist.fail');
		$conf = include $file;
		if (!is_array($conf)) return new PwError('ADMIN:founder.file.read.fail');
		return $conf;
	}

	/**
	 * 根据配置文件，查看是否管理员
	 *
	 * @param string $username
	 */
	public function isFounder($username) {
		$founders = $this->_getFounders();
		return isset($founders[$username]);
	}
}
?>