<?php
defined('WEKIT_VERSION') || exit('Forbidden');

Wind::import('WIND:utility.WindCookie');

/**
 * 工具类库
 *
 * @author Jianmin Chen <sky_hold@163.com>
 * @copyright ©2003-2103 phpwind.com
 * @license http://www.phpwind.com
 * @version $Id: Pw.php 18147 2012-09-11 10:59:25Z liusanbian $
 * @package library
 */
class Pw {

	/**
	 * 取得指定名称的cookie值
	 *
	 * @param string $name cookie名称
	 * @param string $pre cookie前缀,默认为null即没有前缀
	 * @return boolean
	 */
	public static function getCookie($name) {
		$pre = Wekit::config('site', 'cookie.pre');
		$pre && $name = $pre . '_' . $name;
		return WindCookie::get($name);
	}

	/**
	 * 设置cookie
	 *
	 * @param string $name cookie名称
	 * @param string $value cookie值,默认为null
	 * @param string|int $expires 过期时间,默认为null即会话cookie,随着会话结束将会销毁
	 * @param string $pre cookie前缀,默认为null即没有前缀
	 * @param boolean $httponly
	 * @return boolean
	 */
	public static function setCookie($name, $value = null, $expires = null, $httponly = false) {
		$domain = Wekit::config('site', 'cookie.domain');
		$path = Wekit::config('site', 'cookie.path');
		$pre = Wekit::config('site', 'cookie.pre');
		$pre && $name = $pre . '_' . $name;
		$expires && $expires += self::getTime();
		return WindCookie::set($name, $value, false, $expires, $path, $domain, false, $httponly);
	}

	/**
	 * 加密方法
	 *
	 * @param string $str
	 * @param string $key
	 * @return string
	 */
	public static function encrypt($str, $key = '') {
		$key || $key = Wekit::config('site', 'hash');
		/* @var $security IWindSecurity */
		$security = Wind::getApp()->getComponent('security');
		return base64_encode($security->encrypt($str, $key));
	}
	
	/**
	 * 解密方法
	 *
	 * @param string $str
	 * @param string $key
	 * @return string
	 */
	public static function decrypt($str, $key = '') {
		$key || $key = Wekit::config('site', 'hash');
		/* @var $security IWindSecurity */
		$security = Wind::getApp()->getComponent('security');
		return $security->decrypt(base64_decode($str), $key);
	}

	/**
	 * 密码加密存储
	 *
	 * @param string $pwd
	 * @return string
	 */
	public static function getPwdCode($pwd) {
		return md5($pwd . Wekit::config('site', 'hash'));
	}

	/**
	 * 求取字符串长度
	 *
	 * @param string $string
	 * @return string
	 */
	public static function strlen($string) {
		return WindString::strlen($string, Wind::getApp()->getResponse()->getCharset());
	}

	/**
	 * 字符串截取
	 *
	 * @param string $string
	 * @param int $length
	 * @param int $start
	 * @param bool $dot
	 */
	public static function substrs($string, $length, $start = 0, $dot = true) {
		return WindString::substr($string, $start, $length, Wind::getApp()->getResponse()->getCharset(), $dot);
	}
	
	/**
	 * 清理包含WindCode的字符串
	 *
	 * @param string $text
	 * @param bool $stripTags
	 */
	public static function stripWindCode($text,$stripTags = false) {
		$pattern = array();
		if (strpos($text, '[post]') !== false && strpos($text, '[/post]') !== false) {
			$pattern[] = '/\[post\].+?\[\/post\]/is';
		}
		if (strpos($text, '[img]') !== false && strpos($text, '[/img]') !== false) {
			$pattern[] = '/\[img\].+?\[\/img\]/is';
		}
		if (strpos($text, '[hide=') !== false && strpos($text, '[/hide]') !== false) {
			$pattern[] = '/\[hide=.+?\].+?\[\/hide\]/is';
		}
		if (strpos($text, '[sell') !== false && strpos($text, '[/sell]') !== false) {
			$pattern[] = '/\[sell=.+?\].+?\[\/sell\]/is';
		}
		$pattern[] = '/\[[a-zA-Z]+[^]]*?\]/is';
		$pattern[] = '/\[\/[a-zA-Z]*[^]]\]/is';
	
		$text = preg_replace($pattern, '', $text);
		$stripTags && $text = strip_tags($text);
		return $text;
	}

	/**
	 * 将数据用json加密
	 *
	 * @param mixed $value 需要加密的数据
	 * @param string $charset 字符编码
	 * @return string 加密后的数据
	 */
	public static function jsonEncode($value) {
		Wind::import('Wind:utility.WindJson');
		return WindJson::encode($value, Wind::getApp()->getResponse()->getCharset());
	}

	/**
	 * 将json格式数据解密
	 *
	 * @param string $value 待解密的数据
	 * @param string $charset 解密后字符串编码
	 * @return mixed 解密后的数据
	 */
	public static function jsonDecode($value) {
		Wind::import('Wind:utility.WindJson');
		return WindJson::decode($value, true, Wind::getApp()->getResponse()->getCharset());
	}
	
	/**
	 * 将数组简易地转换成json格式
	 *
	 * @param array $var
	 * @return string
	 */
	public static function array2str($var) {
		if (empty($var) || !is_array($var)) return '{}';
		$str = '';
		foreach ($var as $k => $v) {
			$str .= "'" . addcslashes($k, "'") . "' : " . (is_array($v) ? self::array2str($v) : "'" . addcslashes($v, "'") . "'") . ",";
		}
		return '{' . rtrim($str, ',') . '}';
	}
	
	/**
	 * 从数组(A)中找出指定键值的子集
	 *
	 * @param array $var 数组(A)
	 * @param array $vkeys 指定键值
	 * @return array
	 */
	public static function subArray($var, $vkeys) {
		if (!is_array($var) || !is_array($vkeys)) return array();
		$result = array();
		foreach ($vkeys as $key) {
			if (isset($var[$key])) $result[$key] = $var[$key];
		}
		return $result;
	}
	
	/**
	 * 页码转sql
	 *
	 * @param int $page 分页
	 * @param int $perpage 每页显示数
	 * @return array <1.start 2.limit>
	 */
	public static function page2limit($page, $perpage = 10) {
		$limit = intval($perpage);
		$start = max(($page - 1) * $limit, 0);
		return array($start, $limit);
	}

	/**
	 * 将时间字串转化成零时区时间戳返回
	 *
	 * @param string $str 格式良好的时间串
	 * @return int
	 */
	public static function str2time($str) {
		$timestamp = strtotime($str);
		$timezone = Wekit::config('site', 'time.timezone');
		$sOffset = $timezone ? $timezone * 3600 : 0;
		return $timestamp - $sOffset;
	}

	/**
	 * 时间戳转字符串
	 *
	 * @example Y-m-d H:i:s  2012-12-12 12:12:12
	 * @param int $timestamp 时间戳
	 * @param string $format 时间格式
	 * @param int $sOffset 时间矫正值
	 * @return string
	 */
	public static function time2str($timestamp, $format = 'Y-m-d H:i') {
		if (!$timestamp) return '';
		if ($format == 'auto') return self::_time2cpstr($timestamp);
		if ($timezone = Wekit::config('site', 'time.timezone')) $timestamp += $timezone * 3600;
		return gmdate($format, $timestamp);
	}

	protected static function _time2cpstr($timestamp) {
		$current = self::getTime();
		$decrease = $current - $timestamp;
		if ($decrease < 0) return self::time2str($timestamp);
		if ($decrease < 60) return $decrease . '秒前';
		if ($decrease < 3600) return ceil($decrease / 60) . '分钟前';
		$decrease = self::str2time(self::time2str($current, 'Y-m-d')) - self::str2time(self::time2str($timestamp, 'Y-m-d'));
		if ($decrease == 0) return self::time2str($timestamp, 'H:i');
		if ($decrease == 86400) return '昨天' . self::time2str($timestamp, 'H:i');
		if ($decrease == 172800) return '前天' . self::time2str($timestamp, 'H:i');
		if (self::time2str($timestamp, 'Y') == self::time2str($current, 'Y')) return self::time2str($timestamp, 'm-d H:i');
		return self::time2str($timestamp);
	}

	/**
	 * 获取矫正过的时间戳值
	 *
	 * @return int
	 */
	public static function getTime() {
// 		return Wekit::app()->timestamp;
		if ($timestamp = Wekit::getGlobal('timestamp')) return $timestamp;
		$timestamp = time();
		if ($cvtime = Wekit::config('site', 'time.cv')) $timestamp += $cvtime * 60;
		Wekit::setGlobal($timestamp, 'timestamp');
		return $timestamp;
	}
	
	/**
	 * 获取图片路径
	 *
	 * @param string $path
	 * @param int $thumb
	 * @return string
	 */
	public static function getPath($path, $ifthumb) {
		$storage = Wind::getApp()->getComponent('storage');
		return $storage->get($path, $ifthumb);
	}
	
	/**
	 * 获取用户头像地址
	 *
	 * @param int $uid
	 * @param string $size <m.中头像 s.小头像>
	 * @return string
	 */
	public static function getAvatar($uid, $size = 'middle') {
		$file = $uid . (in_array($size, array('middle', 'small')) ? '_' . $size : '') . '.jpg';
		return Wekit::app()->attach . '/avatar/' . self::getUserDir($uid) . '/' . $file;
	}
	
	/**
	 * 获取用户头像存储目录
	 *
	 * @param int $uid
	 * @return string
	 */
	public static function getUserDir($uid) {
		$uid = sprintf("%09d", $uid);
		return substr($uid, 0, 3) . '/' . substr($uid, 3, 2) . '/' . substr($uid, 5, 2);
	}
	
	/**
	 * 删除附件
	 *
	 * @param string $path 附件相对地址
	 * @param int $ifthumb 缩略图
	 * @return bool
	 */
	public static function deleteAttach($path, $ifthumb = 0) {
		$storage = Wind::getApp()->getComponent('storage');
		return $storage->delete($path, $ifthumb);
	}
	
	/**
	 * 删除本地文件
	 *
	 * @param string $filename 文件绝对地址
	 * @return bool
	 */
	public static function deleteFile($filename) {
		return WindFile::del(WindSecurity::escapePath($filename, true));
	}

	/**
	 * 返回html checked
	 *
	 * @param boolean $var
	 * @return string
	 */
	public static function ifcheck($var) {
		return $var ? ' checked' : '';
	}

	/**
	 * 返回html selected
	 *
	 * @param boolean $var
	 * @return string
	 */
	public static function isSelected($var) {
		return $var ? ' selected' : '';
	}

	/**
	 * 返回html current
	 *
	 * @param boolean $var
	 * @return string
	 */
	public static function isCurrent($var) {
		return $var ? ' current' : '';
	}

	/**
	 * 编码转换
	 *
	 * @param string $string 内容字符串
	 * @param string $fromEncoding 原编码
	 * @return string
	 */
	public static function convert($string, $toEncoding, $fromEncoding = '') {
		!$fromEncoding && $fromEncoding = Wind::getApp()->getResponse()->getCharset();
		return WindConvert::convert($string, $toEncoding, $fromEncoding);
	}

	/**
	 * 检查是否在线
	 *
	 * @param int $time lastvisit
	 * @return bool
	 */
	public static function checkOnline($time) {
		$onlinetime = $pre = Wekit::config('site', 'onlinetime');
		if ($time + $onlinetime * 60 > self::getTime()) {
			return true;
		}
		return false;
	}

	public static function getstatus($status, $b, $len = 1) {
		return $status >> --$b & (1 << $len) - 1;
	}
}