<?php
defined('WEKIT_VERSION') || exit('Forbidden');

Wind::import('WINDID:service.user.WindidUser');
Wind::import('SRV:user.dm.PwUserInfoDm');

/**
 * 用户操作数据接口
 *
 * @author xiaoxia.xu <xiaoxia.xuxx@aliyun-inc.com>
 * @copyright ©2003-2103 phpwind.com
 * @license http://www.phpwind.com
 * @version $Id: PwUser.php 14889 2012-07-27 08:08:42Z xiaoxia.xuxx $
 * @package src.service.user
 */
class PwUser {
	
	const FETCH_MAIN = 1; //获取用户基本信息，比如帐号、邮箱
	const FETCH_DATA = 2; //获取用户数据，比如积分、发帖数等论坛统计信息
	const FETCH_INFO = 4; //获取用户基本资料
	const FETCH_ALL = 7;

	const STATUS_UNCHECK = 1;//未验证用户
	const STATUS_UNACTIVE = 2;//未激活用户
	const STATUS_BAN_AVATAR = 4;//用户被禁止头像
	const STATUS_BAN_SIGN = 8;//用户被禁止签名
	
	const ALLOW_LOGIN_ADMIN = 16;//该用户是否后后台权限
	
	/** 
	 * 根据用户ID获得用户数据信息
	 *
	 * @param int $uid 	用户ID
	 * @param int $type 获得信息的类型
	 * @return array
	 */
	public function getUserByUid($uid, $type = self::FETCH_MAIN) {
		if (empty($uid)) return array();
		return $this->_getDao($type)->getUserByUid($uid);
	}

	/** 
	 * 根据用户名字获得用户数据信息
	 *
	 * @param string $username  用户名
	 * @param int $type   		获得信息的类型
	 * @return array
	 */
	public function getUserByName($username, $type = self::FETCH_MAIN) {
		if (empty($username)) return array();
		return $this->_getDao($type)->getUserByName($username);
	}
	
	/**
	 * 通过邮箱获取用户信息
	 *
	 * @param string $email 邮箱
	 * @param int $type 用户信息类型
	 * @return array
	 */
	public function getUserByEmail($email, $type = self::FETCH_MAIN) {
		if (empty($email)) return array();
		return $this->_getDao($type)->getUserByEmail($email);
	}
	
	/** 
	 * 跟据用户ID列表获取用户列表
	 *
	 * @param array $uids 用户列表
	 * @param int $type
	 */
	public function fetchUserByUid($uids, $type = self::FETCH_MAIN) {//getUserListByUids($uids, $type = self::FETCH_MAIN) {
		if (empty($uids) || !is_array($uids)) return array();
		return $this->_getDao($type)->fetchUserByUid($uids);
	}
	
	/** 
	 * 根据用户名
	 *
	 * @param array $names 用户名列表
	 * @param int $type
	 * @return array
	 */
	public function fetchUserByName($names, $type = PwUser::FETCH_MAIN) {//getUserListByNames($names, $type = PwUser::FETCH_MAIN) {
		if (empty($names)) return array();
		return $this->_getDao($type)->fetchUserByName($names);
	}

	/** 
	 * 根据用户组ID获得该组下的用户列表
	 *
	 * @param int|array $gid  0：普通用户组，-1：VIP用户组，>0特殊组
	 * @param int $limit      查询的条数
	 * @param int $offset 	  开始位置
	 * @return array  array(totalCount, resultList);
	 */
	public function getUserListByGid($gid, $limit = 10, $offset = 1) {
		if (!is_array($gid)) $gid = array(intval($gid));
		return $this->_getDao(self::FETCH_MAIN)->fetchByGroupid($gid, $limit, $offset);
	}
	
	/** 
	 * 根据组ID统计用户信息
	 *
	 * @param array $gids  0：普通用户组，-1：VIP用户组，>0特殊组
	 * @return int
	 */
	public function countByGids($gids) {
		$gids = (array)$gids;
		return $this->_getDao(self::FETCH_MAIN)->countByGroupid($gids);
	}

	/** 
	 * 检查用户的email是否存在 
	 *
	 * @param string $email    待检查的用户email
	 * @param string $username 排除的用户名字
	 * @return boolean
	 */
	public function checkEmailExist($email, $username = '') {
		return $this->_getWindid()->checkEmailExists($email, $username);
	}

	/** 
	 * 编辑用户信息
	 *
	 * @param PwUserInfoDm $dm 用户信息DM
	 * @param int $type 	   更新类型
	 * @return boolean|PwError
	 */
	public function editUser(PwUserInfoDm $dm, $type = self::FETCH_ALL) {
		if (true !== ($result = $dm->beforeUpdate())) return $result;
		if (is_object($dm->dm)) {
			$result = $this->_getWindid()->editUser($dm->dm);
			if ($result instanceof WindidError) return new PwError('USER:user.error.' . $result->getCode());
		}
		$this->_getDao($type)->editUser($dm->uid, $dm->getData(), $dm->getIncreaseData());
		PwSimpleHook::getInstance('PwUser_update')->runDo($dm);
		return true;
	}
	
	/**
	 * 更新用户积分
	 *
	 * @param object $dm
	 * @return bool
	 */
	public function updateCredit(WindidCreditDm $dm) {
		$this->_getWindid()->updateCredit($dm);
		return $this->_getDao(self::FETCH_DATA)->editUser($dm->uid, $this->_getWindid()->getCredit($dm->uid));
	}

	/** 
	 * 添加用户
	 *
	 * @param PwUserInfoDm $dm 用户信息DM
	 * @param int $type 	   添加表
	 * @return int|PwError
	 */
	public function addUser(PwUserInfoDm $dm) {
		if (true !== ($result = $dm->beforeAdd())) return $result;
		if (($uid = $this->_getWindid()->addUser($dm->dm)) instanceof WindidError) {
			return new PwError('USER:user.error.' . $uid->getCode());
		}
		$dm->setUid($uid);
		$this->_getDao(self::FETCH_ALL)->addUser($dm->getData());
		PwSimpleHook::getInstance('PwUser_add')->runDo($dm);
		return $uid;
	}
	
	/**
	 * 激活用户
	 *
	 * @param int $uid
	 * @return bool
	 */
	public function activeUser($uid) {
		if (!$data = $this->_getWindid()->getUserByUid($uid, self::FETCH_ALL)) {
			return false;
		}
		return $this->_getDao(self::FETCH_ALL)->addUser($data);
	}

	/** 
	 * 删除用户信息
	 *
	 * @param int $uid 用户ID
	 * @return boolean
	 */
	public function deleteUser($uid) {
		if (!($uid = intval($uid))) return new PwError('USER:illegal.id');
		$this->_getDao(self::FETCH_ALL)->deleteUser($uid);
		$this->_getWindid()->deleteUser($uid);
		PwSimpleHook::getInstance('PwUser_delete')->runDo($uid);
		return true;
	}
	
	/** 
	 * 根据用户ID列表批量删除用户信息
	 *
	 * @param array $uids 用户ID列表
	 * @return boolean
	 */
	public function batchDeleteUserByUid($uids) {
		if (empty($uids)) return false;
		$uids = (array)$uids;
		$this->_getDao(self::FETCH_ALL)->batchDeleteUser($uids);
		$this->_getWindid()->batchDeleteUser($uids);
		PwSimpleHook::getInstance('PwUser_batchDelete')->runDo($uids);
		return true;
	}
	
	/** 
	 * 获得用户中心对像
	 * 
	 * @return PwBaseDao
	 */
	protected function _getDao($type = self::FETCH_MAIN) {
		$daoMap = array();
		$daoMap[self::FETCH_MAIN] = 'user.dao.PwUserDao';
		$daoMap[self::FETCH_DATA] = 'user.dao.PwUserDataDao';
		$daoMap[self::FETCH_INFO] = 'user.dao.PwUserInfoDao';
		return Wekit::loadDaoFromMap($type, $daoMap, 'user');
	}
	
	/** 
	 * 获得windidDS
	 *
	 * @return WindidUser
	 */
	protected function _getWindid() {
		return Windid::load('user.WindidUser');
	}
}