<?php
Wind::import('SRV:user.dm.PwUserInfoDm');
/**
 * 用户的业务接口
 *
 * @author xiaoxia.xu <xiaoxia.xuxx@aliyun-inc.com>
 * @copyright ©2003-2103 phpwind.com
 * @license http://www.phpwind.com
 * @version $Id: PwUserService.php 15902 2012-08-15 07:41:10Z xiaoxia.xuxx $
 * @package src.service.user.srv
 */
class PwUserService {

	/** 
	 * 验证用户
	 * 
	 * 返回错误码 PwError->getError():
	 * <ul>
	 * <li>-1: 用户不存在</li>
	 * <li>-2: 用户密码错误</li>
	 * <li>-3: 用户安全问题回答不正确</li>
	 * </ul>
	 * 
	 * @param string $username 用户
	 * @param string $password 用户密码
	 * @param int $type 验证类型 <0: 用户名，1：用户id,2：用户email>
	 * @param string $question 安全问题
	 * @param string $answer   安全问题的答案
	 * @return PwError|array
	 */
	public function verifyUser($username, $password, $type = 0, $question = null, $answer = null) {
		$checkSafe = is_null($question) && is_null($answer) ? false : true;
		$result = $this->_getWindidUser()->auth($username, $password, $type, $checkSafe, $question, $answer);
		switch ($result['status']) {
			case 1://用户信息正常
				return $result;
			case -1://用户不存在
				return new PwError('USER:verify.error.name');
			case -2://用户密码错误
				return new PwError('USER:verify.error.pwd');
			case -3://用户安全问题错误
				return new PwError('USER:verify.error.question');
		}
		return $result;
	}

	/** 
	 * 退出系统
	 * 
	 * @return boolean
	 */
	public function logout() {
		$loginUser = PwUserBo::getLoginUser();
		PwSimpleHook::getInstance('PwUserService_logout')->runDo($loginUser);
		$loginUser->reset();
		return Pw::setCookie('winduser', '', -1);
	}
	
	/**
	 * 编辑用户
	 *
	 * @param PwUserInfoDm $dm
	 * @param int $type
	 * @return boolean|PwError
	 */
	public function editUser($dm, $type = PwUser::FETCH_MAIN) {
		/* @var $userDs PwUser */
		$userDs = Wekit::load('user.PwUser');
		$result = $userDs->editUser($dm, $type);
		if ($result instanceof PwError) return $result;
		/*用户资料设置完成-基本资料-service钩子点:s_PwUserService_editUser*/
		PwSimpleHook::getInstance('PwUserService_editUser')->runDo($dm);
		return true;
	}
	
	/** 
	 * 根据用户拥有组信息获得该用户的当前使用的用户身份
	 * <pre>
	 * 用户拥有的组的使用顺序：
	 * <ol>
	 * <li>系统组 》荣誉组 》会员组</li>
	 * <li>然后按组内的id顺序进行变更。</li>
	 * </ol>
	 * </pre>
	 *
	 * @param int $groupid 用户当前用户组
	 * @param array $userGroups 用户拥有的组
	 * @return array (gid, groups)
	 */
	public function caculateUserGroupid($groupid, $userGroups) {
		//TODO 如果是游客/禁止发言/未验证用户组则不变动该用户组
		if (in_array($groupid, array(1, 2, 6))) return array($groupid, array());
		//TODO【当前用户组】 当前用户组设置顺序:系统组/特殊组/普通组
		$groupQue = array('system', 'special');
		$temp = array();
		$time = Pw::getTime();
		foreach ($userGroups as $_gid => $_time) {
			if ($_time == 0 || $_time > $time) {
				$temp[$_gid] = $_time;
			}
		}
		$gid = 0;
		if ($temp) {
			/* @var $groupDs PwUserGroups */
			$groupDs = Wekit::load('usergroup.PwUserGroups');
			$groups = $groupDs->getClassifiedGroups();
			$userGids = array_keys($temp);
			foreach ($groupQue as $_tmpType) {
				/*如果用户拥有系统组或是特殊组，根据用户组ID从小到大排序的第一个用户组为用户当前显示的组*/
				$_tmp = array_intersect(array_keys($groups[$_tmpType]), $userGids);
				if ($_tmp) {
					ksort($_tmp);
					$gid = array_shift($_tmp);
					break;
				}
			}
		}
		//当前用户组过期
		if (!$temp[$groupid]) {
			$groupid = $gid;
		}
		return array($groupid, $temp);
	}

	/** 
	 * 创建登录用户标识
	 *
	 * @param int $uid 用户ID
	 * @param string $password 用户密码
	 * @return boolean
	 */
	public function createIdentity($uid, $password) {
		$identity = Pw::encrypt($uid . "\t" . Pw::getPwdCode($password));
		return Pw::setCookie('winduser', $identity, 31536000);
	}

	/** 
	 * 更新用户登录信息
	 *
	 * @param int $uid 用户ID
	 * @param string $ip 用户IP
	 * @return boolean
	 */
	public function updateLastLoginData($uid, $ip) {
		/* 更新用户登录信息 */
		$dm = new PwUserInfoDm();
		$dm->setUid($uid)->setLastvisit(Pw::getTime())->setLastloginip($ip);
		return $this->_getUserDs()->editUser($dm, PwUser::FETCH_DATA);
	}
	
	/** 
	 * 判断用户是否必须设置安全问题
	 * 如果当前用户所有拥有的组(包括附加组)中有一个组是必须要设置安全问题和答案的，则该用户必须设置安全问题和答案
	 * 
	 * @param int $uid 用户ID
	 * @return boolean
	 */
	public function mustSettingSafeQuestion($uid) {
		$groups = $this->getGidsByUid($uid);
		$mustSettingGroups = Wekit::config('login', 'question.groups');
		return array_intersect($groups, $mustSettingGroups) ? true : false;
	}
	
	/**
	 * 根据用户ID获得该用户拥有的用户组
	 *
	 * @param int $uid 
	 * @return array
	 */
	public function getGidsByUid($uid) {
		if (!$uid) return array();
		$info = Wekit::load('user.PwUser')->getUserByUid($uid, PwUser::FETCH_MAIN);
		return array_merge(explode(',', $info['groups']), array($info['groupid'], $info['memberid']));
	}
	
	/**
	 * 删除用户头像
	 *
	 * TODO 删除用户头像
	 * 
	 * @param int $uid
	 * @return boolean
	 */
	public function deleteAvatar($uid) {
		$filePath = sprintf('avatar/%s/%d', Pw::getUserDir($uid), $uid);
		$_avatar = array('.jpg', '_middle.jpg', '_small.jpg');
		foreach ($_avatar as $_item) {
			Pw::deleteAttach($filePath . $_item, 0);
		}
		return true;
	}
	
	/** 
	 * 获得用户Ds
	 *
	 * @return PwUser
	 */
	private function _getUserDs() {
		return Wekit::load('user.PwUser');
	}
	
	/** 
	 * 获得用户组Ds
	 *
	 * @return PwUserGroups
	 */
	private function _getUserGroupDs() {
		return Wekit::load('usergroup.PwUserGroups');
	}

	/** 
	 * 获得windidUser
	 *
	 * @return WindidUser
	 */
	private function _getWindidUser() {
		return Windid::load('user.WindidUser');
	}
}
