<?php

Wind::import('WINDID:library.WindidRegisterValidator');
/**
 * 用户Api接口
 * 
 * @author Jianmin Chen <sky_hold@163.com>
 * @license http://www.phpwind.com
 * @version $Id: WindidApi.php 8008 2012-04-13 11:27:58Z liusanbian $
 * @package windid.service
 */
class WindidUserApi {
	
	/**
	 * 根据用户id获得用户信息
	 *
	 * @param int $uid 用户ID
	 * @param int $type 信息包含类型
	 * @return array
	 */
	public function getUserByUid($uid, $type = WindidUser::FETCH_ALL) {
		return $this->_getUser()->getUserByUid($uid, $type);
	}
	
	/**
	 * 根据用户名字获得用户信息
	 *
	 * @param string $name 用户名字
	 * @param int $type  信息包含类型
	 * @return array
	 */
	public function getUserByName($name, $type = WindidUser::FETCH_ALL) {
		return $this->_getUser()->getUserByName($name, $type);
	}
	
	/**
	 * 根据用户email获得用户信息
	 *
	 * @param string $email 用户email地址
	 * @param int $type  信息包含类型
	 * @return array
	 */
	public function getUserByEmail($email, $type = WindidUser::FETCH_ALL) {
		return $this->_getUser()->getUserByEmail($email, $type);
	}

	/**
	 * 注册用户
	 * 
	 * @param string $username 用户名
	 * @param string $password 密码
	 * @param string $email 邮箱
	 * @param string $question 安全问题
	 * @param string $answer 答案
	 * @param string $regip 注册用户ip
	 * @return bool|inf 注册是否成功|验证失败时返回一个错误代码
	 */
	public function register($username, $password, $email, $question = '', $answer = '', $regip = '') {
		$data = array();
		$data['username'] = $username;
		$data['password'] = $password;
		$data['email'] = $email;
		$data['question'] = $question;
		$data['answer'] = $answer;
		$data['regip'] = $regip;
		return $this->_getUser()->addUser($data);
	}

	/**
	 * 验证用户信息
	 * 
	 * @param string  $username 用户名
	 * @param string  $password 密码
	 * @param int     $logintype 登录类型<0.用户名 1.uid 2.邮箱>
	 * @param boolean $checkques 是否需要验证安全问题
	 * @param string  $question 安全问题
	 * @param string  $answer 答案
	 * @return array|int 成功时返回用户信息|失败时返回一个错误代码
	 */
	public function auth($username, $password, $logintype = 0, $checkques = false, $question = '', $answer = '') {
		$info = $this->logInfo($username, $logintype);
		if (empty($info)) return USER_NOT_EXISTS;//用户不存在
		$password = $this->_getUser()->buildPassword($password, $info['salt']);
		if ($password != $info['password']) return PASSWORD_ERROR;//用户密码错误
		if (false === $checkques) return $info;
		if ($question == $info['question'] && $answer == $info['answer']) return $info;
		return QUESTION_ERROR;//用户安全问题错误
	}

	/**
	 * 获取同步登录代码
	 *
	 * @param int $uid 用户uid
	 * @return string 返回一窜可以实现同步登录的<script>代码
	 */
	public function synlogin($uid) {
		$app = Windid::load('app.WindidApp');
		$applist = $app->getList();
		$synlogin = '';
		if (count($applist) > 1) {
			$info = $this->_getUser()->getUserByUid($uid);
			foreach ($applist as $key => $value) {
				$url = $app->urlformat($value['siteurl'], $value['apifile'], $value['secretkey'], 'User', 'synlogin', array(
					'user' => Windid::strcode($uid . "\t" . $info['username'] . "\t" . $info['password'] . "\t" . TIMESTAMP, $value['secretkey'])));
				$synlogin .= "<script type=\"text/javascript\" src=\"$url\"></script>";
			}
		}
		return $synlogin;
	}

	/**
	 * 获取同步退出代码
	 *
	 * @return string 返回一窜可以实现同步退出的<script>代码
	 */
	public function synloginout() {}

	/**
	 * 修改用户密码
	 * 
	 * @param int $uid 用户ID
	 * @param string $username 用户名
	 * @param string $oldpw 用户密码
	 * @param string $newpw 要修改为的新密码
	 * @param strng $email 要修改为的新邮箱
	 * @return bool|int true|失败时返回一个错误代码
	 */
	public function changePassword($uid, $username, $oldpw, $newpw, $email) {
		return $this->_getUser()->changePassword($uid, $username, $oldpw, $newpw, $email);
	}

	/**
	 * 更改用户信息
	 *
	 * @param int $uid 用户信息
	 * @param array $data 用户数据
	 * @return boolean
	 */
	public function editUser($uid, $data, $type = WindidUser::FETCH_MAIN) {
		return $this->_getUser()->editUser($uid, $data, $type);
	}
	
	/**
	 * 删除用户及相关信息
	 *
	 * @param int $uid 用户uid
	 * @return bool
	 */
	public function deleteUser($uid, $type = WindidUser::FETCH_ALL) {
		return $this->_getUser()->deleteUser($uid, $type);
	}
	
	/**
	 * 验证用户密码是否合法
	 *
	 * @param string $password 用户密码
	 * @return bool|int true|失败时返回一个错误
	 */
	public function checkPassword($password) {
		return WindidRegisterValidator::checkPassword($password);
	}

	/**
	 * 验证用户名是否合法
	 *
	 * @param string $username 用户名
	 * @return bool|string true|失败时返回一个错误
	 */
	public function checkName($username) {
		return WindidRegisterValidator::checkName($username);
	}

	/**
	 * 检查指定的邮箱是否合法
	 *
	 * @param string $email 邮箱
	 * @param string $username 除该用户外...
	 * @return bool true|false
	 */
	public function checkEmail($email, $username = '') {
		if (true !== ($check = WindidRegisterValidator::checkEmail($email))) return $check;
		return $this->_getUser()->checkEmailExists($email, $username);
	}

	/**
	 * 获取用户信息
	 * 
	 * @param string $value 用户唯一凭证(可能是用户名、uid、email)
	 * @param int $loginType 登录类型，0为用户名，1为用户ID，2为用户email
	 * @return array 用户信息
	 */
	private function logInfo($value, $loginType = 0) {
		$user = $this->_getUser();
		switch (intval($loginType)) {
			case 0:
				$info = $user->getUserByName($value);
				break;
			case 1:
				$info = $user->getUserByUid($value);
				break;
			case 2:
				$info = $user->getUserByEmail($value);
				break;
			default:
				break;
		}
		return $info;
	}
	
	/**
	 * 获得用户信息
	 * 
	 * @return WindidUserService
	 */
	private function _getUser() {
		return Windid::load('user.srv.WindidUserService');
	}
}