<?php
Wind::import('WINDID:service.config.WindidConfig');
Wind::import('WINDID:library.WindidError');

/**
 * 用户信息合法性校验器
 * 
 * @author Jianmin Chen <sky_hold@163.com>
 * @license http://www.phpwind.com
 * @version $Id: WindidUserValidator.php 12158 2012-06-19 06:53:36Z liusanbian $
 * @package Windid.library
 */
class WindidUserValidator {
	
	const NAME_EMPTY = -1;
	const NAME_LEN = -2;
	const NAME_ILLEGAL_CHAR = -3;
	const NAME_FORBIDDENNAME = -4;
	const NAME_DUPLICATE = -5;
	
	const EMAIL_EMPTY = -6;
	const EMAIL_ILLEGAL = -7;
	const EMAIL_WHITE_LIST = -8;
	const EMAIL_BLACK_LIST = -9;
	const EMAIL_DUPLICATE = -10;

	const PASSWORD_LEN = -11;
	const PASSWORD_ILLEGAL_CHAR = -12;
	const PASSWORD_ERROR = -13;

	const USER_NOT_EXISTS = -14;

	private static $_illegalChar = array("\\", '&', ' ', "'", '"', '/', '*', ',', '<', '>', "\r", "\t", "\n", '#', '%', '?', '　');
	
	/**
	 * 是否含有非法字符
	 *
	 * @param string $str 待检查的字符串
	 * @return string
	 */
	public static function hasIllegalChar($str) {
		return str_replace(self::$_illegalChar, '', $str) != $str;
	}

	/**
	 * 检查用户名是否通过
	 *
	 * @param string $username 用户名
	 * return bool|WindidValidatorException true|失败时返回一个错误对象
	 */
	public static function checkName($username) {
		if (($result = self::isNameValid($username)) !== true) {
			return $result;
		}
		if (self::_getUserService()->getUserByName($username)) {
			return new WindidError(self::NAME_DUPLICATE);
		}
		return true;
	}
	
	/**
	 * 用户名是否有效
	 *
	 * @param string $username 用户名
	 * @return int|boolean
	 */
	public static function isNameValid($username) {
		if (!$username) return new WindidError(self::NAME_EMPTY);
		if (!self::isNameLenValid($username)) return new WindidError(self::NAME_LEN);
		if (self::hasIllegalChar($username)) return new WindidError(self::NAME_ILLEGAL_CHAR);
		if ($forbiddenname = self::getConfig()->forbiddenname) {
			foreach ($forbiddenname as $key => $value) {
				if ($value !== '' && strpos($username, $value) !== false) {
					return new WindidError(self::NAME_FORBIDDENNAME);
				}
			}
		}
		return true;
	}
	
	/**
	 * 用户名长度是否有效
	 *
	 * @param string $username 判断的长度
	 * @return boolean
	 */
	public static function isNameLenValid($username, $charset = 'utf8') {
		Wind::import('WIND:utility.WindString');
		$len = WindString::strlen($username, $charset);
		return $len >= self::getConfig()->namelength->min && $len <= self::getConfig()->namelength->max;
	}

	/**
	 * 验证邮箱是否通过
	 *
	 * @param string $email 邮箱
	 * @param string $username 用户名<除该用户之外是否存在重名邮箱>
	 * @return bool|WindidValidatorException true|失败时返回一个错误对象
	 */
	public static function checkEmail($email, $username = '') {
		if (($result = self::isEmailValid($email)) !== true) {
			return $result;
		}
		if (self::_getUserService()->checkEmailExists($email, $username)) {
			return new WindidError(self::EMAIL_DUPLICATE);
		}
		return true;
	}

	/**
	 * 检查用户邮箱
	 *
	 * @param string $email 待检查的邮箱
	 * @return boolean|int
	 */
	public static function isEmailValid($email) {
		if (!$email) return new WindidError(self::EMAIL_EMPTY);
		if (false === WindValidator::isEmail($email)) return new WindidError(self::EMAIL_ILLEGAL);
		$config = self::getConfig();
		if ($config->emailverifytype == 1 && !self::_inEmailWhiteList($email)) return new WindidError(self::EMAIL_WHITE_LIST);
		if ($config->emailverifytype == 2 && self::_inEmailBlackList($email)) return new WindidError(self::EMAIL_BLACK_LIST);
		return true;
	}

	/**
	 * 检查密码
	 *
	 * @param string $password 待检查的密码
	 * @return boolean|int
	 */
	public static function isPasswordValid($password) {
		if (!self::isPasswordLenValid(strlen($password))) return new WindidError(self::PASSWORD_LEN);
		//if (self::hasIllegalChar($password)) return new WindidError(self::PASSWORD_ILLEGAL_CHAR);
		return true;
	}

	/**
	 * 检查密码长度
	 *
	 * @param int $len 待检查的密码长度
	 * @return boolean
	 */
	public static function isPasswordLenValid($len) {
		$config = self::getConfig();
		if ($config->passwordlength->min && $len < $config->passwordlength->min) return false;
		if ($config->passwordlength->max && $len > $config->passwordlength->max) return false;
		return true;
	}

	public static function addCheck(WindidUserDm $dm) {
		$data = $dm->getData();
		(($result = self::checkName($data['username'])) === true) &&
			(($result = self::checkEmail($data['email'])) === true) &&
			(($result = self::isPasswordValid($data['password'])) === true);
		return $result;
	}

	public static function updateCheck(WindidUserDm $dm) {
		if (!$user = self::_getUserService()->getUserByUid($dm->uid)) {
			return new WindidError(self::USER_NOT_EXISTS);
		}
		$data = $dm->getData();
		if (isset($data['username']) && $data['username'] != $user['username']) {
			if (($result = self::checkName($data['username'])) !== true) return $result;
			$dm->isNameChange = true;
		}
		if (isset($data['email']) && $data['email'] != $user['email']
			&& ($result = self::checkEmail($data['email'], $user['username'])) !== true) {
			return $result;
		}
		if ($dm->isVerifyPwd && WindidUtility::buildPassword($dm->oldPassword, $user['salt']) != $user['password']) {
			return new WindidError(self::PASSWORD_ERROR);
		}
		if (isset($data['password'])) {
			if (($result = self::isPasswordValid($data['password'])) !== true) return $result;
			$dm->isPasswordChange = true;
		}
		return true;
	}
	
	/**
	 * 获得注册配置信息
	 * 
	 * @return WindidConfig
	 */
	public static function getConfig() {
		static $config = null;
		if (null === $config) {
			Wind::import('WINDID:service.config.bo.WindidConfigBo');
			$config = WindidConfigBo::getInstance('reg');
		}
		return $config;
	}

	/**
	 * email是否在白名单中
	 *
	 * @param string $email 待检查的email
	 * @return boolean
	 */
	private static function _inEmailWhiteList($email) {
		if (!($whitelist = self::getConfig()->emailwhitelist)) return true;
		foreach ($whitelist as $key => $val) {
			if (strpos($email, "@" . $val) !== false) return true;
		}
		return false;
	}

	/**
	 * email是否在黑名单中
	 *
	 * @param string $email 待检查的email
	 * @return boolean
	 */
	private static function _inEmailBlackList($email) {
		if (!($blacklist = self::getConfig()->emailblacklist)) return false;
		foreach ($blacklist as $key => $val) {
			if (strpos($email, "@" . $val) !== false) return true;
		}
		return false;
	}

	/** 
	 *
	 * @return WindidUser
	 */
	private static function _getUserService() {
		return Windid::load('user.WindidUser');
	}
}