<?php
class Logger {
	const LEVEL_INFO = 1;
	const LEVEL_TRACE = 2;
	const LEVEL_DEBUG = 3;
	const LEVEL_ERROR = 4;
	const LEVEL_PROFILE = 5;
	const TOKEN_BEGIN = 'begin:';
	const TOKEN_END = 'end:';
	private $_autoFlush = 1000;
	private $_logs = array();
	private $_logCount = 0;
	private $_profiles = array();
	private $_logFile;
	private $_maxFileSize = 100;

	/**
	 * @param string $msg
	 */
	public function info($msg, $type = 'wind.system') {
		$this->log($msg, self::LEVEL_INFO, $type);
	}

	/**
	 * @param string $msg
	 */
	public function trace($msg, $type = 'wind.system') {
		$this->log($msg, self::LEVEL_TRACE, $type);
	}

	/**
	 * @param string $msg
	 */
	public function debug($msg, $type = 'wind.system') {
		$this->log($msg, self::LEVEL_DEBUG, $type);
	}

	/**
	 * @param string $msg
	 */
	public function error($msg, $type = 'wind.core') {
		$this->log($msg, self::LEVEL_ERROR, $type);
	}

	/**
	 * @param $msg
	 * @param $type
	 */
	public function profileBegin($msg, $type = 'wind.core') {
		$this->log('begin:' . trim($msg), self::LEVEL_PROFILE, $type);
	}

	/**
	 * @param $msg
	 * @param $type
	 */
	public function profileEnd($msg, $type = 'wind.core') {
		$this->log('end:' . trim($msg), self::LEVEL_PROFILE, $type);
	}

	/**
	 * @param string $msg
	 * @param const  $logType
	 */
	public function log($msg, $level = self::INFO, $type = 'wind.system') {
		$this->_logCount >= $this->_autoFlush && $this->flush();
		if ($level === self::LEVEL_PROFILE)
			$this->_logs[] = $this->_build($msg, $level, $type, microtime(true), $this->getMemoryUsage(false));
		elseif ($level === self::LEVEL_DEBUG)
			$this->_logs[] = $this->_build($msg, $level, $type, microtime(true));
		else
			$this->_logs[] = $this->_build($msg, $level, $type);
		$this->_logCount++;
	}

	/**
	 * @param string $dst
	 * @return boolean
	 */
	public function flush() {
		if (empty($this->_logs)) return false;
		if (!$fileName = $this->_getFileName()) return false;
		writeover($fileName, join("", $this->_logs), 'a');
		$this->_logs = array();
		$this->_logCount = 0;
		return true;
	}

	/**
	 * @param $peak
	 * @return int
	 */
	public function getMemoryUsage($peak = true) {
		if ($peak && function_exists('memory_get_peak_usage'))
			return memory_get_peak_usage();
		elseif (function_exists('memory_get_usage'))
			return memory_get_usage();
		$pid = getmypid();
		if (strncmp(PHP_OS, 'WIN', 3) === 0) {
			exec('tasklist /FI "PID eq ' . $pid . '" /FO LIST', $output);
			return isset($output[5]) ? preg_replace('/[\D]/', '', $output[5]) * 1024 : 0;
		} else {
			exec("ps -eo%mem,rss,pid | grep $pid", $output);
			$output = explode("  ", $output[0]);
			return isset($output[1]) ? $output[1] * 1024 : 0;
		}
	}

	/**
	 * @param string $msg
	 * @param const  $logType
	 * @return string
	 */
	private function _build($msg, $level, $type, $timer = 0, $mem = 0) {
		$msg = stripslashes(str_replace("<br/>", "\r\n", trim($msg)));
		switch ($level) {
			case self::LEVEL_INFO:
				$msg .= "\t(" . $type . ")";
				$result = $this->_buildInfo($msg);
				break;
			case self::LEVEL_ERROR:
				$msg .= "\t(" . $type . ")";
				$result = $this->_buildError($msg);
				break;
			case self::LEVEL_DEBUG:
				$msg .= "\t(" . $type . " timer: " . sprintf('%0.5f', ($timer - DEBUG_TIME)) . ")\r\n";
				$result = $this->_buildDebug($msg);
				break;
			case self::LEVEL_TRACE:
				$msg .= "\t(" . $type . ")";
				$result = $this->_buildTrace($msg);
				break;
			case self::LEVEL_PROFILE:
				$result = $this->_buildProfile($msg, $type, $timer, $mem);
				break;
			default:
				break;
		}
		return $result ? '[' . date('Y-m-d H:i:s') . '] ' . $result . "\r\n" : '';
	}

	/**
	 * @param $msg
	 * @param $type
	 * @param $timer
	 * @param $mem
	 */
	private function _buildProfile($msg, $type, $timer, $mem) {
		$_msg = '';
		if (strncasecmp($msg, self::TOKEN_BEGIN, strlen(self::TOKEN_BEGIN)) == 0) {
			$_token = substr($msg, strlen(self::TOKEN_BEGIN));
			$_token = substr($_token, 0, strpos($_token, ':'));
			$this->_profiles[] = array($_token, substr($msg, strpos($msg, ':', strlen(self::TOKEN_BEGIN)) + 1), $type, $timer, $mem);
		} elseif (strncasecmp(self::TOKEN_END, $msg, strlen(self::TOKEN_END)) == 0) {
			$_msg = "PROFILE! Message: \r\n";
			$_token = substr($msg, strlen(self::TOKEN_END));
			$_token = substr($_token, 0, strpos($_token, ':'));
			foreach ($this->_profiles as $key => $profile) {
				if ($profile[0] !== $_token) continue;
				if ($profile[1])
					$_msg .= $profile[1] . "\r\n";
				else
					$_msg .= substr($msg, strpos($msg, ':', strlen(self::TOKEN_END)) + 1) . "\r\n";
				$_msg .= "(type: $profile[2] time: " . ($timer - $profile[3]) . " mem: " . ($mem - $profile[4]) . ")";
				break;
			}
			unset($this->_profiles[$key]);
		}
		return $_msg;
	}

	/**
	 * <code>
	 * [2011-01-24 10:00:00] INFO! Message: $msg
	 * </code>
	 * @param string $msg
	 * @return string
	 */
	private function _buildInfo($msg) {
		return "INFO! Message:  " . $msg;
	}

	/**
	 * <code>
	 * [2011-01-24 10:00:00] TRACE! Message: $msg
	 * #1 trace1
	 * #2 trace2
	 * </code>
	 * @param string $msg
	 * @return string
	 */
	private function _buildTrace($msg) {
		return "TRACE! Message:  " . $msg . implode("\r\n", $this->_getTrace());
	}

	/**
	 * <code>
	 * [2011-01-24 10:00:00] DEBUG! Message: $msg
	 * #1 trace1
	 * #2 trace2
	 * </code>
	 * @param string $msg
	 * @return string
	 */
	private function _buildDebug($msg) {
		return 'DEBUG! Message:  ' . $msg . implode("\r\n", $this->_getTrace());
	}

	/**
	 * <code>
	 * [2011-01-24 10:00:00] ERROR! Message: $msg
	 * #1 trace1
	 * #2 trace2
	 * </code>
	 * @param string $msg
	 * @return string
	 */
	private function _buildError($msg) {
		return 'ERROR! Message:  ' . $msg;
	}

	/**
	 * <code>
	 * #1 trace
	 * #2 trace
	 * </code>
	 * @return string
	 */
	private function _getTrace() {
		$num = 0;
		$info[] = 'Stack trace:';
		$traces = debug_backtrace(false);
		foreach ($traces as $traceKey => $trace) {
			if ($num >= 7) break;
			if ((isset($trace['class']) && $trace['class'] == __CLASS__) || isset($trace['file']) && strrpos($trace['file'], __CLASS__ . '.php') !== false) continue;
			$file = isset($trace['file']) ? $trace['file'] . '(' . $trace['line'] . '): ' : '[internal function]: ';
			$function = isset($trace['class']) ? $trace['class'] . $trace['type'] . $trace['function'] : $trace['function'];
			if ($function == 'WindBase::log') continue;
			$args = array_map(array($this, '_buildArg'), $trace['args']);
			$info[] = '#' . ($num++) . ' ' . $file . $function . '(' . implode(',', $args) . ')';
		}
		return $info;
	}

	/**
	 * @param mixed $arg
	 */
	private function _buildArg($arg) {
		switch (gettype($arg)) {
			case 'array':
				return 'Array';
				break;
			case 'object':
				return 'Object ' . get_class($arg);
				break;
			default:
				return "'" . $arg . "'";
				break;
		}
	}

	/**
	 * @return string 
	 */
	private function _getFileName() {
		$_maxsize = ($this->_maxFileSize ? $this->_maxFileSize : 100) * 1024;
		$_logfile = $this->_logFile;
		if (is_file($_logfile) && $_maxsize <= filesize($_logfile)) {
			do {
				$counter = $counter ? ($counter + 1) : 1;
				$_newFile = $_logfile . '_' . date("Y_m_d_{$counter}");
			} while (is_file($_newFile));
			@rename($_logfile, $_newFile);
		}
		return $_logfile;
	}

	public function __destruct() {
		$this->flush();
	}

	/**
	 * @param field_type $_logFile
	 */
	public function setLogFile($logFile) {
		$this->_logFile = $logFile;
	}

	/**
	 * @param field_type $_maxFileSize
	 */
	public function setMaxFileSize($maxFileSize) {
		$this->_maxFileSize = (int) $maxFileSize;
	}
}

	