"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.stringify = exports.parse = exports.decode = exports.encode = void 0;
const iconv_lite_1 = __importDefault(require("iconv-lite"));
function isUTF8(charset) {
    if (!charset) {
        return true;
    }
    charset = charset.toLowerCase();
    return charset === 'utf8' || charset === 'utf-8';
}
function encode(str, charset) {
    if (!charset || isUTF8(charset)) {
        return encodeURIComponent(str);
    }
    const buf = iconv_lite_1.default.encode(str, charset);
    let encodeStr = '';
    let ch = '';
    for (let i = 0; i < buf.length; i++) {
        ch = buf[i].toString(16);
        if (ch.length === 1) {
            ch = '0' + ch;
        }
        encodeStr += '%' + ch;
    }
    encodeStr = encodeStr.toUpperCase();
    return encodeStr;
}
exports.encode = encode;
exports.default = encode;
function decode(str, charset) {
    if (!charset || isUTF8(charset)) {
        return decodeURIComponent(str);
    }
    const bytes = [];
    for (let i = 0; i < str.length;) {
        if (str[i] === '%') {
            i++;
            bytes.push(parseInt(str.substring(i, i + 2), 16));
            i += 2;
        }
        else {
            bytes.push(str.charCodeAt(i));
            i++;
        }
    }
    const buf = Buffer.from(bytes);
    return iconv_lite_1.default.decode(buf, charset);
}
exports.decode = decode;
function parse(qs, sepOrOptions, eq, options) {
    let sep;
    if (typeof sepOrOptions === 'object') {
        // parse(qs, options)
        options = sepOrOptions;
    }
    else {
        // parse(qs, sep, eq, options)
        sep = sepOrOptions;
    }
    sep = sep || '&';
    eq = eq || '=';
    const obj = {};
    if (typeof qs !== 'string' || qs.length === 0) {
        return obj;
    }
    const regexp = /\+/g;
    const splits = qs.split(sep);
    let maxKeys = 1000;
    let charset = '';
    if (options) {
        if (typeof options.maxKeys === 'number') {
            maxKeys = options.maxKeys;
        }
        if (typeof options.charset === 'string') {
            charset = options.charset;
        }
    }
    let len = splits.length;
    // maxKeys <= 0 means that we should not limit keys count
    if (maxKeys > 0 && len > maxKeys) {
        len = maxKeys;
    }
    for (let i = 0; i < len; ++i) {
        const x = splits[i].replace(regexp, '%20');
        const idx = x.indexOf(eq);
        let keyString;
        let valueString;
        let k;
        let v;
        if (idx >= 0) {
            keyString = x.substring(0, idx);
            valueString = x.substring(idx + 1);
        }
        else {
            keyString = x;
            valueString = '';
        }
        if (keyString && keyString.includes('%')) {
            try {
                k = decode(keyString, charset);
            }
            catch (e) {
                k = keyString;
            }
        }
        else {
            k = keyString;
        }
        if (valueString && valueString.includes('%')) {
            try {
                v = decode(valueString, charset);
            }
            catch (e) {
                v = valueString;
            }
        }
        else {
            v = valueString;
        }
        if (!has(obj, k)) {
            obj[k] = v;
        }
        else if (Array.isArray(obj[k])) {
            obj[k].push(v);
        }
        else {
            obj[k] = [obj[k], v];
        }
    }
    return obj;
}
exports.parse = parse;
function has(obj, prop) {
    return Object.prototype.hasOwnProperty.call(obj, prop);
}
function isASCII(str) {
    // eslint-disable-next-line no-control-regex
    return /^[\x00-\x7F]*$/.test(str);
}
function encodeComponent(item, charset) {
    item = String(item);
    if (isASCII(item)) {
        item = encodeURIComponent(item);
    }
    else {
        item = encode(item, charset);
    }
    return item;
}
function stringifyArray(values, prefix, options) {
    const items = [];
    for (const [index, value] of values.entries()) {
        items.push(stringify(value, `${prefix}[${index}]`, options));
    }
    return items.join('&');
}
function stringifyObject(obj, prefix, options) {
    const items = [];
    const charset = options.charset;
    for (const key in obj) {
        if (key === '') {
            continue;
        }
        const value = obj[key];
        if (value === null || value === undefined) {
            items.push(encode(key, charset) + '=');
        }
        else {
            const keyPrefix = prefix ? prefix + '[' + encodeComponent(key, charset) + ']' : encodeComponent(key, charset);
            items.push(stringify(value, keyPrefix, options));
        }
    }
    return items.join('&');
}
function stringify(obj, prefixOrOptions, options) {
    let prefix;
    if (typeof prefixOrOptions !== 'string') {
        options = prefixOrOptions || {};
    }
    else {
        prefix = prefixOrOptions;
    }
    options = options ?? {};
    if (Array.isArray(obj)) {
        if (!prefix) {
            throw new TypeError('stringify expects an object');
        }
        return stringifyArray(obj, prefix, options);
    }
    const objValue = String(obj);
    if (obj && typeof obj === 'object' && objValue === '[object Object]') {
        return stringifyObject(obj, prefix ?? '', options);
    }
    if (!prefix) {
        throw new TypeError('stringify expects an object');
    }
    const charset = options?.charset ?? 'utf-8';
    return `${prefix}=${encodeComponent(objValue, charset)}`;
}
exports.stringify = stringify;
//# sourceMappingURL=data:application/json;base64,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