import iconv from 'iconv-lite';
function isUTF8(charset) {
    if (!charset) {
        return true;
    }
    charset = charset.toLowerCase();
    return charset === 'utf8' || charset === 'utf-8';
}
export function encode(str, charset) {
    if (!charset || isUTF8(charset)) {
        return encodeURIComponent(str);
    }
    const buf = iconv.encode(str, charset);
    let encodeStr = '';
    let ch = '';
    for (let i = 0; i < buf.length; i++) {
        ch = buf[i].toString(16);
        if (ch.length === 1) {
            ch = '0' + ch;
        }
        encodeStr += '%' + ch;
    }
    encodeStr = encodeStr.toUpperCase();
    return encodeStr;
}
export default encode;
export function decode(str, charset) {
    if (!charset || isUTF8(charset)) {
        return decodeURIComponent(str);
    }
    const bytes = [];
    for (let i = 0; i < str.length;) {
        if (str[i] === '%') {
            i++;
            bytes.push(parseInt(str.substring(i, i + 2), 16));
            i += 2;
        }
        else {
            bytes.push(str.charCodeAt(i));
            i++;
        }
    }
    const buf = Buffer.from(bytes);
    return iconv.decode(buf, charset);
}
export function parse(qs, sepOrOptions, eq, options) {
    let sep;
    if (typeof sepOrOptions === 'object') {
        // parse(qs, options)
        options = sepOrOptions;
    }
    else {
        // parse(qs, sep, eq, options)
        sep = sepOrOptions;
    }
    sep = sep || '&';
    eq = eq || '=';
    const obj = {};
    if (typeof qs !== 'string' || qs.length === 0) {
        return obj;
    }
    const regexp = /\+/g;
    const splits = qs.split(sep);
    let maxKeys = 1000;
    let charset = '';
    if (options) {
        if (typeof options.maxKeys === 'number') {
            maxKeys = options.maxKeys;
        }
        if (typeof options.charset === 'string') {
            charset = options.charset;
        }
    }
    let len = splits.length;
    // maxKeys <= 0 means that we should not limit keys count
    if (maxKeys > 0 && len > maxKeys) {
        len = maxKeys;
    }
    for (let i = 0; i < len; ++i) {
        const x = splits[i].replace(regexp, '%20');
        const idx = x.indexOf(eq);
        let keyString;
        let valueString;
        let k;
        let v;
        if (idx >= 0) {
            keyString = x.substring(0, idx);
            valueString = x.substring(idx + 1);
        }
        else {
            keyString = x;
            valueString = '';
        }
        if (keyString && keyString.includes('%')) {
            try {
                k = decode(keyString, charset);
            }
            catch (e) {
                k = keyString;
            }
        }
        else {
            k = keyString;
        }
        if (valueString && valueString.includes('%')) {
            try {
                v = decode(valueString, charset);
            }
            catch (e) {
                v = valueString;
            }
        }
        else {
            v = valueString;
        }
        if (!has(obj, k)) {
            obj[k] = v;
        }
        else if (Array.isArray(obj[k])) {
            obj[k].push(v);
        }
        else {
            obj[k] = [obj[k], v];
        }
    }
    return obj;
}
function has(obj, prop) {
    return Object.prototype.hasOwnProperty.call(obj, prop);
}
function isASCII(str) {
    // eslint-disable-next-line no-control-regex
    return /^[\x00-\x7F]*$/.test(str);
}
function encodeComponent(item, charset) {
    item = String(item);
    if (isASCII(item)) {
        item = encodeURIComponent(item);
    }
    else {
        item = encode(item, charset);
    }
    return item;
}
function stringifyArray(values, prefix, options) {
    const items = [];
    for (const [index, value] of values.entries()) {
        items.push(stringify(value, `${prefix}[${index}]`, options));
    }
    return items.join('&');
}
function stringifyObject(obj, prefix, options) {
    const items = [];
    const charset = options.charset;
    for (const key in obj) {
        if (key === '') {
            continue;
        }
        const value = obj[key];
        if (value === null || value === undefined) {
            items.push(encode(key, charset) + '=');
        }
        else {
            const keyPrefix = prefix ? prefix + '[' + encodeComponent(key, charset) + ']' : encodeComponent(key, charset);
            items.push(stringify(value, keyPrefix, options));
        }
    }
    return items.join('&');
}
export function stringify(obj, prefixOrOptions, options) {
    let prefix;
    if (typeof prefixOrOptions !== 'string') {
        options = prefixOrOptions || {};
    }
    else {
        prefix = prefixOrOptions;
    }
    options = options ?? {};
    if (Array.isArray(obj)) {
        if (!prefix) {
            throw new TypeError('stringify expects an object');
        }
        return stringifyArray(obj, prefix, options);
    }
    const objValue = String(obj);
    if (obj && typeof obj === 'object' && objValue === '[object Object]') {
        return stringifyObject(obj, prefix ?? '', options);
    }
    if (!prefix) {
        throw new TypeError('stringify expects an object');
    }
    const charset = options?.charset ?? 'utf-8';
    return `${prefix}=${encodeComponent(objValue, charset)}`;
}
//# sourceMappingURL=data:application/json;base64,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