﻿using System;
using System.Collections.Generic;
using UnityEngine;
using Verse;
using Verse.AI;
using Verse.Sound;
using RimWorld;


namespace RecreationalBallpit
{

	public class RecBallpitSettings : ModSettings
	{
		public static int ballAmount = 200;
		public static string ballAmountEditBuffer = "200";
		public static bool ballsSpeedUpWithTime = true;

		public override void ExposeData()
		{
			Scribe_Values.Look(ref ballAmount, "ballAmount", 200);
			Scribe_Values.Look(ref ballsSpeedUpWithTime, "ballsSpeedUpWithTime", true);
			ballAmountEditBuffer = ballAmount.ToString();
			base.ExposeData();
		}
	}

	[StaticConstructorOnStartup]
	public class RecreationalBallpit : Mod
	{
		RecBallpitSettings settings;
		public static ThingDef[] ballList;
		public static Thing[] ballDummies = new Thing[11];
		public static Mesh[] ballMeshes = new Mesh[11];
		public static Material[] ballMaterials = new Material[11];
		public static int[] preBakedBallOnBallInteractionCPURequirementScores = new int[BallpitMotionComp.hardCap + 1];
		private static bool initialised = false;
		public static int cpuReqScore = 0;
		
		
		public RecreationalBallpit(ModContentPack content) : base(content)
		{
			settings = GetSettings<RecBallpitSettings>();
			BallpitMotionComp.preBakedAppropriateTimeToSpeedScales[0] = 1f;
			BallpitMotionComp.preBakedAppropriateTimeToWindReistanceScales[0] = BallpitMotionComp.windResistance;
			BallpitMotionComp.preBakedAppropriateTimeToCollisionForceScales[0] = 1.5f;
			for (int i = 1, prev = 0; i < 25; i++, prev++)
			{
				if (i < 7)
				{
					BallpitMotionComp.preBakedAppropriateTimeToCollisionForceScales[i] = BallpitMotionComp.preBakedAppropriateTimeToCollisionForceScales[prev] + 0.3f;
					BallpitMotionComp.preBakedAppropriateTimeToSpeedScales[i] = BallpitMotionComp.preBakedAppropriateTimeToSpeedScales[prev] + 0.3f;
					BallpitMotionComp.preBakedAppropriateTimeToWindReistanceScales[i] = BallpitMotionComp.preBakedAppropriateTimeToWindReistanceScales[prev] * BallpitMotionComp.windResistance;
				}
				else
				{
					BallpitMotionComp.preBakedAppropriateTimeToCollisionForceScales[i] = BallpitMotionComp.preBakedAppropriateTimeToCollisionForceScales[prev] + 0.1f;
					BallpitMotionComp.preBakedAppropriateTimeToSpeedScales[i] = BallpitMotionComp.preBakedAppropriateTimeToSpeedScales[prev] + 0.1f;
					BallpitMotionComp.preBakedAppropriateTimeToWindReistanceScales[i] = BallpitMotionComp.preBakedAppropriateTimeToWindReistanceScales[prev] * BallpitMotionComp.windResistance;
				}
			}
			for (int preBakeIter = 0; preBakeIter < BallpitMotionComp.hardCap + 1; preBakeIter++)
            {
				for (int a = 0; a < preBakeIter-1; a++)
				{
					int b = a + 1;
					preBakedBallOnBallInteractionCPURequirementScores[preBakeIter] += preBakeIter - b;
				}
            }
		}
		public static string GetCPURequirementScore(int ballAmount)
        {
			ballAmount = Math.Min(ballAmount, BallpitMotionComp.hardCap);
			cpuReqScore = 0;
			cpuReqScore += preBakedBallOnBallInteractionCPURequirementScores[ballAmount];
			string cpuReqStr = cpuReqScore.ToString();
			string result = "";
			for(int i = cpuReqStr.Length; i >= 0; i -= 3)
            {
				if (i < 3)
                {
					result = cpuReqStr.Substring(0, i) + result;
                }
				else
				{
					result = cpuReqStr.Substring(i - 3, 3) + result;
					if (i > 3)
					{
						result = "," + result;
					}
				}
            }
			return result;
        }

		public static void InitialiseBalls()
        {
			if (!initialised)
			{
				initialised = true;
				// For some reason, this cannot be done in the class constructor due to MyBalls not being instantiated yet.
				// This problem only occurs if the class is inheriting the Mod class (RecreationalBallpit : Mod).
				// So far I have only had to inherit Mod in order to register a mod settings menu.
				ballList = new ThingDef[] {
					MyBalls.Mote_BallBlue,
					MyBalls.Mote_BallCyan,
					MyBalls.Mote_BallDark,
					MyBalls.Mote_BallGreen,
					MyBalls.Mote_BallMagenta,
					MyBalls.Mote_BallOrange,
					MyBalls.Mote_BallPink,
					MyBalls.Mote_BallPurple,
					MyBalls.Mote_BallRed,
					MyBalls.Mote_BallWhite,
					MyBalls.Mote_BallYellow
				};
				for (int t = 0; t < 11; t++)
				{
					ballDummies[t] = ThingMaker.MakeThing(BallFromNumber(t));

					// If anything, this is the culprit code causing the blue balls to render on top of the rest.
					// Blue is the first element in the ball list as you can see above.
					ballMaterials[t] = ballDummies[t].Graphic.MatSingle;
					ballMeshes[t] = ballDummies[t].Graphic.MeshAt(Rot4.North);
				}
			}
        }
		public static ThingDef BallFromNumber(int num)
		{
			return ballList[num % 11];
		}
		public override void DoSettingsWindowContents(Rect inRect)
		{
			Listing_Standard listingStandard = new Listing_Standard();
			listingStandard.Begin(inRect);
			string ballAmountLabel = "Amount of balls per ball pit. Higher amounts will impact performance. More ball pits in the world will also add to this. ";
			string ballAmountTooltip = "This value effectively caps out at 2,000 but I wouldn't recommend setting this over a few hundred.";
			listingStandard.Label(ballAmountLabel + ballAmountTooltip);
			listingStandard.IntEntry(ref RecBallpitSettings.ballAmount, ref RecBallpitSettings.ballAmountEditBuffer);
			listingStandard.Label("");
			string ballsSpeedUpToolTip = "Makes balls travel faster with higher game speed. Balls still stop while paused regardless. This should have little impact on performance toggled on or off.";
			listingStandard.CheckboxLabeled("Balls speed up with time", ref RecBallpitSettings.ballsSpeedUpWithTime, ballsSpeedUpToolTip);
			listingStandard.Label("");
			listingStandard.Label("");
			listingStandard.Label("");
			listingStandard.Label("CPU requirement score for 1 ballpit: " + GetCPURequirementScore(RecBallpitSettings.ballAmount));
			listingStandard.Label("You can use this score to determine how much your computer can process and how high you should have the ball amount set. The score won't mean much until you notice the difference in performance between different settings and correlate that with the CPU score.");
			listingStandard.Label("*This value only reflects the compounding effect of adding more balls to the pit. (1 more ball = 1 more calculation required per ball before that)");
			listingStandard.End();
			base.DoSettingsWindowContents(inRect);
		}
		public override string SettingsCategory()
		{
			return "RecreationalBallpit";
		}
	}

	public class BallpitMotionComp : ThingComp
	{
		public BallpitDynamicBallProperties[] ballProperties;
		public int maxBalls = 200;
		public int amountInstantiated = 0;
		public static int hardCap = 2000;
		public static float ballRadius = 0.085f;
		public float separationDistance;
		private IntVec3 middle;
		private bool skippedAFrame = false;
		private bool skippedAFrameAndBallsSpeedUpWithTime = true;
		private int tickIteratingNow = 1;
		public static float windResistance = 0.985f;
		public float topBoundary;
		public float bottomBoundary;
		public float leftBoundary;
		public float rightBoundary;
		private float mouseInBallpitSize = 0.3f;
		public float pawnInBallpitSize = 0.4f;
		public float tempSharedVarX = 0f;
		public float tempSharedVarY = 0f;
		private float topBoundaryForMouse;
		private float bottomBoundaryForMouse;
		private float leftBoundaryForMouse;
		private float rightBoundaryForMouse;
		private float topBoundaryForPawns;
		private float bottomBoundaryForPawns;
		private float leftBoundaryForPawns;
		private float rightBoundaryForPawns;
		public bool justRespawned = false;
		public static float[] preBakedAppropriateTimeToSpeedScales = new float[25];
		public static float[] preBakedAppropriateTimeToWindReistanceScales = new float[25];
		public static float[] preBakedAppropriateTimeToCollisionForceScales = new float[25];
		public override void PostSpawnSetup(bool respawningAfterLoad)
		{
			base.PostSpawnSetup(respawningAfterLoad);
			justRespawned = respawningAfterLoad;
			amountInstantiated = 0;
			ballProperties = new BallpitDynamicBallProperties[hardCap];
			separationDistance = (ballRadius * 2) + 0.0000005f;
			CellRect occupied = parent.OccupiedRect();
			middle = occupied.CenterCell;
			topBoundary = (middle.z + 2f) - 0.283333f;
			bottomBoundary = (middle.z - 1f) + 0.566667f;
			leftBoundary = (middle.x - 1f) + 0.216667f;
			rightBoundary = (middle.x + 2f) - 0.216667f;
			topBoundaryForMouse = topBoundary + mouseInBallpitSize;
			bottomBoundaryForMouse = bottomBoundary - mouseInBallpitSize;
			rightBoundaryForMouse = rightBoundary + mouseInBallpitSize;
			leftBoundaryForMouse = leftBoundary - mouseInBallpitSize;
			topBoundaryForPawns = topBoundary + pawnInBallpitSize;
			bottomBoundaryForPawns = bottomBoundary - pawnInBallpitSize;
			rightBoundaryForPawns = rightBoundary + pawnInBallpitSize;
			leftBoundaryForPawns = leftBoundary - pawnInBallpitSize;
			mouseInBallpitSize += ballRadius;
			SetNewMaxBalls(RecBallpitSettings.ballAmount);
		}
		public void SetNewMaxBalls(int amount)
		{
			amount = Math.Max(0, Math.Min(amount, hardCap));
			while (amount > amountInstantiated)
				ballProperties[amountInstantiated++] = new BallpitDynamicBallProperties(this);
			maxBalls = amount;
			if (RecBallpitSettings.ballAmount != amount)
			{
				// This effectively auto-clamps the settings menu's int input box with a hard min and max (0 and hardCap)
				RecBallpitSettings.ballAmountEditBuffer = amount.ToString();
				RecBallpitSettings.ballAmount = amount;
			}
        }
		public void Collide(BallpitDynamicBallProperties ballA, BallpitDynamicBallProperties ballB, float detectionBounds, float dampScale=200f)
        {
			float distance = FindDistance(tempSharedVarX, tempSharedVarY);
			if (distance < detectionBounds)
			{
				float separationPower = (detectionBounds - distance) / detectionBounds / dampScale;
                if (skippedAFrameAndBallsSpeedUpWithTime)
                    separationPower *= preBakedAppropriateTimeToCollisionForceScales[tickIteratingNow];
                float contactAngle = Mathf.Atan2(ballB.position.z - ballA.position.z, ballB.position.x - ballA.position.x);
				float xVariate = separationPower * Mathf.Cos(contactAngle);
				float yVariate = separationPower * Mathf.Sin(contactAngle);
				ballA.xVel -= xVariate;
				ballA.yVel -= yVariate;
				ballB.xVel += xVariate;
				ballB.yVel += yVariate;
			}
		}
		public void Collide(BallpitDynamicBallProperties ball, Vector3 location, float detectionBounds, float dampScale = 10f)
		{
			float distance = FindDistance(tempSharedVarX, tempSharedVarY);
			if (distance < detectionBounds)
			{
				float separationPower = (detectionBounds - distance) / detectionBounds / dampScale;
                if (skippedAFrameAndBallsSpeedUpWithTime)
                    separationPower *= preBakedAppropriateTimeToCollisionForceScales[tickIteratingNow];
                float contactAngle = Mathf.Atan2(location.z - ball.position.z, location.x - ball.position.x);
				ball.xVel -= separationPower * Mathf.Cos(contactAngle);
				ball.yVel -= separationPower * Mathf.Sin(contactAngle);
			}
		}
		public override void CompTick()
		{
			base.CompTick();
			if (parent.Spawned)
			{
				if (Find.TickManager.TickRateMultiplier > tickIteratingNow)
				{
					tickIteratingNow += 1;
					skippedAFrame = true;
				}
				else
				{
					if (tickIteratingNow > 24)
						tickIteratingNow = 24;
					// Very misleading having this here I know, but having it in the MotionComp's PostSpawnSetup causes the game to crash
					//   if you are loading into a world that has a ballpit already placed and I have no idea why.
					RecreationalBallpit.InitialiseBalls();
					if (maxBalls != RecBallpitSettings.ballAmount)
                    {
						SetNewMaxBalls(RecBallpitSettings.ballAmount);
                    }
					skippedAFrameAndBallsSpeedUpWithTime = skippedAFrame & RecBallpitSettings.ballsSpeedUpWithTime;
					Vector3 mousePos = UI.MouseMapPosition();
					bool collideMouse = false;
					if (mousePos.x > leftBoundaryForMouse)
						if (mousePos.x < rightBoundaryForMouse)
							if (mousePos.z < topBoundaryForMouse)
								if (mousePos.z > bottomBoundaryForMouse)
									collideMouse = true;

					for (int i = 0; i < maxBalls; i++)
					{
						BallpitDynamicBallProperties ballA = ballProperties[i];
						for (int j = i + 1; j < maxBalls; j++)
						{
							BallpitDynamicBallProperties ballB = ballProperties[j];
							tempSharedVarX = Mathf.Abs(ballA.position.x - ballB.position.x);
							if (tempSharedVarX < separationDistance)
							{
								tempSharedVarY = Mathf.Abs(ballA.position.z - ballB.position.z);
								if (tempSharedVarY < separationDistance)
								{
									Collide(ballA, ballB, separationDistance, 180f);
								}
							}
						}
						if (collideMouse)
						{
							tempSharedVarX = Mathf.Abs(mousePos.x - ballA.position.x);
							if (tempSharedVarX < mouseInBallpitSize)
							{
								tempSharedVarY = Mathf.Abs(mousePos.z - ballA.position.z);
								if (tempSharedVarY < mouseInBallpitSize)
								{
									Collide(ballA, mousePos, mouseInBallpitSize, 50f);
								}
							}
						}
					}
					foreach (Pawn pawn in parent.Map.mapPawns.AllPawns)
					{
						if (pawn == null)
						{
							continue;
						}
						if (pawn.DrawPos == null)
						{
							continue;
						}
						if (!pawn.Spawned)
                        {
							continue;
                        }
						if (pawn.DrawPos.x > leftBoundaryForPawns)
							if (pawn.DrawPos.x < rightBoundaryForPawns)
								if (pawn.DrawPos.z > bottomBoundaryForPawns)
									if (pawn.DrawPos.z < topBoundaryForPawns)
									{
										for (int i = 0; i < maxBalls; i++)
										{
											BallpitDynamicBallProperties ball = ballProperties[i];
											tempSharedVarX = Mathf.Abs(pawn.DrawPos.x - ball.position.x);
											if (tempSharedVarX < pawnInBallpitSize)
											{
												tempSharedVarY = Mathf.Abs(pawn.DrawPos.z - ball.position.z);
												if (tempSharedVarY < pawnInBallpitSize)
												{
													Collide(ball, pawn.DrawPos, pawnInBallpitSize, 130f);
												}
											}
										}
									}
					}
					// Seems more optimal to me other wise with 1 for loop I end up with a bunch of checks for skippedAFrameAndBallsSpeedUpWithTime per iteration
					if (skippedAFrameAndBallsSpeedUpWithTime)
					{
						for (int i = 0; i < maxBalls; i++)
						{
							BallpitDynamicBallProperties ball = ballProperties[i];
							if (ball.position.z < bottomBoundary)
							{
								ball.yVel = Mathf.Abs(ball.yVel);
							}
							else if (ball.position.z > topBoundary)
							{
								ball.yVel = -Mathf.Abs(ball.yVel);
							}
							if (ball.position.x < leftBoundary)
							{
								ball.xVel = Mathf.Abs(ball.xVel);
							}
							else if (ball.position.x > rightBoundary)
							{
								ball.xVel = -Mathf.Abs(ball.xVel);
							}
							ball.xVel *= preBakedAppropriateTimeToWindReistanceScales[tickIteratingNow];
							ball.yVel *= preBakedAppropriateTimeToWindReistanceScales[tickIteratingNow];
							ball.position.x += ball.xVel * preBakedAppropriateTimeToSpeedScales[tickIteratingNow];
							ball.position.z += ball.yVel * preBakedAppropriateTimeToSpeedScales[tickIteratingNow];
						}
					}
					else
					{
						for (int i = 0; i < maxBalls; i++)
						{
							BallpitDynamicBallProperties ball = ballProperties[i];
							bool inWall = false;
							if (ball.position.z < bottomBoundary)
							{
								ball.yVel = Mathf.Abs(ball.yVel);
								inWall = true;
							}
							else if (ball.position.z > topBoundary)
							{
								ball.yVel = -Mathf.Abs(ball.yVel);
								inWall = true;
							}
							if (ball.position.x < leftBoundary)
							{
								ball.xVel = Mathf.Abs(ball.xVel);
								inWall = true;
							}
							else if (ball.position.x > rightBoundary)
							{
								ball.xVel = -Mathf.Abs(ball.xVel);
								inWall = true;
							}
							if (!inWall)
							{
								ball.xVel *= windResistance;
								ball.yVel *= windResistance;
							}
							else
							{
								ball.xVel *= 0.75f;
								ball.yVel *= 0.75f;
							}
							ball.position.x += ball.xVel;
							ball.position.z += ball.yVel;
						}
					}
					tickIteratingNow = 1;
					skippedAFrame = false;
				}
			}
		}
        public override void PostDeSpawn(Map map)
        {
            base.PostDeSpawn(map);

        }
        public override void PostDraw()
		{
			base.PostDraw();
            Vector3 drawSize = new Vector3(MyBalls.Mote_BallBlue.graphic.drawSize.x, 1f, MyBalls.Mote_BallBlue.graphic.drawSize.y);
            for (int i = 0; i < maxBalls; i++)
            {
                int wrapI = i % 11;
                Matrix4x4 matrix = default(Matrix4x4);
                matrix.SetTRS(ballProperties[i].position, Rot4.North.AsQuat, Vector3.one);
                Graphics.DrawMesh(RecreationalBallpit.ballMeshes[wrapI], matrix, RecreationalBallpit.ballMaterials[wrapI], 0);

                // RimWorld v1.2 rendering method
                //Graphics.DrawMesh(RecreationalBallpit.ballMeshes[wrapI], ballProperties[i].position, Rot4.North.AsQuat, RecreationalBallpit.ballMaterials[wrapI], 0);
            }
			justRespawned = false;
        }
		public float FindDistance(float xx, float yy)
        {
			return Mathf.Sqrt(Mathf.Pow(xx, 2f) + Mathf.Pow(yy, 2f));
		}
		public float FindDistance(float x1, float y1, float x2, float y2)
        {
			return Mathf.Sqrt(Mathf.Pow(Mathf.Abs(x1 - x2), 2f) + Mathf.Pow(Mathf.Abs(y1 - y2), 2f));
        }
	}


    public class BallpitDynamicBallProperties
    {
		public Vector3 position;
        public float xVel;
        public float yVel;
		public BallpitDynamicBallProperties(BallpitMotionComp motionComp)
        {
			position.x = Rand.Range(motionComp.leftBoundary, motionComp.rightBoundary);
			position.y = 5.5f;
			position.z = Rand.Range(motionComp.bottomBoundary, motionComp.topBoundary);
			xVel = Rand.Range(-0.01f, 0.01f);
			yVel = Rand.Range(-0.01f, 0.01f);
		}
    }



	public class JoyGiver_PlayBallpit : JoyGiver_InteractBuilding
	{
		protected override bool CanDoDuringGathering
		{
			get
			{
				return true;
			}
		}

		protected override Job TryGivePlayJob(Pawn pawn, Thing t)
		{
			return JobMaker.MakeJob(this.def.jobDef, t);
		}

	}


	public class JobDriver_PlayBallpit : JobDriver
	{
		public override bool TryMakePreToilReservations(bool errorOnFailed)
		{
			return this.pawn.Reserve(this.job.targetA, this.job, this.job.def.joyMaxParticipants, 0, null, errorOnFailed);
		}

        protected override IEnumerable<Toil> MakeNewToils()
		{
			this.EndOnDespawnedOrNull(TargetIndex.A, JobCondition.Incompletable);
			CellRect cellChoices = this.TargetThingA.OccupiedRect();
			yield return Toils_Goto.GotoCell(cellChoices.RandomCell, PathEndMode.OnCell);
			Toil toil = new Toil();
			toil.initAction = delegate ()
			{
				this.job.locomotionUrgency = LocomotionUrgency.Walk;
			};
			toil.tickAction = delegate ()
			{

				// Once on the 25th tick and then once more every time 100 ticks elapse, the last of which will be 25 ticks before the end of the toil
				//   (ticksLeftThisToil starts at 350)
				if ((this.ticksLeftThisToil - 25) % 100 == 0)
				{
					BallpitMotionComp comp = TargetThingA.TryGetComp<BallpitMotionComp>();
					if (comp.justRespawned)
					{
						base.EndJobWith(JobCondition.Succeeded);
						return;
					}
					Vector3 actorPos = pawn.DrawPos;
					for (int i = 0; i < comp.maxBalls; i++)
					{
						comp.tempSharedVarX = Mathf.Abs(actorPos.x - comp.ballProperties[i].position.x);
						if (comp.tempSharedVarX < 1.2f)
						{
							comp.tempSharedVarY = Mathf.Abs(actorPos.z - comp.ballProperties[i].position.z);
							if (comp.tempSharedVarY < 1.2f)
							{
								comp.Collide(comp.ballProperties[i], actorPos, 1.2f, 15f);
							}
						}
					}
					SoundDefOf.PlayBilliards.PlayOneShot(new TargetInfo(this.pawn.Position, this.pawn.Map, false));
					int amt = Rand.Range(3, 7);
					int ballThrowMaxDist = 3;
					int downCap = this.pawn.Position.z - ballThrowMaxDist;
					int upCap = this.pawn.Position.z + ballThrowMaxDist + 1;
					int leftCap = this.pawn.Position.x - ballThrowMaxDist;
					int rightCap = this.pawn.Position.x + ballThrowMaxDist + 1;
					for (int i = 0; i < amt; i++)
					{
						int thisX = Rand.Range(leftCap, rightCap);
						int thisZ = Rand.Range(downCap, upCap);

						if (thisX == pawn.Position.x & thisZ == pawn.Position.z)
							continue;
						int thisBall = Rand.Range(0, 11);
						ThrowObjectAt(this.pawn, new IntVec3(thisX, this.pawn.Position.y, thisZ), RecreationalBallpit.BallFromNumber(thisBall));
					}
				}
				if (Find.TickManager.TicksGame > this.startTick + this.job.def.joyDuration)
				{
					base.EndJobWith(JobCondition.Succeeded);
					return;
				}
				JoyUtility.JoyTickCheckEnd(this.pawn, JoyTickFullJoyAction.EndJob, 1f, (Building)base.TargetThingA);
			};
			toil.socialMode = RandomSocialMode.SuperActive;
			toil.defaultCompleteMode = ToilCompleteMode.Delay;
			toil.defaultDuration = 350;
			toil.AddFinishAction(delegate
			{
				JoyUtility.TryGainRecRoomThought(this.pawn);
			});
			yield return toil;
			yield break;
		}

		private static void ThrowObjectAt(Pawn thrower, IntVec3 targetCell, ThingDef mote)
		{
			if (!thrower.Position.ShouldSpawnMotesAt(thrower.Map) || thrower.Map.moteCounter.Saturated)
			{
				return;
			}
			float num = Rand.Range(4f, 7.5f);
			Vector3 vector = targetCell.ToVector3Shifted();
			vector.y = thrower.DrawPos.y;
			MoteThrown moteThrown = (MoteThrown)ThingMaker.MakeThing(mote, null);
			moteThrown.Scale = 1f;
			moteThrown.rotationRate = (float)Rand.Range(-300, 300);
			moteThrown.exactPosition = thrower.DrawPos;
			moteThrown.SetVelocity((vector - moteThrown.exactPosition).AngleFlat(), num);
			moteThrown.airTimeLeft = (float)Mathf.RoundToInt((moteThrown.exactPosition - vector).MagnitudeHorizontal() / num);
			GenSpawn.Spawn(moteThrown, thrower.Position, thrower.Map, WipeMode.Vanish);
		}

		public override object[] TaleParameters()
		{
			return new object[]
			{
				this.pawn,
				base.TargetA.Thing.def
			};
		}

		private const int ShotDuration = 700;
	}


    [DefOf]
    public class MyBalls
	{
		static MyBalls()
		{
			
            DefOfHelper.EnsureInitializedInCtor(typeof(MyBalls));
        }

		public static ThingDef Mote_BallBlue;
		public static ThingDef Mote_BallCyan;
		public static ThingDef Mote_BallDark;
		public static ThingDef Mote_BallOrange;
		public static ThingDef Mote_BallGreen;
		public static ThingDef Mote_BallMagenta;
		public static ThingDef Mote_BallPink;
		public static ThingDef Mote_BallPurple;
		public static ThingDef Mote_BallRed;
		public static ThingDef Mote_BallWhite;
		public static ThingDef Mote_BallYellow;
        public static ThingDef BuildingRecBallpit;
    }
}

