/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;

/**
 * A little-endian <code>DataInput</code> that is wraps another with a bound.
 *
 * @author Mark Fairchild
 * @version 2016/09/07
 */
public class BoundedLittleEndianInput extends LittleEndianInput {

    /**
     * Create a new <code>BoundedLittleEndianInput</code> that wraps an
     * <code>InputStream</code>.
     * @param original The original <code>InputStream</code>.
     * @param bound The upper boundary on the number of bytes that can be
     * read.
     */
    public BoundedLittleEndianInput(InputStream original, int bound) {
        this.ORIGINAL = Objects.requireNonNull(original);
        this.BOUND = bound;
        this.position = 0;
    }

    /**
     * @see InputStream#available()
     * @return
     * @throws IOException
     */
    @Override
    public int available() throws IOException {
        int remaining = this.BOUND - this.position;
        assert remaining >= 0;
        
        //if (this.ORIGINAL.available() < remaining) {
        //    return this.ORIGINAL.available();
        //}
        
        return remaining;
    }

    /**
     * @see InputStream#read()
     * @return
     * @throws IOException
     */
    @Override
    public int read() throws IOException {
        if (this.available() <= 0) {
            return -1;
        }

        this.position++;
        return this.ORIGINAL.read();
    }

    /**
     * @see InputStream#read(byte[])
     * @param b
     * @return
     * @throws IOException
     */
    @Override
    public int read(byte[] b) throws IOException {
        return this.read(b, 0, b.length);
    }

    /**
     * @see InputStream#read(byte[], int, int)
     * @param b
     * @param off
     * @param len
     * @return
     * @throws IOException
     */
    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        if (this.available() <= 0) {
            return -1;
        } else if (this.available() < len) {
            int bytesRead = this.ORIGINAL.read(b, off, this.available());
            assert bytesRead == this.available() : "Expected " + len + " but read " + bytesRead;
            this.position += bytesRead;
            return bytesRead;
        } else {
            int bytesRead = this.ORIGINAL.read(b, off, len);
            assert bytesRead == len : "Expected " + len + " but read " + bytesRead;
            this.position += bytesRead;
            return bytesRead;
        }
    }

    @Override
    public int skipBytes(int n) throws IOException {
        int limit = Math.min(n, this.available());
        for (int i = 0; i < limit; i++) {
            this.read();
        }
        return limit;
    }

    @Override
    public String toString() {
        return String.format("Stream %d / %d (%s)", this.position, this.BOUND, this.ORIGINAL);
    }

    final private InputStream ORIGINAL;
    final int BOUND;
    int position;

}
