/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.io.Closeable;

/**
 * An <code>InputStream</code> that prints the contents of the file to a
 * <code>Writer</code> as it is read.
 *
 * @author Mark Fairchild
 * @version 2016/04/26
 */
public class DebuggingInputStream extends InputStream implements Closeable {

    /**
     * Creates a new <code>DebuggingInputStream</code> that wraps the specified
     * <code>InputStream</code> and outputs to the specified
     * <code>Writer</code>. The output will be formatted with 32 bytes per line,
     * written in hexadecimal.
     *
     * @param input The <code>InputStream</code> from which to read. Must be
     * non-null.
     * @param output The <code>Writer</code> that will receive the output. If it
     * is null then no ouput will be printed.
     */
    public DebuggingInputStream(InputStream input, PrintStream output) {
        this(input, output, 32);
    }

    /**
     * Creates a new <code>DebuggingInputStream</code> that wraps the specified
     * <code>InputStream</code> and outputs to the specified
     * <code>Writer</code>. The output will be formatted with a specified number
     * bytes per line, written in hexadecimal.
     *
     * @param input The <code>InputStream</code> from which to read. Must be
     * non-null.
     * @param output The <code>Writer</code> that will receive the output. If it
     * is null then no output will be printed.
     * @param width The number of byte per line to print.
     */
    public DebuggingInputStream(InputStream input, PrintStream output, int width) {
        if (null == input) {
            throw new NullPointerException("The inputstream must not be null.");
        }

        if (width < 1) {
            throw new IllegalArgumentException("The line width must be greater than zero.");
        }

        this.INPUT = input;
        this.OUTPUT = output;
        this.offset = 0;
        this.width = width;
        this.position = 0;
    }

    /**
     * @see InputStream#read()
     * @return The next byte in the stream.
     * @throws IOException
     */
    @Override
    public int read() throws IOException {
        int val = this.INPUT.read();

        if (null != this.OUTPUT) {
            if (this.offset == 0) {
                this.OUTPUT.printf("%08x:  ", this.position);
            }

            this.OUTPUT.printf("%02x ", val);
            this.position++;
            this.offset++;

            if (this.offset >= this.width) {
                this.offset = 0;
                this.OUTPUT.println();
            }
        }

        return val;
    }

    /**
     * @see Closeable#close()
     * @throws IOException
     */
    @Override
    public void close() throws IOException {
        this.INPUT.close();
    }

    /**
     * @see java.io.InputStream#available()
     * @return The number of bytes available to be read.
     * @throws IOException
     */
    @Override
    public int available() throws IOException {
        return this.INPUT.available();
    }

    final private InputStream INPUT;
    final private PrintStream OUTPUT;
    private int offset;
    private int width;
    private int position;
}
