/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.Closeable;
import java.io.DataInput;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;

/**
 * An abstract base class for little-endian <code>DataInput</code>
 * implementations.
 *
 * @author Mark Fairchild
 * @version 2016/06/25
 */
abstract public class LittleEndianInput extends InputStream implements DataInput, Closeable {

    /**
     * @see InputStream#available()
     * @return
     * @throws IOException
     */
    @Override
    abstract public int available() throws IOException;

    /**
     * @see InputStream#read()
     * @return
     * @throws IOException
     */
    @Override
    abstract public int read() throws IOException;

    /**
     * @see InputStream#read(byte[])
     * @param b
     * @return
     * @throws IOException
     */
    @Override
    abstract public int read(byte[] b) throws IOException;

    /**
     * @see InputStream#read(byte[], int, int)
     * @param b
     * @param off
     * @param len
     * @return
     * @throws IOException
     */
    @Override
    abstract public int read(byte[] b, int off, int len) throws IOException;

    /**
     * Creates a new <code>AbstractLittleEndian</code>.
     */
    public LittleEndianInput() {
        this.BUFFER = new byte[8];
    }

    /**
     * @see java.io.DataInput#read(byte[])
     * @param b
     * @throws IOException
     */
    @Override
    public void readFully(byte[] b) throws IOException {
        int bytesRead = this.read(b);
        if (bytesRead != b.length) {
            throw new IOException(String.format("Incorrect length; read %d bytes, expected %d.", bytesRead, b.length));
        }
    }

    /**
     * @see java.io.DataInput#read(byte[],int,int)
     * @param b
     * @param off
     * @param len
     * @throws IOException
     */
    @Override
    public void readFully(byte[] b, int off, int len) throws IOException {
        int bytesRead = this.read(b, off, len);
        if (bytesRead != len) {
            throw new IOException(String.format("Incorrect length; read %d bytes, expected %d.", bytesRead, len));
        }
    }

    /**
     * @see java.io.DataInput#readByte()
     * @return The value that was readFully.
     * @throws IOException
     */
    @Override
    public byte readByte() throws IOException {
        int val = this.read();
        byte val2 = (byte) val;
        return val2;
    }

    /**
     * @see java.io.DataInput#readUnsignedByte()
     * @return The value that was read.
     * @throws IOException
     */
    @Override
    public int readUnsignedByte() throws IOException {
        return 0xFF & this.read();
    }

    /**
     * @see java.io.DataInput#readShort()
     * @return The value that was read.
     * @throws IOException
     */
    @Override
    public short readShort() throws IOException {
        this.readFully(BUFFER, 0, 2);
        return (short) ((BUFFER[0] & 0xFF)
                | (BUFFER[1] & 0xFF) << 8);
    }

    /**
     * @see java.io.DataInput#readUnsignedShort()
     * @return The value that was readFully.
     * @throws IOException
     */
    @Override
    public int readUnsignedShort() throws IOException {
        this.readFully(BUFFER, 0, 2);
        return (BUFFER[0] & 0xFF)
                | (BUFFER[1] & 0xFF) << 8;
    }

    /**
     * @see java.io.DataInput#readInt()
     * @return The value that was readFully.
     * @throws IOException
     */
    @Override
    public int readInt() throws IOException {
        this.readFully(BUFFER, 0, 4);
        return (BUFFER[0] & 0xFF)
                | (BUFFER[1] & 0xFF) << 8
                | (BUFFER[2] & 0xFF) << 16
                | (BUFFER[3] & 0xFF) << 24;
    }

    /**
     * @see java.io.DataInput#readLong()
     * @return The value that was readFully.
     * @throws IOException
     */
    @Override
    public long readLong() throws IOException {
        this.readFully(BUFFER, 0, 8);
        long val = (BUFFER[0] & 0xFF)
                | (BUFFER[1] & 0xFFL) << 8
                | (BUFFER[2] & 0xFFL) << 16
                | (BUFFER[3] & 0xFFL) << 24
                | (BUFFER[4] & 0xFFL) << 32
                | (BUFFER[5] & 0xFFL) << 40
                | (BUFFER[6] & 0xFFL) << 48
                | (BUFFER[7] & 0xFFL) << 56;
        return val;
    }

    /**
     * @see java.io.DataInput#readBoolean()
     * @return The value that was readFully.
     * @throws IOException
     */
    @Override
    public boolean readBoolean() throws IOException {
        return this.read() != 0;
    }

    /**
     * @see java.io.DataInput#readChar()
     * @return The value that was readFully.
     * @throws IOException
     */
    @Override
    public char readChar() throws IOException {
        return (char) this.readShort();
    }

    /**
     * @see java.io.DataInput#readFloat()
     * @return The value that was readFully.
     * @throws IOException
     */
    @Override
    public float readFloat() throws IOException {
        int val = this.readInt();
        return Float.intBitsToFloat(val);
    }

    /**
     * @see java.io.DataInput#readDouble()
     * @return The value that was readFully.
     * @throws IOException
     */
    @Override
    public double readDouble() throws IOException {
        long val = this.readLong();
        return Double.longBitsToDouble(val);
    }

    /**
     * @see DataInput#readLine()
     * @return
     * @throws IOException
     */
    @Override
    public String readLine() throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     * Reads a UTF8 string.
     *
     * @return The string that was readFully.
     * @throws IOException
     */
    @Override
    public String readUTF() throws IOException {
        short length = this.readShort();
        byte[] bytes = new byte[length];
        int bytesRead = this.read(bytes);
        assert bytesRead == length;

        String str = new String(bytes, "UTF-8");
        return str;
    }

    /**
     * Reads a BString (1-byte length-prefixed).
     *
     * @return The string that was readFully.
     * @throws IOException
     */
    public String readBString() throws IOException {
        int length = this.readUnsignedByte();
        byte[] bytes = new byte[length];
        int bytesRead = this.read(bytes);
        assert bytesRead == length;

        String str = new String(bytes, StandardCharsets.UTF_8);
        return str;
    }

    /**
     * Reads a WString (2-byte length-prefixed).
     *
     * @return The string that was readFully.
     * @throws IOException
     */
    public String readWString() throws IOException {
        int length = this.readUnsignedShort();
        byte[] bytes = new byte[length];
        int bytesRead = this.read(bytes);
        assert bytesRead == length : String.format("bytes read %d, expected length %d", bytesRead, length);

        String str = new String(bytes, StandardCharsets.UTF_8);
        return str;
    }

    /**
     * Reads an LString (4-byte length-prefixed).
     *
     * @return The string that was readFully.
     * @throws IOException
     */
    public String readLString() throws IOException {
        int length = this.readInt();
        byte[] bytes = new byte[length];
        int bytesRead = this.read(bytes);
        assert bytesRead == length;

        String str = new String(bytes, StandardCharsets.UTF_8);
        return str;
    }

    /**
     * Reads a ZString (zero-terminated). Limit of 256 characters.
     *
     * @return The string that was readFully.
     * @throws IOException
     */
    public String readZString() throws IOException {
        byte[] buffer = readZStringRecursive(0);
        String str = new String(buffer, StandardCharsets.UTF_8);
        return str;
    };

    /**
     * Reads a ZString (zero-terminated). Accepts a size value that will be
     * used to pre-allocate space for the string. If the size value is 
     * incorrect, performance will be degraded.
     * 
     * @param size An estimation (or exact value) of the size of the string.
     * @return The string that was readFully.
     * @throws IOException
     */
    public String readZString(int size) throws IOException {
        final StringBuilder BUF = new StringBuilder(size);
        
        for (byte b = (byte)this.read(); b != 0; b = (byte)this.read()) {
            BUF.append((char) b);
        }

        return BUF.toString();
    };

    /**
     * Reads a BZString (byte-prefixed and zero-terminated).
     *
     * @return The string that was readFully.
     * @throws IOException
     */
    public String readBZString() throws IOException {
        int length = this.readUnsignedByte();
        byte[] buffer = readZStringRecursive(0);
        assert buffer.length + 1 == length;

        String str = new String(buffer, StandardCharsets.UTF_8);
        return str;
    }

    /**
     * Reads a LZString (long-prefixed and zero-terminated).
     *
     * @return The string that was readFully.
     * @throws IOException
     */
    public String readLZString() throws IOException {
        int length = this.readInt();
        String s = readZString(length);
        assert s.length() + 1 == length;
        return s;
    }

    /**
     * Recursively reads a ZString, effectively storing it on the stack.
     *
     * @param len The length of zstring already readFully.
     * @return A byte array large enough to store the zstring.
     * @throws IOException
     */
    private byte[] readZStringRecursive(int len) throws IOException {
        // Switch to a brute-force approach of the zstring is too long.
        if (len > 256) {
            it.unimi.dsi.fastutil.bytes.ByteList tail = new it.unimi.dsi.fastutil.bytes.ByteArrayList(256);
            for (byte b = (byte)this.read(); b != 0; b = (byte)this.read()) {
                tail.add(b);
            }
            
            byte[] result = new byte[len + tail.size()];
            for (int i = 0; i < tail.size(); i++) {
                result[i+len] = tail.get(i);
            }
            return result;
        }

        // Recursive version.
        
        byte b = (byte) this.read(); 
        
        if (0 == b) {
            return new byte[len];

        } else {
            byte[] buffer = readZStringRecursive(len + 1);
            buffer[len] = b;
            return buffer;
        }
    }

    final private byte[] BUFFER;

}
