/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.DataInput;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.LinkedList;
import java.util.Objects;

/**
 * A rough variant of <code>java.io.DataInputStream</code> that is
 * little-endian.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
public class LittleEndianInputStream extends LittleEndianInput {

    /**
     * Creates a new <code>LittleEndianInputStream</code> that wraps a byte
     * array.
     *
     * @param buf The byte array to wrap.
     * @return The new <code>LittleEndianInputStream</code>.
     *
     */
    static public LittleEndianInputStream wrap(byte[] buf) {
        Objects.requireNonNull(buf);

        final ByteArrayInputStream BIS = new ByteArrayInputStream(buf);
        return new LittleEndianInputStream(BIS, false, false);
    }

    /**
     * Creates a new <code>LittleEndianInputStream</code> that wraps an
     * <code>InputStream</code>.
     *
     * @param orig The <code>InputStream</code> to wrap.
     * @return The new <code>LittleEndianInputStream</code>.
     *
     */
    static public LittleEndianInputStream wrap(InputStream orig) {
        Objects.requireNonNull(orig);
        return new LittleEndianInputStream(orig, false, false);
    }

    /**
     * Creates a new <code>LittleEndianInputStream</code> that opens a
     * <code>File</code>.
     *
     * @param file The <code>File</code> to open.
     * @return The new <code>LittleEndianInputStream</code>.
     * @throws FileNotFoundException
     *
     */
    static public LittleEndianInputStream open(File file) throws FileNotFoundException {
        Objects.requireNonNull(file);
        assert file.isFile();
        assert file.canRead();
        assert file.exists();

        final BufferedInputStream BIS = new BufferedInputStream(new FileInputStream(file));
        return new LittleEndianInputStream(BIS, false, false);
    }

    /**
     * Creates a new <code>LittleEndianInputStream</code> that wraps a byte
     * array and maintains a context window for debugging.
     *
     * @param buf The byte array to wrap.
     * @return The new <code>LittleEndianInputStream</code>.
     *
     */
    static public LittleEndianInputStream debug(byte[] buf) {
        Objects.requireNonNull(buf);

        final ByteArrayInputStream BIS = new ByteArrayInputStream(buf);
        return new LittleEndianInputStream(BIS, true, false);
    }

    /**
     * Creates a new <code>LittleEndianInputStream</code> that wraps an
     * <code>InputStream</code> and maintains a context window for debugging.
     *
     * @param orig The <code>InputStream</code> to wrap.
     * @return The new <code>LittleEndianInputStream</code>.
     *
     */
    static public LittleEndianInputStream debug(InputStream orig) {
        Objects.requireNonNull(orig);
        return new LittleEndianInputStream(orig, true, false);
    }

    /**
     * Creates a new <code>LittleEndianInputStream</code> that wraps an
     * <code>InputStream</code> and maintains a digest.
     *
     * @param orig The <code>InputStream</code> to wrap.
     * @return The new <code>LittleEndianInputStream</code>.
     *
     */
    static public LittleEndianInputStream wrapD(InputStream orig) {
        Objects.requireNonNull(orig);
        return new LittleEndianInputStream(orig, false, true);
    }

    /**
     * Creates a new <code>LittleEndianInputStream</code> that opens a
     * <code>File</code> and maintains a digest.
     *
     * @param file The <code>File</code> to open.
     * @return The new <code>LittleEndianInputStream</code>.
     * @throws FileNotFoundException
     *
     */
    static public LittleEndianInputStream openD(File file) throws FileNotFoundException {
        Objects.requireNonNull(file);
        assert file.isFile();
        assert file.canRead();
        assert file.exists();

        final BufferedInputStream BIS = new BufferedInputStream(new FileInputStream(file));
        return new LittleEndianInputStream(BIS, false, true);
    }

    /**
     * Creates a new <code>LittleEndianDataInput</code> that wraps around a
     * supplied <code>InputStream</code>.
     *
     * @param bigEnd The <code>InputStream</code> to wrap.
     * @param context A flag indicating whether to maintain a context window for
     * debugging.
     * @param digest A flag indicating whether to calculate a digest.
     *
     */
    protected LittleEndianInputStream(InputStream bigEnd, boolean context, boolean digest) {
        Objects.requireNonNull(bigEnd);

        if (digest) {
            InputStream i = bigEnd;

            try {
                final MessageDigest DIGEST = MessageDigest.getInstance("MD5");
                final DigestInputStream DIGESTER = new DigestInputStream(bigEnd, DIGEST);
                i = DIGESTER;
            } catch (NoSuchAlgorithmException ex) {
            } finally {
                this.BIGEND = i;
            }

        } else {
            this.BIGEND = bigEnd;
        }

        this.CTX_NEXT = (context ? new LinkedList<>() : null);
        this.CTX_PREV = (context ? new LinkedList<>() : null);
        this.position = 0;
    }

    /**
     * Accessor for the underlying big-endian <code>InputStream</code>.
     *
     * @return The underlying stream.
     */
    public InputStream getBigEndian() {
        return this.BIGEND;
    }

    /**
     * @see java.io.InputStream#available()
     * @return The number of bytes available to be read.
     * @throws IOException
     */
    @Override
    public int available() throws IOException {
        if (this.hasContext()) {
            return this.BIGEND.available() + this.CTX_NEXT.size();
        }

        return this.BIGEND.available();
    }

    /**
     * @see java.io.InputStream#close()
     * @throws IOException
     */
    @Override
    public void close() throws IOException {
        this.BIGEND.close();
    }

    /**
     * @see DataInput#skipBytes(int)
     * @param n
     * @return
     * @throws IOException
     */
    @Override
    public int skipBytes(int n) throws IOException {
        if (this.hasContext()) {
            for (int i = 0; i < n; i++) {
                int b = this.read();
                if (b == -1) {
                    return b - i;
                }
            }
            return n;
        }

        return (int) this.BIGEND.skip(n);
    }

    /**
     * @see LittleEndianInput#read()
     * @return
     * @throws IOException
     */
    @Override
    public int read() throws IOException {
        int val = (this.hasContext() ? this.contextRead() : this.BIGEND.read());
        if (val != -1) {
            this.position++;
        } else {
            assert false;
        }
        return val;
    }

    /**
     * @see LittleEndianInput#read(byte[])
     * @param b
     * @return
     * @throws IOException
     */
    @Override
    public int read(byte[] b) throws IOException {
        return this.read(b, 0, b.length);
    }

    /**
     * @see LittleEndianInput#read(byte[], int, int)
     * @param b
     * @param off
     * @param len
     * @return
     * @throws IOException
     */
    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        Objects.requireNonNull(b);
        assert off >= 0;
        assert off <= b.length;
        assert len >= 0;
        assert len <= b.length;
        assert off + len <= b.length;

        int bytesRead = (this.hasContext()
                ? this.contextRead(b, off, len)
                : this.BIGEND.read(b, off, len));

        if (bytesRead < 0) {
            return 0;
        }

        this.position += bytesRead;
        return bytesRead;
    }

    /**
     * @see LittleEndianInput#toString()
     * @return
     */
    @Override
    public String toString() {
        long available = -1;
        try {
            available = this.available();
        } catch (IOException ex) {
        }

        String s = String.format("%s (pos=%d/%d)", super.toString(), this.getPosition(), available);

        if (this.hasContext()) {
            return s + '\n' + this.contextString();
        } else {
            return s;
        }
    }

    /**
     * @return A flag indicating that the stream has a context.
     */
    private boolean hasContext() {
        return null != this.CTX_NEXT;
    }

    /**
     * @see LittleEndianInput#read()
     * @return
     * @throws IOException
     */
    private int contextRead() throws IOException {
        // Make sure the forward context has been populated.
        // If there's nothing to put in it, return -1.
        this.fillContext();
        if (this.CTX_NEXT.isEmpty()) {
            return -1;
        }

        // Get the next context byte, append it to the prev context.
        int val = this.CTX_NEXT.pop();
        this.CTX_PREV.push(val);

        // Eliminate excess context.
        this.trimContext();

        assert val != -1;
        return val;
    }

    /**
     * @see LittleEndianInput#read(byte[], int, int)
     * @param b
     * @param off
     * @param len
     * @return
     * @throws IOException
     */
    private int contextRead(byte[] b, int off, int len) throws IOException {
        this.fillContext();
        int ctxSize = this.CTX_NEXT.size();

        // Handle big arrays.
        if (len > ctxSize) {
            for (int i = 0; i < ctxSize; i++) {
                b[off + i] = this.CTX_NEXT.pop().byteValue();
            }

            int r = this.BIGEND.read(b, off + ctxSize, len - ctxSize);
            r += ctxSize;

            this.CTX_PREV.clear();
            this.fillContext();
            return r;
        }

        // Handle small arrays.
        for (int i = 0; i < len; i++) {
            Integer val = this.CTX_NEXT.pop();
            this.CTX_PREV.push(val);
            b[off + i] = val.byteValue();
        }

        this.fillContext();
        this.trimContext();
        return len;
    }

    /**
     * Refill the forward context. Only available bytes will be added, no -1s or
     * anything.
     *
     * @throws IOException
     */
    private void fillContext() throws IOException {
        assert null != this.CTX_NEXT;

        while (this.CTX_NEXT.size() < CONTEXT_SIZE) {
            int b = this.BIGEND.read();
            if (b == -1) {
                break;
            }

            this.CTX_NEXT.addLast(b);
        }
    }

    /**
     * Trims the backward context.
     *
     * @throws IOException
     */
    private void trimContext() throws IOException {
        assert null != this.CTX_PREV;

        while (this.CTX_PREV.size() > CONTEXT_SIZE) {
            this.CTX_PREV.removeLast();
        }
    }

    /**
     *
     * @return
     */
    private String contextString() {
        java.io.StringWriter writer = new java.io.StringWriter();

        final int[] STUFF = new int[this.CTX_PREV.size() + this.CTX_NEXT.size()];
        int pos = this.CTX_PREV.size();

        int p = 0;
        int n1 = this.CTX_PREV.size();

        for (int val : this.CTX_PREV) {
            STUFF[n1 - p - 1] = val;
            p++;
        }
        for (int val : this.CTX_NEXT) {
            STUFF[p] = val;
            p++;
        }

        for (int k = 0; k < STUFF.length; k += 32) {

            /*for (int i = 0; i < 32 && i + k < STUFF.length; i++) {
                if (i % 32 == 0) {
                    writer.append(' ');
                }

                int val = STUFF[k + i];
                char ch = (char) val;
                if (PRED.test(Character.toString(ch))) {
                    writer.append(String.format(" %s", (char) STUFF[k + i]));
                } else {
                    writer.append("  ");
                }

                writer.append(' ');
            }
            writer.append('\n');*/
            for (int i = 0; i < 32 && i + k < STUFF.length; i++) {
                p = k + i;

                if (p == pos) {
                    writer.append('[');
                } else if (i % 32 == 0) {
                    writer.append(' ');
                }

                writer.append(String.format("%02x", STUFF[k + i]));

                if (p == pos) {
                    writer.append(']');
                } else {
                    writer.append(' ');
                }
            }
            writer.append('\n');
        }

        return writer.toString();
    }

    /**
     * If the stream was constructed with a <code>DigestInputStream</code>,
     * retrieves the digest.
     *
     * @return
     */
    public Long getDigest() {
        if (this.BIGEND instanceof DigestInputStream) {
            DigestInputStream DIS = (DigestInputStream) this.getBigEndian();

            try {
                MessageDigest DIGEST = (MessageDigest) DIS.getMessageDigest().clone();
                byte[] buf = DIGEST.digest();
                java.util.zip.CRC32 CRC32 = new java.util.zip.CRC32();
                CRC32.update(buf);
                long digest = CRC32.getValue();
                return digest;

            } catch (CloneNotSupportedException ex) {
                return null;
            }

        } else {
            return null;
        }
    }

    /**
     * @return The position field.
     */
    public int getPosition() {
        return this.position;
    }

    static final private int CONTEXT_SIZE = 256;
    final private InputStream BIGEND;
    final private LinkedList<Integer> CTX_NEXT;
    final private LinkedList<Integer> CTX_PREV;
    private int position;

    final java.util.regex.Pattern PATTERN = java.util.regex.Pattern.compile("[ -~]");
    final java.util.function.Predicate<String> PRED = PATTERN.asPredicate();
}
