/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.*;
import java.util.*;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import restringer.pex.Pex;
import restringer.bsa.BSAParser;
import restringer.esp.StringsFile;
import restringer.pex.PexObject;

/**
 * Describes a mod as a 4-tuple of a directory, a list of ESPs/ESMs, a list of
 * BSAs, and a list of script files.
 *
 * @author Mark Fairchild
 * @version 2016/07/07
 */
final public class Mod implements java.io.Serializable {

    /**
     * Creates a new <code>Mod</code> from a <code>File</code> representing the
     * directory containing the mod's files. The directory will be scanned to
     * create lists of all the important files.
     *
     * @param dir The directory containing the mod.
     */
    public Mod(File dir) {
        if (dir == null) {
            throw new NullPointerException("The mod directory can't be null.");
        } else if (dir.exists() && !dir.isDirectory()) {
            throw new IllegalArgumentException("The 'directory' argument isn't a directory.");
        }

        // This bit of fanciness lets the directory also be a directory
        // containing a data folder.
        if (dir.exists()
                && 0 == dir.listFiles(BSA_FILTER).length
                && 0 == dir.listFiles(PEX_FILTER).length
                && 0 == dir.listFiles(ESP_FILTER).length
                && 0 < dir.listFiles(DATA_FILTER).length) {
            this.DIRECTORY = dir.listFiles(DATA_FILTER)[0];
        } else {
            this.DIRECTORY = dir;
        }

        String dirName = this.DIRECTORY.getName();
        String parentName = this.DIRECTORY.getParentFile().getName();
        if (dirName.equalsIgnoreCase("data") && parentName.equalsIgnoreCase("skyrim")) {
            this.MODNAME = "Skyrim DATA Directory";
        } else {
            this.MODNAME = this.DIRECTORY.getName();
        }

        this.SCRIPT_DIR = new File(this.DIRECTORY, Mod.SCRIPTS_PATH);
        this.STRING_DIR = new File(this.DIRECTORY, Mod.STRINGS_PATH);
        this.SHORTNAME = (this.MODNAME.length() < 25 ? this.MODNAME : this.MODNAME.substring(0, 22) + "...");
        this.TIMER = new Timer(String.format("Timer for %s", this.MODNAME));
        this.ESP_FILES = new java.util.ArrayList<>();
        this.BSA_FILES = new java.util.ArrayList<>();
        this.PEX_FILES = new java.util.ArrayList<>();
        this.STR_FILES = new java.util.ArrayList<>();
        this.SCRIPTS = new java.util.concurrent.ConcurrentHashMap<>();
        this.fastAnalysis = null;

        File[] espFiles = this.DIRECTORY.listFiles(ESP_FILTER);
        File[] bsaFiles = this.DIRECTORY.listFiles(BSA_FILTER);
        File[] pexFiles = this.SCRIPT_DIR.listFiles(PEX_FILTER);
        File[] strFiles = this.STRING_DIR.listFiles(STR_FILTER);

        if (!dir.exists()) {
            this.status = Status.DISABLED;
            LOG.warning(String.format("Mod \"%s\" doesn't exist.", this.MODNAME));

        } else {
            if (espFiles.length == 0) {
                LOG.fine(String.format("Mod \"%s\" contains no ESP or ESM files.", this.MODNAME));
            } else {
                this.ESP_FILES.addAll(Arrays.asList(espFiles));
                LOG.fine(String.format("Mod \"%s\" contains %d ESP/ESM files.", this.MODNAME, espFiles.length));
            }

            if (bsaFiles.length == 0) {
                LOG.fine(String.format("Mod \"%s\" contains no BSA files.", this.MODNAME));
            } else {
                this.BSA_FILES.addAll(Arrays.asList(bsaFiles));
                LOG.fine(String.format("Mod \"%s\" contains %d BSA files.", this.MODNAME, bsaFiles.length));
            }

            if (null == pexFiles) {
                LOG.fine(String.format("Mod \"%s\" contains no \"scripts\" folder.", this.MODNAME));
            } else if (pexFiles.length == 0) {
                LOG.fine(String.format("Mod \"%s\" contains no loose scripts.", this.MODNAME));
            } else {
                this.PEX_FILES.addAll(Arrays.asList(pexFiles));
                LOG.fine(String.format("Mod \"%s\" contains %d loose scripts.", this.MODNAME, pexFiles.length));
            }

            if (null == strFiles) {
                LOG.fine(String.format("Mod \"%s\" contains no \"scripts\" folder.", this.MODNAME));
            } else if (strFiles.length == 0) {
                LOG.fine(String.format("Mod \"%s\" contains no loose localization files.", this.MODNAME));
            } else {
                this.STR_FILES.addAll(Arrays.asList(strFiles));
                LOG.fine(String.format("Mod \"%s\" contains %d loose localization files.", this.MODNAME, strFiles.length));
            }

            this.status = (this.isEmpty() ? Status.DISABLED : Status.CHECKED);
        }
    }

    /**
     * @return The value of the status field.
     */
    public Status getStatus() {
        return status;
    }

    /**
     * Sets the status field.
     *
     * @param newStatus
     */
    public void setStatus(Status newStatus) {
        this.status = Objects.requireNonNull(newStatus);
    }

    /**
     *
     * @return Returns true if the mod contains no ESPs, ESMs, BSAs, or loose
     * script files.
     */
    public boolean isEmpty() {
        return this.BSA_FILES.isEmpty() && this.ESP_FILES.isEmpty() && this.PEX_FILES.isEmpty();
    }

    /**
     * @return The directory storing the <code>Mod</code>.
     */
    public File getDirectory() {
        return this.DIRECTORY;
    }

    /**
     * @return The number of BSA files.
     */
    public int getNumBSAs() {
        return this.BSA_FILES.size();
    }

    /**
     * @return The number of loose script files.
     */
    public int getNumLooseScripts() {
        return this.PEX_FILES.size();
    }

    /**
     * @return The number of loose script files.
     */
    public int getNumLooseStrings() {
        return this.STR_FILES.size();
    }

    /**
     * @return The number of ESP/ESM files.
     */
    public int getNumESPs() {
        return this.ESP_FILES.size();
    }

    /**
     * @return The name of the mod.
     */
    public String getName() {
        return this.MODNAME;
    }

    /**
     * @return A list of the names of the esp files in the mod.
     */
    public List<String> getESPNames() {
        final List<String> NAMES = new ArrayList<>(this.ESP_FILES.size());
        this.ESP_FILES.forEach(v -> NAMES.add(v.getName()));
        return NAMES;
    }

    /**
     * Reads the mod's scripts from its BSA files (if any) and loose script
     * files (if any).
     *
     * @param bestEffort If true, the method returns whatever it successfully
     * read rather than throwing an exception if something goes wrong.
     * @throws ScriptReadError Thrown to indicate that BSA files or loose
     * scripts could not be read.
     */
    public void readScripts(boolean bestEffort) throws ScriptReadError {
        TIMER.reset();
        TIMER.start();

        List<String> bsaErrorNames = new LinkedList<>();
        List<String> scriptErrorNames = new LinkedList<>();

        // Read the BSA files.
        this.BSA_FILES.parallelStream().forEach(bsaFile -> {
            try (LittleEndianRAF input = LittleEndianRAF.open(bsaFile)) {
                // Parse the BSA and then store its scripts.
                final BSAParser BSA = new BSAParser(bsaFile.getName(), input);
                Map<File, Pex> bsaScripts = BSA.getScripts();

                bsaScripts.forEach((f, pex) -> {
                    File filename = new File(bsaFile, f.getName());
                    this.SCRIPTS.put(filename, pex);
                });

                LOG.fine(String.format("Read %d BSA scripts from file \"%s\".", bsaScripts.size(), bsaFile.getName()));

            } catch (IOException ex) {
                // Store the names of corrupt BSA files.
                bsaErrorNames.add(bsaFile.getName());
                LOG.severe(String.format("Error while reading \"%s\".", bsaFile.getName()));
            }
        });

        int bsaFilesRead = BSA_FILES.size() - bsaErrorNames.size();
        LOG.fine(String.format("Mod \"%s\": read %d scripts from %d BSA files.", this.SHORTNAME, this.SCRIPTS.size(), bsaFilesRead));

        // Read the loose script files.
        this.PEX_FILES.forEach(scriptFile -> {
            try {
                // Parse the script.
                Pex script = Pex.readScript(scriptFile);
                IString filename = script.getFilename();

                // Check if this script is overwriting another one.
                // Resolve conflicts using the compilation date.
                Optional<File> match = this.SCRIPTS.keySet().stream().filter(f -> filename.equals(f.getName())).findAny();
                if (match.isPresent()) {
                    Pex existing = this.SCRIPTS.get(match.get());
                    long originalDate = existing.getDate();
                    long newDate = script.getDate();

                    if (newDate > originalDate) {
                        this.SCRIPTS.put(scriptFile, script);
                    }
                } else {
                    this.SCRIPTS.put(scriptFile, script);
                }
            } catch (IOException ex) {
                // Store the names of corrupt script files.
                scriptErrorNames.add(scriptFile.getName());
                LOG.severe(String.format("Error while reading \"%s\".", scriptFile.getName()));
            }
        });

        TIMER.stop();
        LOG.info(String.format("Mod \"%s\": finished reading %d script files; took %s", this.SHORTNAME, this.SCRIPTS.size(), this.TIMER.getFormattedTime()));

        if (!bestEffort && (!scriptErrorNames.isEmpty() || !bsaErrorNames.isEmpty())) {
            throw new ScriptReadError(bsaErrorNames, scriptErrorNames);
        } else if (!scriptErrorNames.isEmpty() || !bsaErrorNames.isEmpty()) {
            LOG.warning(String.format("Mod \"%s\": %d scripts and %d BSAs could not be read.", this.SHORTNAME, scriptErrorNames.size(), bsaErrorNames.size()));
        }
    }

    /**
     * Reads the mod's strings from its BSA files (if any) and loose stringtable
     * files (if any).
     *
     * @param espFile The espFile whose localizations are to be loaded.
     * @param language The language to retrieve.
     * @param bestEffort If true, the method returns whatever it successfully
     * read rather than throwing an exception if something goes wrong.
     * @return A list of <code>StringsFile</code>.
     * @throws StringsReadError Thrown to indicate that BSA files or loose
     * stringtables could not be read.
     */
    public List<StringsFile> readStrings(File espFile, String language, boolean bestEffort) throws StringsReadError {
        Objects.requireNonNull(espFile);
        if (!this.ESP_FILES.contains(espFile)) {
            throw new IllegalArgumentException("That plugin is not part of the Mod.");
        }

        final String NAME = espFile.getName().split("\\.(?=[^\\.]+$)")[0].toLowerCase();
        final String LANG = "_" + language.toLowerCase();

        final File BSAFILE = new File(espFile.getParentFile(), NAME + ".bsa");

        TIMER.reset();
        TIMER.start();

        final List<StringsFile> STRINGS = new ArrayList<>(3);
        List<String> bsaErrorNames = new LinkedList<>();
        List<String> strErrorNames = new LinkedList<>();

        // Read the BSA file, if any.
        if (this.BSA_FILES.contains(BSAFILE)) {
            try (LittleEndianRAF input = LittleEndianRAF.open(BSAFILE)) {
                // Parse the BSA and then store its scripts.
                final BSAParser BSA = new BSAParser(BSAFILE.getName(), input);
                List<StringsFile> bsaStrings = BSA.getStrings(language);
                STRINGS.addAll(bsaStrings);
                LOG.fine(String.format("Read %d BSA stringtables from file \"%s\".", bsaStrings.size(), BSAFILE.getName()));

            } catch (IOException ex) {
                // Store the names of corrupt BSA files.
                bsaErrorNames.add(BSAFILE.getName());
                LOG.severe(String.format("Error while reading \"%s\".", BSAFILE.getName()));
            }
        }

        int bsaStrCount = STRINGS.stream().mapToInt(s -> s.getMap().size()).sum();
        LOG.info(String.format("Mod \"%s\": read %d strings from %s.", this.SHORTNAME, bsaStrCount, BSAFILE.getName()));

        // Read the loose stringtable files.
        this.STR_FILES.stream()
                .filter(f -> f.getName().toLowerCase().contains(LANG))
                .filter(f -> f.getName().toLowerCase().contains(NAME))
                .forEach(strFile -> {
                    try (LittleEndianInput input = LittleEndianInputStream.open(strFile)) {

                        // Parse the stringtable.
                        StringsFile.Type type = StringsFile.Type.match(strFile.getName());
                        StringsFile str = new StringsFile(strFile.getName(), input, type);
                        STRINGS.add(str);

                    } catch (IOException ex) {
                        // Store the names of corrupt stringtable files.
                        strErrorNames.add(strFile.getName());
                        LOG.severe(String.format("Error while reading \"%s\".", strFile.getName()));
                    }
                });

        TIMER.stop();

        int strCount = STRINGS.stream().mapToInt(s -> s.getMap().size()).sum();
        LOG.info(String.format("Mod \"%s\": finished reading %d strings from %d stringtables; took %s", this.SHORTNAME, strCount, STRINGS.size(), this.TIMER.getFormattedTime()));

        if (!bestEffort && (!strErrorNames.isEmpty() || !bsaErrorNames.isEmpty())) {
            throw new StringsReadError(bsaErrorNames, strErrorNames);
        } else if (!strErrorNames.isEmpty() || !bsaErrorNames.isEmpty()) {
            LOG.warning(String.format("Mod \"%s\": %d scripts and %d BSAs could not be read.", this.SHORTNAME, strErrorNames.size(), bsaErrorNames.size()));
        }

        return STRINGS;
    }

    /**
     * Clears the scripts from memory.
     */
    public void clearScripts() {
        this.SCRIPTS.clear();
        System.gc();
    }

    /**
     * @return The Mod's scripts.
     */
    public Map<File, Pex> getScripts() {
        return Collections.unmodifiableMap(this.SCRIPTS);
    }

    /**
     * Just used to make serialization and deserialization from JSON files
     * easier.
     *
     * @return
     */
    @Override
    public String toString() {
        return this.DIRECTORY.getPath();
    }

    /**
     * @see Mod#analysis()
     * @return An analysis, if one is ready. Null otherwise.
     */
    public Analysis fastAnalysis() {
        return this.fastAnalysis;
    }

    /**
     * Reads the absolute minimum amount of the mod's data to make an estimate
     * of the number of strings that can be usefully restrung.
     *
     * @return The analysis.
     */
    public Analysis analysis() {
        synchronized (this) {
            if (null != this.fastAnalysis) {
                return this.fastAnalysis;
            }
            LOG.fine(String.format("Analyzing mod \"%s\"", this.SHORTNAME));
            TIMER.reset();
            TIMER.start();

            long scriptBytes = 0L;
            long bsaBytes = 0L;
            int numScripts = 0;
            int numStrings = 0;
            int numBSAs = 0;

            try {
                this.readScripts(true);
            } catch (ScriptReadError ex) {
            }

            for (File bsaFile : this.BSA_FILES) {
                bsaBytes += bsaFile.length();
                numBSAs++;
            }

            for (Map.Entry<File, Pex> e : this.SCRIPTS.entrySet()) {
                File file = e.getKey();
                Pex pex = e.getValue();
                numStrings += pex.STRINGS.size();
                scriptBytes += file.length();
                numScripts++;
            }

            final Set<IString> STRINGS = new it.unimi.dsi.fastutil.objects.ObjectOpenHashSet<>(numStrings);
            this.SCRIPTS.values().stream().map(v -> v.OBJECT)
                    .forEach(v -> STRINGS.addAll(v.getVariableNames()));

            this.fastAnalysis = new Analysis(scriptBytes, bsaBytes, numScripts, STRINGS.size(), numBSAs);

            TIMER.stop();
            LOG.info(String.format("Analyzed mod \"%s\", took %s.", this.SHORTNAME, this.TIMER.getFormattedTime()));
        }
        return this.fastAnalysis;
    }

    /**
     * @return A copy of the list of ESP files.
     */
    public List<File> getESPFiles() {
        return new ArrayList<>(this.ESP_FILES);
    }

    /**
     * @return A copy of the list of BSA files.
     */
    public List<File> getBSAFiles() {
        return new ArrayList<>(this.BSA_FILES);
    }

    /**
     * @return A copy of the list of PEX files.
     */
    public List<File> getPexFiles() {
        return new ArrayList<>(this.PEX_FILES);
    }

    /**
     * @see Object#hashCode()
     * @return
     */
    @Override
    public int hashCode() {
        return this.DIRECTORY.hashCode();
    }

    /**
     * @see Object#equals(java.lang.Object)
     * @param obj
     * @return
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Mod other = (Mod) obj;
        return Objects.equals(this.DIRECTORY, other.DIRECTORY);
    }

    final private File DIRECTORY;
    final private File SCRIPT_DIR;
    final private File STRING_DIR;
    final private String MODNAME;
    final private String SHORTNAME;
    final private Map<File, Pex> SCRIPTS;
    private Analysis fastAnalysis;
    private Status status;

    final private List<File> ESP_FILES;
    final private List<File> BSA_FILES;
    final private List<File> PEX_FILES;
    final private List<File> STR_FILES;
    final private Timer TIMER;

    static final private String SCRIPTS_PATH = "\\scripts";
    static final private String STRINGS_PATH = "\\strings";
    static final private String ESP_PATTERN = "(.+)\\.es[mp]";
    static final private String PEX_PATTERN = "(.+)\\.pex";
    static final private String BSA_PATTERN = "(.+)\\.bsa";
    static final private String STR_PATTERN = "^(.+)_(.+)\\.(.?.?strings)$";
    static final public Pattern ESP_REGEX = Pattern.compile(ESP_PATTERN, Pattern.CASE_INSENSITIVE);
    static final public Pattern PEX_REGEX = Pattern.compile(PEX_PATTERN, Pattern.CASE_INSENSITIVE);
    static final public Pattern BSA_REGEX = Pattern.compile(BSA_PATTERN, Pattern.CASE_INSENSITIVE);
    static final public Pattern STR_REGEX = Pattern.compile(STR_PATTERN, Pattern.CASE_INSENSITIVE);
    static final public FileFilter ESP_FILTER = (File f) -> ESP_REGEX.matcher(f.getName()).matches();
    static final public FileFilter PEX_FILTER = (File f) -> PEX_REGEX.matcher(f.getName()).matches();
    static final public FileFilter BSA_FILTER = (File f) -> BSA_REGEX.matcher(f.getName()).matches();
    static final public FileFilter STR_FILTER = (File f) -> STR_REGEX.matcher(f.getName()).matches();
    static final private FileFilter DATA_FILTER = (File f) -> (f.isDirectory() && f.getName().equalsIgnoreCase("data"));
    static final private Logger LOG = Logger.getLogger(Mod.class.getCanonicalName());

    /**
     * The status of an individual mod within a profile.
     */
    static public enum Status {
        CHECKED,
        UNCHECKED,
        DISABLED,
    }

    /**
     * Stores data about the mod that requires analysis to produce.
     *
     * @author Mark Fairchild
     * @version 2016/06/06
     */
    static public class Analysis {

        static Analysis combine(Analysis a1, Analysis a2) {
            return new Analysis(
                    a1.SCRIPTBYTES + a2.SCRIPTBYTES,
                    a1.BSABYTES + a2.BSABYTES,
                    a1.NUMSCRIPTS + a2.NUMSCRIPTS,
                    a1.NUMSTRINGS + a2.NUMSTRINGS,
                    a1.NUMBSAS + a2.NUMBSAS);
        }

        public Analysis() {
            this(0, 0, 0, 0, 0);
        }

        public Analysis(long scriptBytes, long bsaBytes, int scripts, int strings, int bsas) {
            assert 0 <= scriptBytes;
            assert 0 <= bsaBytes;
            assert 0 <= scripts;
            assert 0 <= strings;
            assert 0 <= bsas;
            this.NUMBYTES = scriptBytes + bsaBytes;
            this.BSABYTES = bsaBytes;
            this.SCRIPTBYTES = scriptBytes;
            this.NUMSCRIPTS = scripts;
            this.NUMSTRINGS = strings;
            this.NUMBSAS = bsas;
        }

        @Override
        public String toString() {
            return "Analysis{" + "NUMBYTES=" + NUMBYTES + ", SCRIPTBYTES=" + SCRIPTBYTES + ", BSABYTES=" + BSABYTES + ", NUMSCRIPTS=" + NUMSCRIPTS + ", NUMSTRINGS=" + NUMSTRINGS + ", NUMBSAS=" + NUMBSAS + '}';
        }

        final public long NUMBYTES;
        final public long SCRIPTBYTES;
        final public long BSABYTES;

        final public int NUMSCRIPTS;
        final public int NUMSTRINGS;
        final public int NUMBSAS;

        static public Analysis INPROGRESS = new Analysis(Long.MAX_VALUE, Long.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE);
    }

    /**
     * An exception that indicates a script file couldn't be read.
     */
    public class ScriptReadError extends IOException {

        /**
         * Creates a new <code>ScriptReadError</code> with a list of script
         * files and BSA files that were corrupt.
         *
         * @param bsaNames The list of names of the BSA files that were
         * unreadable.
         * @param scriptNames The list of names of the script files that were
         * unreadable.
         * @see Exception#Exception()
         */
        private ScriptReadError(List<String> scriptNames, List<String> bsaNames) {
            super("Some scripts could not be read: " + scriptNames.toString());
            Objects.requireNonNull(bsaNames);
            Objects.requireNonNull(scriptNames);
            this.MOD = Mod.this;
            this.BSA_NAMES = Collections.unmodifiableList(new ArrayList<>(bsaNames));
            this.SCRIPT_NAMES = Collections.unmodifiableList(new ArrayList<>(scriptNames));
        }

        /**
         * @return The <code>Mod</code> that generated the exception.
         */
        public Mod getMod() {
            return this.MOD;
        }

        /**
         * @return The list of BSA filenames that could not be read.
         */
        public List<String> getBSANames() {
            return this.BSA_NAMES;
        }

        /**
         * @return The list of script filenames that could not be read.
         */
        public List<String> getScriptNames() {
            return this.SCRIPT_NAMES;
        }

        final private Mod MOD;
        final private List<String> BSA_NAMES;
        final private List<String> SCRIPT_NAMES;
    }

    /**
     * An exception that indicates a stringtable file couldn't be read.
     */
    public class StringsReadError extends IOException {

        /**
         * Creates a new <code>StringsReadError</code> with a list of
         * stringtable files and BSA files that were corrupt.
         *
         * @param bsaNames The list of names of the BSA files that were
         * unreadable.
         * @param stringsNames The list of names of the script files that were
         * unreadable.
         * @see Exception#Exception()
         */
        private StringsReadError(List<String> strNames, List<String> bsaNames) {
            super("Some stringtables could not be read: " + strNames.toString());
            Objects.requireNonNull(bsaNames);
            Objects.requireNonNull(strNames);
            this.MOD = Mod.this;
            this.BSA_NAMES = Collections.unmodifiableList(new ArrayList<>(bsaNames));
            this.STR_NAMES = Collections.unmodifiableList(new ArrayList<>(strNames));
        }

        /**
         * @return The <code>Mod</code> that generated the exception.
         */
        public Mod getMod() {
            return this.MOD;
        }

        /**
         * @return The list of BSA filenames that could not be read.
         */
        public List<String> getBSANames() {
            return this.BSA_NAMES;
        }

        /**
         * @return The list of stringtable filenames that could not be read.
         */
        public List<String> getStringsNames() {
            return this.STR_NAMES;
        }

        final private Mod MOD;
        final private List<String> BSA_NAMES;
        final private List<String> STR_NAMES;
    }

}
