/*
 * Copyright 2016 Mark.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import java.io.*;
import java.util.logging.*;
import restringer.pex.*;
import org.apache.commons.cli.*;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.JOptionPane;

/**
 * Entry class for ReAssembler.
 *
 * @author Mark Fairchild
 * @version 2016/09/05
 */
public class ReAssembler {

    /**
     * @param line The parsed command line.
     */
    public static void execute(CommandLine line) {       
        // Set up the logger.
        try {
            LOG.getParent().addHandler(new FileHandler("ReAssembler.%g.log.xml", 1 << 20, 1));
        } catch (IOException ex) {
            LOG.warning("Couldn't create log file.");
            LOG.warning(ex.toString());
        }

        LOG.getParent().getHandlers()[0].setFormatter(new java.util.logging.Formatter() {
            @Override
            public String format(LogRecord record) {
                final java.util.logging.Level LEVEL = record.getLevel();
                final String MSG = record.getMessage();
                final String SRC = record.getSourceClassName() + "." + record.getSourceMethodName();
                return String.format("%s: %s (in %s)\n", LEVEL, MSG, SRC);
            }
        });

        LOG.setLevel(Level.FINE);
        LOG.getParent().getHandlers()[0].setLevel(Level.FINE);

        final AssemblyLevel LEVEL;

        if (line.hasOption('s')) {
            LEVEL = AssemblyLevel.STRIPPED;
        } else if (line.hasOption('b')) {
            LEVEL = AssemblyLevel.BYTECODE;
        } else {
            LEVEL = AssemblyLevel.FULL;
        }

        final boolean CONSOLE = line.hasOption('c');
        final List<String> FILES = line.getArgList();

        // If there's no console and there IS a windowing system of some kind, 
        // operate graphically.
        if (null == System.console() && !java.awt.GraphicsEnvironment.isHeadless()) {
            if (FILES.isEmpty()) {
                JOptionPane.showMessageDialog(null, "No script files!", "Error", JOptionPane.ERROR_MESSAGE);
                return;
            }

            final StringWriter BUF = new StringWriter(20480);
            final PrintWriter OUT = new PrintWriter(BUF);
            OUT.println("<html>");

            for (String filename : FILES) {
                final Matcher REGEX = FILENAMEPATTERN.matcher(filename);
                if (!REGEX.matches()) {
                    OUT.printf("<p>Skipped invalid filename: %s\n</p>", filename);
                    continue;
                }

                final File SCRIPTFILE = new File(filename);
                final Pex SCRIPT;

                try {
                    SCRIPT = Pex.readScript(SCRIPTFILE);
                } catch (IOException ex) {
                    OUT.printf("<p>Error reading script: %s\n</p>", filename);
                    continue;
                }

                try (StringWriter BUF2 = new StringWriter(2048)) {
                    SCRIPT.disassemble(BUF2, LEVEL);
                    OUT.println("<code><pre>");
                    OUT.append(BUF2.toString());
                    OUT.println("</pre></code>");
                } catch (Exception | Error ex) {
                    OUT.printf("<p>Error disassembling script: %s\n</p>", filename);
                    continue;
                }

                OUT.println("<hr/>");
            }
            OUT.println("</html>");

            javax.swing.JTextPane pane = new javax.swing.JTextPane();
            pane.setFont(pane.getFont().deriveFont(12.0f));
            pane.setEditable(true);
            pane.setContentType("text/html");
            pane.setPreferredSize(new java.awt.Dimension(850, 700));
            pane.setText(BUF.toString().replace("<", "%lt;"));
            javax.swing.JScrollPane scroller = new javax.swing.JScrollPane(pane);
            JOptionPane.showMessageDialog(null, scroller, "Disassembly", JOptionPane.INFORMATION_MESSAGE);

        } // Run in the console.
        else {
            for (String filename : FILES) {
                final Matcher REGEX = FILENAMEPATTERN.matcher(filename);
                if (!REGEX.matches()) {
                    System.err.println("Skipping invalid filename: " + filename);
                    continue;
                }

                final File SCRIPTFILE = new File(filename);
                final File OUTFILE;

                if (LEVEL != AssemblyLevel.FULL) {
                    OUTFILE = new File(filename.replaceFirst("\\.pex$", ".txt"));
                } else {
                    OUTFILE = new File(filename.replaceFirst("\\.pex$", ".psc"));
                }

                final Pex SCRIPT;

                try {
                    SCRIPT = Pex.readScript(SCRIPTFILE);
                } catch (IOException ex) {
                    System.err.println("Error reading script: " + ex.getMessage());
                    continue;
                }

                if (CONSOLE) {
                    try {
                        SCRIPT.disassemble(new PrintWriter(System.out), LEVEL);
                    } catch (RuntimeException ex) {
                        System.err.println("Error disassembling script: " + ex.getMessage());
                        continue;
                    }
                } else {
                    try (Writer OUT = new BufferedWriter(new FileWriter(OUTFILE))) {
                        SCRIPT.disassemble(OUT, LEVEL);
                    } catch (IOException ex) {
                        System.err.println("Error disassembling script: " + ex.getMessage());
                        continue;
                    }
                }

                if (!CONSOLE && LEVEL != AssemblyLevel.FULL) {
                    System.out.println("Wrote disassembled script for " + filename + " to " + OUTFILE);
                } else {
                    System.out.println("Wrote partially disassembled script for " + filename + " to " + OUTFILE);
                }
            }
        }
    }

    static final Logger LOG = Logger.getLogger(ReAssembler.class.getCanonicalName());
    static final Pattern FILENAMEPATTERN = Pattern.compile("^.*\\.(pex)$", Pattern.CASE_INSENSITIVE);

}
