/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer;

import com.google.gson.*;
import java.io.*;
import java.lang.reflect.Type;
import java.util.*;

/**
 * Stores the behaviour settings.
 *
 * @author Mark
 * @version 2016/05/21
 */
public class Settings {

    /**
     * Reads the settings file and returns a <code>Settings</code> object.
     *
     * @return The <code>Settings</code> object representing the settings file.
     * @throws IOException
     */
    static public Settings readSettings() throws IOException {
        try {
            if (!SETTINGS_FILE.exists()) {
                try (Writer output = new FileWriter(SETTINGS_FILE)) {
                    Settings settings = new Settings();
                    output.write(GSON.toJson(settings));
                }
            }
        } catch (IOException ex) {
            throw new IOException("Can't create the settings file.", ex);
        }

        if (!SETTINGS_FILE.canRead()) {
            throw new IOException("The settings file isn't readable.");
        }

        try (Reader input = new FileReader(SETTINGS_FILE)) {
            Settings settings = GSON.fromJson(input, Settings.class);
            return settings;
        } catch (IOException ex) {
            throw new IOException("Can't read the settings file.", ex);
        }
    }

    /**
     * Write a <code>Settings</code> object to the settings file.
     *
     * @param settings The <code>Settings</code> object to write.
     * @throws IOException
     */
    static public void writeSettings(Settings settings) throws IOException {
        java.util.logging.Logger.getLogger(Settings.class.getCanonicalName()).info("Writing Settings");

        if (SETTINGS_FILE.exists() && !SETTINGS_FILE.canWrite()) {
            throw new IOException("The settings file isn't writable.");
        }

        try (Writer output = new FileWriter(SETTINGS_FILE)) {
            output.write(GSON.toJson(settings));
        } catch (IOException ex) {
            throw new IOException("Can't write the settings file.", ex);
        }
    }

    /**
     * Creates a new <code>Settings</code> with default options.
     */
    public Settings() {
        //this.userModDirectory = null;
        //this.userOutputDirectory = null;
        //this.modOrganizerDirectory = null;
        //this.skyrimDirectory = null;
        this.profiles = new ArrayList<>();
        this.selectedProfile = -1;
    }

    /**
     * @return The currently selected profile. Generates a new default profile
     * if there are no profiles.
     */
    public Profile getSelectedProfile() {
        if (this.profiles.isEmpty()) {
            this.profiles.add(new Profile("Default"));
            this.selectedProfile = 0;
        } else if (this.selectedProfile < 0 || this.selectedProfile >= this.profiles.size()) {
            this.selectedProfile = 0;
        }

        return this.profiles.get(this.selectedProfile);
    }

    /**
     * The list of profiles.
     */
    public List<Profile> profiles;

    /**
     * The index of the currently selected profile.
     */
    public int selectedProfile;

    /**
     * The file storing the behaviour settings of Restringer.
     */
    static final private File SETTINGS_FILE;
    static final private Gson GSON;

    static {
        SETTINGS_FILE = new File("settings.json");

        final GsonBuilder BUILDER = new GsonBuilder();
        BUILDER.setPrettyPrinting();
        BUILDER.registerTypeAdapter(Mod.class, new ModSerializer());
        BUILDER.registerTypeAdapter(File.class, new FileSerializer());
        GSON = BUILDER.create();
    }

    /**
     * Reads and writes Mods.
     */
    static private class ModSerializer implements JsonDeserializer<Mod>, JsonSerializer<Mod> {

        @Override
        public Mod deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
                throws JsonParseException {
            Objects.requireNonNull(json);
            assert json.isJsonObject();

            Pair p = (Pair) GSON.fromJson(json, Pair.class);

            Mod.Status status = Mod.Status.valueOf((String) p.A);
            String path = (String) p.B;
            File directory = new File(path);
            Mod mod = new Mod(directory);
            mod.setStatus(status);
            return mod;
        }

        @Override
        public JsonElement serialize(Mod mod, Type typeOfSrc, JsonSerializationContext context) {
            Objects.requireNonNull(mod);
            Mod.Status status = mod.getStatus();

            File directory = mod.getDirectory();
            String path = directory.getPath();

            Pair<Mod.Status, String> pair = new Pair<>(status, path);
            return GSON.toJsonTree(pair);
        }
    }

    /**
     * Reads and writes Mods.
     */
    static private class FileSerializer implements JsonDeserializer<File>, JsonSerializer<File> {

        @Override
        public File deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
                throws JsonParseException {
            Objects.requireNonNull(json);
            String path = json.getAsJsonPrimitive().getAsString();
            File file = new File(path);
            return file;
        }

        @Override
        public JsonElement serialize(File file, Type typeOfSrc, JsonSerializationContext context) {
            Objects.requireNonNull(file);
            String path = file.getPath();
            return new JsonPrimitive(path);
        }
    }
}
