/*
 * Copyright 2016 Mark Fairchild
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.bsa;

import restringer.LittleEndianRAF;
import java.io.IOException;
import java.util.zip.DataFormatException;
import java.util.zip.Inflater;
import restringer.LittleEndianInput;

/**
 * Describes a block of BSA file data.
 *
 * @author Mark Fairchild
 * @version 2016/06/25
 */
public class FileData {

    /**
     * Creates a new <code>FileData</code> by reading from a
     * <code>LittleEndianRAF</code>.
     *
     * @param input The file from which to readFully.
     * @param record
     * @param header
     * @throws IOException
     *
     */
    FileData(LittleEndianRAF input, FileRecord record, BSAHeader header) throws IOException {
        input.seek(record.OFFSET);

        // If the filename is embedded, readFully it from the data block.
        // Otherwise retrieve it from the file record.
        if (header.isEmbedFilename()) {
            this.NAME = input.readBString();
        } else {
            this.NAME = record.name;
        }

        // If the file is compressed, inflate it. Otherwise just readFully it in.
        if (record.COMPRESSED) {
            this.ORIGINALSIZE = input.readInt();
            this.DATA = new byte[this.ORIGINALSIZE];

            final byte[] ZLIBDATA = new byte[record.SIZE - 4];
            input.readFully(ZLIBDATA);

            final Inflater INFLATER = new Inflater();
            INFLATER.setInput(ZLIBDATA);

            try {
                INFLATER.inflate(this.DATA);
            } catch (DataFormatException ex) {
                throw new IOException(ex);
            } finally {
                INFLATER.end();
            }

        } else {
            this.ORIGINALSIZE = record.SIZE;
            this.DATA = new byte[this.ORIGINALSIZE];
            input.readFully(this.DATA);
        }

    }

    /**
     * @return A little-endian stream view of the data.
     */
    public LittleEndianInput getStream() {
        return restringer.LittleEndianInputStream.wrap(this.DATA);
    }
    
    final public String NAME;
    final public int ORIGINALSIZE;
    final public byte[] DATA;
}
