/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.LinkedList;
import java.util.Objects;
import restringer.LittleEndianInputStream;
import restringer.LittleEndianDataOutput;

/**
 * Describes a Skyrim PEX script and will readFully and write it from iostreams.
 *
 * @author Mark Fairchild
 * @version 2016/04/26
 */
final public class ESP implements Entry {

    /**
     * Reads a mod file and creates an ESP object to represent it.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param modFile The mod file to readFully, which must exist and be
     * readable.
     * @param index The plugin index, for correcting FormIDs.
     * @param espList The list of ESPs, for correcting FormIDs.
     * @return The ESP object.
     *
     * @throws FileNotFoundException
     * @throws IOException
     *
     */
    static public ESP readESP(File modFile, int index, List<String> espList) throws FileNotFoundException, IOException {
        Objects.requireNonNull(modFile);
        assert modFile.isFile();
        assert modFile.canRead();
        assert modFile.exists();

        // Prepare input stream.
        try (LittleEndianInputStream input = LittleEndianInputStream.open(modFile)) {
            return new ESP(input, index, modFile.getName(), espList);
        }
    }

    /**
     * Writes an ESP object to a mod file.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param esp The ESP object to write.
     * @param modFile The mod file to write. If it exists, it must be a file and
     * it must be writable.
     *
     * @throws FileNotFoundException
     * @throws IOException
     *
     */
    static public void writeESP(ESP esp, File modFile) throws FileNotFoundException, IOException {
        assert !modFile.exists() || modFile.isFile();
        assert !modFile.exists() || modFile.canWrite();

        // Prepare output stream.
        try (LittleEndianDataOutput output = new LittleEndianDataOutput(new BufferedOutputStream(new FileOutputStream(modFile)))) {
            esp.write(output);
        }
    }

    /**
     * Skims a mod file and extracts EDIDs and ids.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param modFile The mod file to readFully, which must exist and be
     * readable.
     * @param index The plugin index, for correcting FormIDs.
     * @param espList The list of ESPs, for correcting FormIDs.
     * @return The ESPIDMap.
     *
     * @throws FileNotFoundException
     * @throws IOException
     *
     */
    static public ESPIDMap skimESP(File modFile, int index, List<String> espList) throws FileNotFoundException, IOException {
        assert modFile.isFile();
        assert modFile.canRead();
        assert modFile.exists();

        // Prepare input stream.
        try (LittleEndianInputStream input = LittleEndianInputStream.open(modFile)) {
            assert input.available() > 0;

            RecordTes4 tes4 = new RecordTes4(input, index, modFile.getName(), espList);

            while (input.available() > 0) {
                Record.skimRecord(input, tes4);
            }

            return tes4.IDMAP;
        }
    }

    /**
     * Creates an ESP by reading from a LittleEndianInputStream.
     *
     * @param input A LittleEndianInputStream for a Skyrim PEX file.
     * @param index The plugin index, for correcting FormIDs.
     * @param name The name of the plugin.
     * @param espList The list of ESPs, for correcting FormIDs.
     * @throws IOException Exceptions aren't handled.
     */
    public ESP(LittleEndianInputStream input, int index, String name, List<String> espList) throws IOException {
        assert input.available() > 0;
        this.RECORDS = new LinkedList<>();

        RecordTes4 tes4 = new RecordTes4(input, index, name, espList);
        this.RECORDS.add(tes4);

        while (input.available() > 0) {
            Record record = Record.readRecord(input, tes4);
            this.RECORDS.add(record);
        }

    }

    /**
     * Write the ESP to a <code>LittleEndianDataOutput</code>.
     *
     * @param output The <code>LittleEndianDataOutput</code> to write.
     * @throws IOException IO errors aren't handled at all, they are simply
     * passed on.
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        for (Record record : this.RECORDS) {
            record.write(output);
        }
    }

    /**
     * @return The calculated size of the field.
     * @see Entry#calculateSize()
     */
    @Override
    public int calculateSize() {
        int sum = 0;
        sum += this.RECORDS.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    /**
     * Pretty-prints the ESP.
     *
     * @return A string representation of the ESP.
     */
    @Override
    public String toString() {
        final StringBuilder BUF = new StringBuilder();
        this.RECORDS.forEach(record -> BUF.append(record.toString()));
        return BUF.toString();
    }

    final private List<Record> RECORDS;

}
