/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.util.List;
import java.util.Optional;

/**
 * Stores somethin'.
 *
 * @author Mark Fairchild
 * @version 2016/09/08
 */
final public class ESPIDMap {

    /**
     * Creates a new <code>ESPIDMap</code>.
     *
     * @param espName The name of the ESP.
     */
    public ESPIDMap(String espName) {
        this.MAP = it.unimi.dsi.fastutil.ints.Int2ObjectMaps.synchronize(new it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap<>());
        this.ESPNAME = espName;
    }

    /**
     * Adds an entry for a record.
     *
     * @param formID The formId of the record.
     * @param fields The record's fields, which will be sampled for suitable
     * names.
     */
    public void addRecord(int formID, List<Field> fields) {
        final Info INFO = new Info(fields);
        this.MAP.put(formID, INFO);
    }

    /**
     * Adds all of the entries from on <code>ESPIDMap</code> to another.
     *
     * @param other
     */
    public void addAll(ESPIDMap other) {
        this.MAP.putAll(other.MAP);
    }

    /**
     * Tries to retrieve a suitable name for a formID.
     *
     * @param formID
     * @param strings The StringTable.
     * @return
     */
    public String getName(int formID, StringTable strings) {
        if (!this.MAP.containsKey(formID)) {
            return null;
        }

        final Info INFO = this.MAP.get(formID);

        if (null != INFO.FULL && null != strings) {
            if (INFO.FULL.hasString()) {
                return INFO.FULL.getString();
                
            } else if (INFO.FULL.hasIndex()) {
                int index = INFO.FULL.getIndex();
                String lookup = strings.get(index);
                if (lookup != null) {
                    return lookup;
                }
            }
        }

        if (null != INFO.NAME) {
            int baseID = INFO.NAME.getFormID();
            assert baseID != formID;
            String baseName = this.getName(baseID, strings);
            if (null != baseName) {
                return baseName;
            }
        }

        if (null != INFO.EDID) {
            return INFO.EDID.getValue();
        }

        return null;
    }

    /**
     * @see Object#toString()
     * @return
     */
    @Override
    public String toString() {
        return this.ESPNAME;
    }

    /**
     * The actual mapping.
     */
    final private it.unimi.dsi.fastutil.ints.Int2ObjectMap<Info> MAP;

    /**
     * The name of the map.
     */
    final private String ESPNAME;

    /**
     * Stores ID information.
     */
    static final private class Info {

        public Info(List<Field> fields) {
            Optional<Field> edid = fields.stream().parallel().filter(v -> v instanceof FieldEDID).findAny();
            if (edid.isPresent()) {
                this.EDID = (FieldEDID) edid.get();
            } else {
                this.EDID = null;
            }

            Optional<Field> full = fields.stream().parallel().filter(v -> v instanceof FieldFull).findAny();
            if (full.isPresent()) {
                this.FULL = (FieldFull) full.get();
            } else {
                this.FULL = null;
            }

            Optional<Field> name = fields.stream().parallel().filter(v -> v instanceof FieldName).findAny();
            if (name.isPresent()) {
                this.NAME = (FieldName) name.get();
            } else {
                this.NAME = null;
            }
        }

        final public FieldEDID EDID;
        final public FieldFull FULL;
        final public FieldName NAME;
    }
}
