/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;
import restringer.IString;
import restringer.LittleEndianInputStream;

/**
 * FieldBasic represents all fields that aren't a VMAD section.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
public class FieldBasic implements Field {

    /**
     * Creates a new FieldBasic by reading it from a LittleEndianDataInput.
     *
     * @param code The field code.
     * @param input The field data.
     * @param size The amount of data.
     * @param big A flag indicating that this is a BIG field.
     * @throws IOException Exceptions are not handled at all.
     */
    public FieldBasic(IString code, LittleEndianInput input, int size, boolean big) throws IOException {
        Objects.requireNonNull(input);
        this.CODE = code;
        this.BIG = big;        
        this.DATA = new byte[size];
        input.read(this.DATA);        
    }

    /**
     * @see Entry#write(transposer.LittleEndianDataOutput)
     * @param output The LittleEndianDataOutput.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.write(this.CODE.getBytes());

        if (this.BIG) {
            output.writeShort(0);
        } else {
            output.writeShort(this.DATA.length);
        }

        output.write(this.DATA);
    }

    /**
     * @return The calculated size of the field.
     * @see Entry#calculateSize()
     */
    @Override
    public int calculateSize() {
        return 6 + DATA.length;
    }

    /**
     * Returns the field code.
     *
     * @return The field code.
     */
    @Override
    public IString getCode() {
        return this.CODE;
    }

    /**
     * Returns a copy of the data section.
     *
     * @return A copy of the data array.
     */
    public byte[] getData() {
        return this.DATA;
    }

    /**
     * @return Returns a decoder stream over the data.
     */
    public LittleEndianInput getDecoder() {
        return LittleEndianInputStream.wrap(this.DATA);
    }
    
    /**
     * Returns a String representation of the Field, which will just be the code
     * string.
     *
     * @return A string representation.
     *
     */
    @Override
    public String toString() {
        final StringBuilder BUF = new StringBuilder();
        BUF.append(this.getCode()).append("=").append(Arrays.toString(this.DATA));
        return BUF.toString();
    }

    final private IString CODE;
    final private byte[] DATA;
    final boolean BIG;

}
