/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.List;
import restringer.IString;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;

/**
 * Describes script fragments for INFO records and PACK records.
 *
 * @author Mark
 * @version 2016/04/23
 */
public class FragmentInfoPack extends FragmentBase {

    public FragmentInfoPack(LittleEndianInput input) throws IOException {
        this.UNKNOWN = input.readByte();
        this.FLAGS = input.readByte();
        this.FILENAME = IString.get(input.readUTF());
        this.FRAGMENTS = new java.util.LinkedList<>();

        int flagsCount = FragmentBase.NumberOfSetBits(this.FLAGS);
        for (int i = 0; i < flagsCount; i++) {
            Fragment fragment = new Fragment(input);
            this.FRAGMENTS.add(fragment);
        }
    }

    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.writeByte(this.UNKNOWN);
        output.writeByte(this.FLAGS);
        output.writeUTF(this.FILENAME.toString());

        for (Fragment fragment : this.FRAGMENTS) {
            fragment.write(output);
        }
    }

    @Override
    public int calculateSize() {
        int sum = 4 + this.FILENAME.length();
        sum += this.FRAGMENTS.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    final byte UNKNOWN;
    final byte FLAGS;
    final IString FILENAME;
    final List<Fragment> FRAGMENTS;

    /**
     *
     */
    public class Fragment implements Entry {

        public Fragment(LittleEndianInput input) throws IOException {
            this.UNKNOWN = input.readByte();
            this.SCRIPTNAME = IString.get(input.readUTF());
            this.FRAGMENTNAME = IString.get(input.readUTF());
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeByte(this.UNKNOWN);
            output.writeUTF(this.SCRIPTNAME.toString());
            output.writeUTF(this.FRAGMENTNAME.toString());
        }

        @Override
        public int calculateSize() {
            return 5 + this.SCRIPTNAME.length() + this.FRAGMENTNAME.length();
        }

        final private byte UNKNOWN;
        final private IString SCRIPTNAME;
        final private IString FRAGMENTNAME;
    }
}
