/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.List;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.IString;

/**
 * Describes script fragments for QUST records.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
public class FragmentScen extends FragmentBase {

    public FragmentScen(LittleEndianInput input) throws IOException {
        this.UNKNOWN = input.readByte();
        this.FLAGS = input.readByte();
        this.FILENAME = input.readUTF();
        this.FRAGMENTS = new java.util.LinkedList<>();
        this.PHASES = new java.util.LinkedList<>();

        int flagCount = FragmentBase.NumberOfSetBits(this.FLAGS);
        for (int i = 0; i < flagCount; i++) {
            Fragment fragment = new Fragment(input);
            this.FRAGMENTS.add(fragment);
        }

        int phaseCount = input.readUnsignedShort();
        for (int i = 0; i < phaseCount; i++) {
            Phase phase = new Phase(input);
            this.PHASES.add(phase);
        }
    }

    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.writeByte(this.UNKNOWN);
        output.writeByte(this.FLAGS);
        output.writeUTF(this.FILENAME);

        for (Fragment fragment : this.FRAGMENTS) {
            fragment.write(output);
        }

        output.writeShort(this.PHASES.size());
        for (Phase phase : this.PHASES) {
            phase.write(output);
        }
    }

    @Override
    public int calculateSize() {
        int sum = 6;
        sum += this.FRAGMENTS.stream().mapToInt(v -> v.calculateSize()).sum();
        sum += this.PHASES.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    final private byte UNKNOWN;
    final private byte FLAGS;
    final private String FILENAME;
    final private List<Fragment> FRAGMENTS;
    final private List<Phase> PHASES;

    /**
     *
     */
    public class Fragment implements Entry {

        public Fragment(LittleEndianInput input) throws IOException {
            this.UNKNOWN = input.readByte();
            this.SCRIPTNAME = IString.get(input.readUTF());
            this.FRAGMENTNAME = IString.get(input.readUTF());
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeByte(this.UNKNOWN);
            output.writeUTF(this.SCRIPTNAME.toString());
            output.writeUTF(this.FRAGMENTNAME.toString());
        }

        @Override
        public int calculateSize() {
            return 5 + this.SCRIPTNAME.length() + this.FRAGMENTNAME.length();
        }

        final private byte UNKNOWN;
        final private IString SCRIPTNAME;
        final private IString FRAGMENTNAME;
    }

    /**
     *
     */
    public class Phase implements Entry {

        public Phase(LittleEndianInput input) throws IOException {
            this.UNKNOWN1 = input.readByte();
            this.PHASE = input.readInt();
            this.UNKNOWN2 = input.readByte();
            this.SCRIPTNAME = IString.get(input.readUTF());
            this.FRAGMENTNAME = IString.get(input.readUTF());
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeByte(this.UNKNOWN1);
            output.writeInt(this.PHASE);
            output.writeByte(this.UNKNOWN2);
            output.writeUTF(this.SCRIPTNAME.toString());
            output.writeUTF(this.FRAGMENTNAME.toString());
        }

        @Override
        public int calculateSize() {
            return 10 + this.SCRIPTNAME.length() + this.FRAGMENTNAME.length();
        }

        final private byte UNKNOWN1;
        final private int PHASE;
        final private byte UNKNOWN2;
        final private IString SCRIPTNAME;
        final private IString FRAGMENTNAME;
    }
}
