/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import java.util.List;

/**
 * Stores the data for a script property. Basically a fancy VarArg.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
abstract public class PropertyData implements Entry {

    static public PropertyData readPropertyData(byte type, LittleEndianInput input) throws IOException {
        assert input.available() > 0;

        if (type < 10) {
            switch (type) {
                case 0:
                    return new NullData(input);
                case 1:
                    return new ObjectData(input);
                case 2:
                    return new StringData(input);
                case 3:
                    return new IntData(input);
                case 4:
                    return new FloatData(input);
                case 5:
                    return new BoolData(input);
                default:
                    throw new IOException(String.format("Invalid property type: %d", type));
            }
        } else {
            ArrayData<PropertyData> array = new ArrayData<>(input);
            int itemCount = input.readInt();
            assert 0 <= itemCount;

            switch (type) {
                case 11:
                    for (int i = 0; i < itemCount; i++) {
                        ObjectData data = new ObjectData(input);
                        array.DATA.add(data);
                    }
                    return array;
                case 12:
                    for (int i = 0; i < itemCount; i++) {
                        StringData data = new StringData(input);
                        array.DATA.add(data);
                    }
                    return array;
                case 13:
                    for (int i = 0; i < itemCount; i++) {
                        IntData data = new IntData(input);
                        array.DATA.add(data);
                    }
                    return array;
                case 14:
                    for (int i = 0; i < itemCount; i++) {
                        FloatData data = new FloatData(input);
                        array.DATA.add(data);
                    }
                    return array;
                case 15:
                    for (int i = 0; i < itemCount; i++) {
                        BoolData data = new BoolData(input);
                        array.DATA.add(data);
                    }
                    return array;
                default:
                    throw new IOException();
            }
        }
    }

    /**
     * Stores a Null property data.
     *
     */
    static public class NullData extends PropertyData {
        
        public NullData(LittleEndianInput input) throws IOException {
            //this.DATA = input.readInt();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            //output.writeInt(DATA);
        }

        @Override
        public int calculateSize() {
            return 0;
        }

        //final private int DATA;
    }

    /**
     * Stores an Array property data.
     *
     * @param <T> The type of PropertyData stored in the array.
     */
    static public class ArrayData<T extends PropertyData> extends PropertyData {

        public ArrayData(LittleEndianInput input) throws IOException {
            this.DATA = new java.util.LinkedList<>();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeInt(this.DATA.size());
            for (T t : this.DATA) {
                t.write(output);
            }
        }

        @Override
        public int calculateSize() {
            int sum = 4;
            sum += this.DATA.stream().mapToInt(t -> t.calculateSize()).sum();
            return sum;
        }

        final public List<T> DATA;

    }

    /**
     * Stores an Object property data.
     */
    static public class ObjectData extends PropertyData {

        public ObjectData(LittleEndianInput input) throws IOException {
            this.DATA = input.readLong();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeLong(DATA);
        }

        @Override
        public int calculateSize() {
            return 8;
        }

        final private long DATA;
    }

    /**
     * Stores a String property data.
     */
    static public class StringData extends PropertyData {

        public StringData(LittleEndianInput input) throws IOException {
            this.DATA = input.readUTF();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeUTF(DATA);
        }

        @Override
        public int calculateSize() {
            return 2 + this.DATA.length();
        }

        final private String DATA;
    }

    /**
     * Stores an integer property data.
     */
    static public class IntData extends PropertyData {

        public IntData(LittleEndianInput input) throws IOException {
            this.DATA = input.readInt();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeInt(DATA);
        }

        @Override
        public int calculateSize() {
            return 4;
        }

        final private int DATA;
    }

    /**
     * Stores a float property data.
     */
    static public class FloatData extends PropertyData {

        public FloatData(LittleEndianInput input) throws IOException {
            this.DATA = input.readFloat();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeFloat(DATA);
        }

        @Override
        public int calculateSize() {
            return 4;
        }

        final private float DATA;
    }

    /**
     * Stores a boolean property data.
     */
    static public class BoolData extends PropertyData {

        public BoolData(LittleEndianInput input) throws IOException {
            this.DATA = input.readBoolean();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeBoolean(DATA);
        }

        @Override
        public int calculateSize() {
            return 1;
        }

        final private boolean DATA;
    }

}
