/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import java.io.IOException;
import java.util.List;
import restringer.IString;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;

/**
 * Describes a script entry in a VMAD field.
 *
 * @author Mark Fairchild
 * @version 2016/04/23
 */
public class Script implements Entry {

    /**
     * Creates a new Script by reading it from a LittleEndianInput.
     *
     * @param input The LittleEndianInput to readFully.
     * @throws IOException Exceptions are not handled at all.
     */
    public Script(LittleEndianInput input) throws IOException {
        this.NAME = IString.get(input.readUTF());
        this.STATUS = input.readByte();
        this.PROPERTIES = new java.util.LinkedList<>();

        try {
            int propertyCount = input.readUnsignedShort();
            for (int i = 0; i < propertyCount; i++) {
                Property prop = new Property(input);
                this.PROPERTIES.add(prop);
            }
        } catch (IOException ex) {
            throw new IOException("Error reading script " + this.NAME, ex);
        }
    }

    /**
     * Writes the Script.
     *
     * @param output The LittleEndianDataOutput to write.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        output.writeUTF(this.NAME.toString());
        output.writeByte(this.STATUS);
        output.writeShort(this.PROPERTIES.size());

        for (Property prop : this.PROPERTIES) {
            prop.write(output);
        }
    }

    /**
     * @return The calculated size of the Script.
     */
    @Override
    public int calculateSize() {
        int sum = 5 + NAME.length();
        sum += this.PROPERTIES.stream().mapToInt(v -> v.calculateSize()).sum();
        return sum;
    }

    final private IString NAME;
    final private byte STATUS;
    final List<Property> PROPERTIES;

    /**
     * Describes a property entry in a VMAD's scripts.
     *
     * @author Mark Fairchild
     * @version 2016/04/23
     */
    public class Property implements Entry {

        /**
         * Creates a new Property by reading it from a LittleEndianInput.
         *
         * @param input The LittleEndianInput to readFully.
         * @throws IOException Exceptions are not handled at all.
         */
        public Property(LittleEndianInput input) throws IOException {
            this.NAME = IString.get(input.readUTF());
            this.TYPE = input.readByte();
            this.STATUS = input.readByte();

            try {
                this.DATA = PropertyData.readPropertyData(this.TYPE, input);
            } catch (IOException ex) {
                throw new IOException("Error data for reading property " + this.NAME, ex);
            }
        }

        /**
         * @see Entry#write(transposer.LittleEndianDataOutput)
         * @param output The LittleEndianDataOutput.
         * @throws IOException
         */
        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeUTF(this.NAME.toString());
            output.writeByte(this.TYPE);
            output.writeByte(this.STATUS);
            this.DATA.write(output);
        }

        /**
         * @return The calculated size of the Script.
         */
        @Override
        public int calculateSize() {
            int sum = 4 + this.NAME.length() + this.DATA.calculateSize();
            return sum;
        }

        final private IString NAME;
        final private byte TYPE;
        final private byte STATUS;
        final private PropertyData DATA;
    }
}
