/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMaps;
import java.util.Objects;
import java.util.regex.Pattern;

/**
 * A StringTable stores reads and stores strings from the mod stringtables;
 * mostly just applies to Skyrim.esm and the DLCs.
 *
 * @author Mark Fairchild
 * @version 2016/09/08
 */
public class StringTable {

    public StringTable() {
        this.TABLE = Int2ObjectMaps.synchronize(new Int2ObjectOpenHashMap<>());
    }

    /**
     *
     * @param file
     * @param index
     */
    public void populateFromFile(StringsFile file, int index) {
        Objects.requireNonNull(file);

        Int2ObjectMap<String> temp = new Int2ObjectOpenHashMap<>(file.getMap().size());
        
        file.getMap().forEach((id, str) -> {
            int newID = (index<<24) | (id & 0xFFFFFF);
            temp.put(newID, str);
        });
        
        this.TABLE.putAll(temp);
    }

    /**
     * Retrieves the whole map.
     *
     * @return
     */
    public Int2ObjectMap<String> getMap() {
        return Int2ObjectMaps.unmodifiable(this.TABLE);
    }

    /**
     * Retrieves a string using its formid.
     *
     * @param formID
     * @return
     */
    public String get(int formID) {
        return this.TABLE.get(formID);
    }

    /**
     * The reference for accessing the stringtable.
     */
    final private it.unimi.dsi.fastutil.ints.Int2ObjectMap<String> TABLE;

    static private enum Type {
        STRINGS(Pattern.compile(".+\\.STRINGS$", Pattern.CASE_INSENSITIVE)),
        ILSTRINGS(Pattern.compile(".+\\.ILSTRINGS$", Pattern.CASE_INSENSITIVE)),
        DLSTRINGS(Pattern.compile(".+\\.DLSTRINGS$", Pattern.CASE_INSENSITIVE));

        static Type match(String filename) {
            if (STRINGS.REGEX.asPredicate().test(filename)) {
                return STRINGS;
            }
            if (ILSTRINGS.REGEX.asPredicate().test(filename)) {
                return ILSTRINGS;
            }
            if (DLSTRINGS.REGEX.asPredicate().test(filename)) {
                return DLSTRINGS;
            }
            return null;
        }

        private Type(java.util.regex.Pattern regex) {
            this.REGEX = Objects.requireNonNull(regex);
        }

        final public java.util.regex.Pattern REGEX;
    };

}
