/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.esp;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMaps;
import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;
import java.util.regex.Pattern;
import restringer.LittleEndianInput;

/**
 * A StringsFile stores reads and stores strings from the mod stringtables;
 * mostly just applies to Skyrim.esm and the DLCs.
 *
 * @author Mark Fairchild
 * @version 2016/09/11
 */
public class StringsFile {

    /**
     * Reads a <code>StringsFile</code> from a <code>LittleEndianInput</code>.
     * @param name The name of the stringtable.
     * @param input The input stream.
     * @param type The type of stringtable.
     * @throws IOException 
     */
    public StringsFile(String name, LittleEndianInput input, Type type) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(type);
        
        this.NAME = name;
        final int COUNT = input.readInt();
        final int DATASIZE = input.readInt();
        final long[] OFFSETS = new long[COUNT];

        for (int i = 0; i < COUNT; i++) {
            final int ID = input.readInt();
            final int FORMID = ID;
            final int OFFSET = input.readInt();
            final long CODE = ((long) OFFSET << 32) | FORMID;
            OFFSETS[i] = CODE;
        }

        Arrays.sort(OFFSETS);

        this.TABLE = new Int2ObjectOpenHashMap<>(COUNT);

        final long DATASECTION = 8 + COUNT * 8;
        long filePointer = DATASECTION;
        int prevOffset = -1;
        String str = null;

        for (int i = 0; i < OFFSETS.length; i++) {
            long code = OFFSETS[i];
            int formID = (int) code;
            int currOffset = (int) (code >>> 32);

            if (currOffset == prevOffset) {
                this.TABLE.put(formID, str);

            } else {
                assert filePointer == DATASECTION + currOffset : String.format("Error reading string %d at position %d.", formID, filePointer);

                try {
                    if (type == Type.STRINGS) {
                        str = input.readZString();
                        filePointer += str.length() + 1;
                    } else {
                        str = input.readLZString();
                        filePointer += str.length() + 5;
                    }

                    this.TABLE.put(formID, str);
                    prevOffset = currOffset;

                } catch (IOException ex) {
                    throw new IOException(String.format("Error reading string %d at position %d.", formID, filePointer));
                }
            }
        }
        
        this.LIST = new java.util.ArrayList<>(this.TABLE.size());
        this.TABLE.forEach((id, s) -> {
            restringer.Pair<Integer, String> p = new restringer.Pair<>(id, s);
            this.LIST.add(p);
        });
        java.util.Collections.sort(this.LIST, (p1,p2)-> p1.A.compareTo(p2.A));
    }

    /**
     * Retrieves the whole map.
     *
     * @return
     */
    public Int2ObjectMap<String> getMap() {
        return Int2ObjectMaps.unmodifiable(this.TABLE);
    }

    /**
     * Retrieves a string using its formid.
     *
     * @param formID
     * @return
     */
    public String get(int formID) {
        return this.TABLE.get(formID);
    }

    /**
     * @see Object#toString() 
     * @return 
     */
    @Override public String toString() {    
        return this.NAME;
    }
    
    /**
     * The reference for accessing the stringtable.
     */
    final private it.unimi.dsi.fastutil.ints.Int2ObjectMap<String> TABLE;
    
    final private java.util.ArrayList<restringer.Pair<Integer, String>> LIST;

    /**
     * The name of the stringtable.
     */
    final private String NAME;
    
    /**
     * The three different types of Strings file.
     */
    static public enum Type {
        STRINGS(Pattern.compile(".+\\.STRINGS$", Pattern.CASE_INSENSITIVE)),
        ILSTRINGS(Pattern.compile(".+\\.ILSTRINGS$", Pattern.CASE_INSENSITIVE)),
        DLSTRINGS(Pattern.compile(".+\\.DLSTRINGS$", Pattern.CASE_INSENSITIVE));

        static public Type match(String filename) {
            if (STRINGS.REGEX.asPredicate().test(filename)) {
                return STRINGS;
            }
            if (ILSTRINGS.REGEX.asPredicate().test(filename)) {
                return ILSTRINGS;
            }
            if (DLSTRINGS.REGEX.asPredicate().test(filename)) {
                return DLSTRINGS;
            }
            return null;
        }

        private Type(java.util.regex.Pattern regex) {
            this.REGEX = Objects.requireNonNull(regex);
        }

        final private java.util.regex.Pattern REGEX;
    };

}
