/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.Profile;
import restringer.ess.papyrus.PapyrusElement;

/**
 * Describes a default ChangeForm.
 *
 * @author Mark Fairchild
 * @version 2016/07/16
 */
final public class ChangeFormDefault extends ChangeFormData {

    /**
     * Creates a new <code>ChangeFormDefault</code> by storing a data buffer.
     *
     * @param buf The data buffer.
     */
    public ChangeFormDefault(byte[] buf) {
        Objects.requireNonNull(buf);
        this.BUFFER = buf;
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.write(this.BUFFER);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return this.BUFFER.length;
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();

        BUILDER.append("<hr/><p>RAW DATA:</p><code><pre>");

        for (int i = 0; i < this.BUFFER.length; i++) {
            if (i > 0 && i % 16 == 0) {
                BUILDER.append('\n');
            }
            
            final byte B = this.BUFFER[i];
            BUILDER.append(String.format("%02x ", B));
        }

        BUILDER.append("</pre></code>");
        return BUILDER.toString();
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return ""; //(" + this.BUFFER.length + " bytes)";
    }
    
    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        return false;
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        
    }
    
    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
    }

    /**
     * @see Object#hashCode() 
     * @return 
     */
    @Override
    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + java.util.Arrays.hashCode(this.BUFFER);
        return hash;
    }

    /**
     * @see Object#equals(java.lang.Object) 
     * @return 
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ChangeFormDefault other = (ChangeFormDefault) obj;
        return java.util.Arrays.equals(this.BUFFER, other.BUFFER);
    }

    final private byte[] BUFFER;

}
