/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;
import restringer.LittleEndianInputStream;
import restringer.Profile;
import restringer.ess.papyrus.PapyrusElement;

/**
 * Describes a ChangeForm containing a formlist.
 *
 * @author Mark Fairchild
 * @version 2016/07/16
 */
final public class ChangeFormFLST extends ChangeFormData {

    /**
     * Creates a new <code>ChangeForm</code> by reading from a byte array. No
     * error handling is performed.
     *
     * @param buf The data buffer.
     * @param flags The change form flags.
     * @throws IOException
     */
    public ChangeFormFLST(byte[] buf, Flags.Int flags) throws IOException {
        this(LittleEndianInputStream.wrap(buf), flags);
    }

    /**
     * Creates a new <code>ChangeForm</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param flags The change form flags.
     * @throws IOException
     */
    private ChangeFormFLST(LittleEndianInput input, Flags.Int flags) throws IOException {
        Objects.requireNonNull(input);

        if (flags.getFlag(1)) {
            this.FLAGS = new ChangeFormFlags(input);
        } else {
            this.FLAGS = null;
        }

        if (flags.getFlag(31)) {
            int formCount = input.readInt();
            this.FORMS = new ArrayList<>(formCount);

            for (int i = 0; i < formCount; i++) {
                final RefID REF = new RefID(input);
                this.FORMS.add(REF);
            }
        } else {
            this.FORMS = null;
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);

        if (null != this.FLAGS) {
            this.FLAGS.write(output);
        }

        if (null != this.FORMS) {
            output.writeInt(this.FORMS.size());
            for (RefID ref : this.FORMS) {
                ref.write(output);
            }
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 0;

        if (null != this.FLAGS) {
            sum += this.FLAGS.calculateSize();
        }

        if (null != this.FORMS) {
            sum += 4;
            sum += this.FORMS.stream().mapToInt(v -> v.calculateSize()).sum();
        }

        return sum;
    }

    /**
     * @return The <code>ChangeFormFlags</code> field.
     */
    public ChangeFormFlags getRefID() {
        return this.FLAGS;
    }

    /**
     * Removes null entries.
     *
     * @return The number of entries removed.
     */
    public int cleanse() {
        if (null == this.FORMS) {
            return 0;
        }

        int size = this.FORMS.size();
        this.FORMS.removeIf(v -> v.equals(RefID.NONE));
        return size - this.FORMS.size();
    }

    /**
     * @return A flag indicating that the formlist has nullref entries.
     */
    public boolean containsNullrefs() {
        return this.FORMS.contains(RefID.NONE);
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        if (null == this.FORMS) {
            return "";

        } else if (this.containsNullrefs()) {
            return "(" + this.FORMS.size() + " refs, contains nullrefs)";

        } else {
            return "(" + this.FORMS.size() + " refs)";
        }
    }

    /**
     * @see Object#hashCode()
     * @return
     */
    @Override
    public int hashCode() {
        int hash = 7;
        hash = 41 * hash + Objects.hashCode(this.FLAGS);
        hash = 41 * hash + Objects.hashCode(this.FORMS);
        return hash;
    }

    /**
     * @see Object#equals()
     * @return
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        } else if (obj == null) {
            return false;
        } else if (getClass() != obj.getClass()) {
            return false;
        }

        final ChangeFormFLST other = (ChangeFormFLST) obj;
        return Objects.equals(this.FLAGS, other.FLAGS) && Objects.equals(this.FORMS, other.FORMS);
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();

        BUILDER.append("<hr/><p>FORMLIST:</p>");

        if (null != this.FLAGS) {
            BUILDER.append(String.format("<p>ChangeFormFlags: %s</p>", this.FLAGS));
        }

        if (null != this.FORMS) {
            BUILDER.append(String.format("<p>List size: %d</p><ol start=0>", this.FORMS.size()));
            this.FORMS.forEach(refid -> {
                if (save.getChangeForms().containsKey(refid)) {
                    BUILDER.append(String.format("<li>%s", refid.toHTML()));
                } else {
                    BUILDER.append(String.format("<li>%s", refid));
                }
            });
            BUILDER.append("</ol>");
        }

        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        return false;
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        this.FORMS.forEach(v -> v.addName(names, strings));
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.FORMS.forEach(v -> v.resolveRefs(ess, owner));
    }

    final private ChangeFormFlags FLAGS;
    final private List<RefID> FORMS;

}
