/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;

/**
 * Describes the ChangeForm flags for a ChangeForm.
 *
 * @author Mark Fairchild
 * @version 2016/07/16
 */
final public class ChangeFormFlags implements Element {

    /**
     * Creates a new <code>ChangeFormFlags</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @throws IOException
     */
    public ChangeFormFlags(LittleEndianInput input) throws IOException {
        Objects.requireNonNull(input);
        this.FLAG = input.readInt();
        this.UNKNOWN = input.readShort();
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.writeInt(this.FLAG);
        output.writeShort(this.UNKNOWN);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return 6;
    }

    /**
     * @return The flag field.
     */
    public int getFlag() {
        return this.FLAG;
    }

    /**
     * @return The unknown field.
     */
    public short getUnknown() {
        return this.UNKNOWN;
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        final StringBuilder BUF = new StringBuilder();
        final String S = Integer.toUnsignedString(this.FLAG, 2);

        int idx = 0;

        while (idx < 32 - S.length()) {
            if (idx > 0 && idx % 4 == 0) {
                BUF.append(' ');
            }

            BUF.append('0');
            idx++;
        }

        while (idx < 32) {
            if (idx > 0 && idx % 4 == 0) {
                BUF.append(' ');
            }
            
            BUF.append(S.charAt(idx - 32 + S.length()));
        }

        return BUF.toString();
    }

    final private int FLAG;
    final private short UNKNOWN;

}
