/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;
import restringer.LittleEndianInputStream;
import restringer.Profile;

/**
 * Describes a ChangeForm containing an NPC.
 *
 * @author Mark Fairchild
 * @version 2016/09/02
 *
 * @author Mark Fairchild
 */
public class ChangeFormNPC extends ChangeFormData {

    /**
     * Creates a new <code>ChangeForm</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param buf The data buffer.
     * @param flags The change form flags.
     * @throws IOException
     */
    public ChangeFormNPC(byte[] buf, Flags.Int flags) throws IOException {
        this.BUFFER = Objects.requireNonNull(buf);
        LittleEndianInput input = LittleEndianInputStream.wrap(buf);

        if (flags.getFlag(0)) {
            this.FLAGS = new ChangeFormFlags(input);
        } else {
            this.FLAGS = null;
        }

        if (flags.getFlag(1)) {
            this.ACBS = new byte[24];
            input.read(this.ACBS);
        } else {
            this.ACBS = null;
        }

        if (flags.getFlag(6)) {
            this.NUMFACTIONRANKS = new VSVal(input);

            this.FACTIONRANKS = new ArrayList<>();
            for (int i = 0; i < this.NUMFACTIONRANKS.getValue(); i++) {
                FactionRank rank = new FactionRank(input);
                this.FACTIONRANKS.add(rank);
            }
        } else {
            this.NUMFACTIONRANKS = null;
            this.FACTIONRANKS = null;
        }

        if (flags.getFlag(4)) {
            this.SPELLCOUNT = new VSVal(input);
            this.SPELLS = new ArrayList<>();
            for (int i = 0; i < this.SPELLCOUNT.getValue(); i++) {
                RefID ref = new RefID(input);
                this.SPELLS.add(ref);
            }

            this.LEVELLEDSPELLCOUNT = new VSVal(input);
            this.LEVELLEDSPELLS = new ArrayList<>();
            for (int i = 0; i < this.LEVELLEDSPELLCOUNT.getValue(); i++) {
                RefID ref = new RefID(input);
                this.SPELLS.add(ref);
            }

            this.SHOUTCOUNT = new VSVal(input);
            this.SHOUTS = new ArrayList<>();
            for (int i = 0; i < this.SHOUTCOUNT.getValue(); i++) {
                RefID ref = new RefID(input);
                this.SHOUTS.add(ref);
            }

        } else {
            this.SPELLCOUNT = null;
            this.LEVELLEDSPELLCOUNT = null;
            this.SHOUTCOUNT = null;
            this.SPELLS = null;
            this.LEVELLEDSPELLS = null;
            this.SHOUTS = null;
        }

        if (flags.getFlag(3)) {
            this.AIDT = new byte[20];
            input.read(this.AIDT);
        } else {
            this.AIDT = null;
        }

        if (flags.getFlag(5)) {
            this.FULLNAME = WString.read(input);
        } else {
            this.FULLNAME = null;
        }

        if (flags.getFlag(9)) {
            this.DNAM = new byte[52];
            input.read(this.DNAM);
        } else {
            this.DNAM = null;
        }

        if (flags.getFlag(10)) {
            this.CCLASS = new RefID(input);
        } else {
            this.CCLASS = null;
        }

        if (flags.getFlag(25)) {
            this.RACE = new RefID(input);
            this.OLDRACE = new RefID(input);
        } else {
            this.RACE = null;
            this.OLDRACE = null;
        }

        if (flags.getFlag(11)) {
            this.FACE = new FaceData(input);
        } else {
            this.FACE = null;
        }

        if (flags.getFlag(24)) {
            this.GENDER = input.readByte();
        } else {
            this.GENDER = 0;
        }

        if (flags.getFlag(12)) {
            this.DEFOUTFIT = new RefID(input);
        } else {
            this.DEFOUTFIT = null;
        }

        if (flags.getFlag(13)) {
            this.SLEEPOUTFIT = new RefID(input);
        } else {
            this.SLEEPOUTFIT = null;
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.write(this.BUFFER);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return this.BUFFER.length;
    }

    /**
     * @return The <code>ChangeFormFlags</code> field.
     */
    public ChangeFormFlags getRefID() {
        return this.FLAGS;
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        if (null != this.FULLNAME) {
            return ": " + this.FULLNAME;
        } else {
            return "";
        }
    }

    /**
     * @see Object#hashCode() 
     * @return 
     */
    @Override
    public int hashCode() {
        int hash = 7;
        hash = 41 * hash + Arrays.hashCode(this.BUFFER);
        return hash;
    }

    /**
     * @see Object#equals() 
     * @return 
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        } else if (obj == null) {
            return false;
        } else if (getClass() != obj.getClass()) {
            return false;
        }
        
        final ChangeFormNPC other = (ChangeFormNPC) obj;
        return Arrays.equals(this.BUFFER, other.BUFFER);
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();

        BUILDER.append("<hr/><p>NPC:</p>");

        if (null != this.FULLNAME) {
            BUILDER.append(String.format("<p>FullName: %s\n", this.FULLNAME));
        }

        if (null != this.FLAGS) {
            BUILDER.append(String.format("<p>ChangeFormFlags: %s\n", this.FLAGS));
        }

        if (null != this.ACBS) {
            BUILDER.append("<p>Base stats: ");
            for (byte b : this.ACBS) {
                BUILDER.append(String.format("%02x", b));
            }
            BUILDER.append("\n");
        }

        if (null != this.NUMFACTIONRANKS) {
            BUILDER.append(String.format("<p>%s faction ranks.</p><ul>", this.NUMFACTIONRANKS));

            this.FACTIONRANKS.forEach(v -> BUILDER.append(String.format("<li>%s", v)));
            BUILDER.append("</ul>");
        }

        if (null != this.SPELLCOUNT) {
            BUILDER.append(String.format("<p>%s spells.</p><ul>", this.SPELLCOUNT));
            this.SPELLS.forEach(v -> BUILDER.append(String.format("<li>%s", v)));
            BUILDER.append("</ul>");

            BUILDER.append(String.format("<p>%s levelled spells.</p><ul>", this.LEVELLEDSPELLCOUNT));
            this.LEVELLEDSPELLS.forEach(v -> BUILDER.append(String.format("<li>%s", v)));
            BUILDER.append("</ul>");

            BUILDER.append(String.format("<p>%s shouts.</p><ul>", this.SHOUTCOUNT));
            this.SHOUTS.forEach(v -> BUILDER.append(String.format("<li>%s", v)));
            BUILDER.append("</ul>");
        }

        if (null != this.AIDT) {
            BUILDER.append("<p>AI:</p><code>");
            for (byte b : this.AIDT) {
                BUILDER.append(String.format("%02x", b));
            }
            BUILDER.append("</code>");
        }

        if (null != this.DNAM) {
            BUILDER.append("<p>Skills:</p><code>");
            for (byte b : this.DNAM) {
                BUILDER.append(String.format("%02x", b));
            }
            BUILDER.append("</code>");
        }

        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        return false;
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        if (null != this.FACTIONRANKS) {
            this.FACTIONRANKS.forEach(v -> v.FACTION.addName(names, strings));
        }

        if (null != this.SPELLS) {
            this.SPELLS.forEach(v -> v.addName(names, strings));
            this.LEVELLEDSPELLS.forEach(v -> v.addName(names, strings));
            this.SHOUTS.forEach(v -> v.addName(names, strings));
        }
        
        if (null != CCLASS) {
            this.CCLASS.addName(names, strings);
        }
        
        if (null != RACE) {
            this.RACE.addName(names, strings);
        }
        
        if (null != OLDRACE) {
            this.OLDRACE.addName(names, strings);
        }
        
        if (null != DEFOUTFIT) {
            this.DEFOUTFIT.addName(names, strings);
        }
        
        if (null != SLEEPOUTFIT) {
            this.SLEEPOUTFIT.addName(names, strings);
        }
        
        
    }
    
    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        if (null != this.FACTIONRANKS) {
            this.FACTIONRANKS.forEach(v -> v.FACTION.resolveRefs(ess, owner));
        }

        if (null != this.SPELLS) {
            this.SPELLS.forEach(v -> v.resolveRefs(ess, owner));
            this.LEVELLEDSPELLS.forEach(v -> v.resolveRefs(ess, owner));
            this.SHOUTS.forEach(v -> v.resolveRefs(ess, owner));
        }
        
        if (null != CCLASS) {
            this.CCLASS.resolveRefs(ess, owner);
        }
        
        if (null != RACE) {
            this.RACE.resolveRefs(ess, owner);
        }
        
        if (null != OLDRACE) {
            this.OLDRACE.resolveRefs(ess, owner);
        }
        
        if (null != DEFOUTFIT) {
            this.DEFOUTFIT.resolveRefs(ess, owner);
        }
        
        if (null != SLEEPOUTFIT) {
            this.SLEEPOUTFIT.resolveRefs(ess, owner);
        }
        
    }

    final private byte[] BUFFER;
    final private ChangeFormFlags FLAGS;
    final private byte[] ACBS, AIDT, DNAM;
    final private VSVal NUMFACTIONRANKS;
    final private ArrayList<FactionRank> FACTIONRANKS;
    final private VSVal SPELLCOUNT, LEVELLEDSPELLCOUNT, SHOUTCOUNT;
    final private ArrayList<RefID> SPELLS, LEVELLEDSPELLS, SHOUTS;
    final private WString FULLNAME;
    final private RefID CCLASS, RACE, OLDRACE, DEFOUTFIT, SLEEPOUTFIT;
    final private byte GENDER;
    final private FaceData FACE;

    /**
     * Faction rank.
     */
    static private class FactionRank implements Element {

        public FactionRank(LittleEndianInput input) throws IOException {
            this.FACTION = new RefID(input);
            this.RANK = input.readByte();
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            this.FACTION.write(output);
            output.writeByte(RANK);
        }

        @Override
        public int calculateSize() {
            return 1 + this.FACTION.calculateSize();
        }

        @Override
        public String toString() {
            return String.format("Rank %d with %s", this.RANK, this.FACTION);
        }
        final private RefID FACTION;
        final private byte RANK;
    }

    /**
     * Face
     */
    static private class FaceData implements Element {

        public FaceData(LittleEndianInput input) throws IOException {
            this.FACEPRESENT = input.readBoolean();

            if (!this.FACEPRESENT) {
                this.HAIRCOLOR = null;
                this.SKINTONE = 0;
                this.SKIN = null;
                this.HEADPARTCOUNT = null;
                this.HEADPARTS = null;
                this.FACEDATAPRESENT = false;
                this.MORPHSCOUNT = 0;
                this.MORPHS = null;
                this.PRESETSCOUNT = 0;
                this.PRESETS = null;

            } else {
                this.HAIRCOLOR = new RefID(input);
                this.SKINTONE = input.readInt();
                this.SKIN = new RefID(input);
                this.HEADPARTCOUNT = new VSVal(input);
                this.HEADPARTS = new ArrayList<>();
                for (int i = 0; i < HEADPARTCOUNT.getValue(); i++) {
                    RefID headpart = new RefID(input);
                    this.HEADPARTS.add(headpart);
                }

                this.FACEDATAPRESENT = input.readBoolean();

                if (!this.FACEDATAPRESENT) {
                    this.MORPHSCOUNT = 0;
                    this.MORPHS = null;
                    this.PRESETSCOUNT = 0;
                    this.PRESETS = null;
                } else {
                    this.MORPHSCOUNT = input.readInt();
                    this.MORPHS = new float[this.MORPHSCOUNT];
                    for (int i = 0; i < this.MORPHSCOUNT; i++) {
                        this.MORPHS[i] = input.readFloat();
                    }
                    this.PRESETSCOUNT = input.readInt();
                    this.PRESETS = new int[this.PRESETSCOUNT];
                    for (int i = 0; i < PRESETSCOUNT; i++) {
                        this.PRESETS[i] = input.readInt();
                    }

                }
            }
        }

        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            output.writeBoolean(this.FACEPRESENT);

            if (this.FACEPRESENT) {
                this.HAIRCOLOR.write(output);
                output.writeInt(this.SKINTONE);
                this.SKIN.write(output);
                this.HEADPARTCOUNT.write(output);
                for (RefID headpart : this.HEADPARTS) {
                    headpart.write(output);
                }
                output.writeBoolean(this.FACEDATAPRESENT);

                if (this.FACEDATAPRESENT) {
                    output.writeInt(this.MORPHSCOUNT);
                    for (int i = 0; i < this.MORPHSCOUNT; i++) {
                        output.writeFloat(this.MORPHS[i]);
                    }
                    output.writeInt(this.PRESETSCOUNT);
                    for (int i = 0; i < PRESETSCOUNT; i++) {
                        output.writeInt(this.PRESETS[i]);
                    }
                }
            }
        }

        @Override
        public int calculateSize() {
            int sum = 1;

            if (this.FACEPRESENT) {
                sum += this.HAIRCOLOR.calculateSize();
                sum += 4;
                sum += this.SKIN.calculateSize();
                sum += this.HEADPARTCOUNT.calculateSize();
                sum += this.HEADPARTS.stream().mapToInt(v -> v.calculateSize()).sum();
                sum += 1;

                if (this.FACEDATAPRESENT) {
                    sum += 8;
                    sum += 4 * this.MORPHSCOUNT;
                    sum += 4 * this.PRESETSCOUNT;
                }
            }
            return sum;
        }

        final private boolean FACEPRESENT;
        final private RefID HAIRCOLOR;
        final private int SKINTONE;
        final private RefID SKIN;
        final private VSVal HEADPARTCOUNT;
        final private ArrayList<RefID> HEADPARTS;
        final private boolean FACEDATAPRESENT;

        final private int MORPHSCOUNT;
        final private float[] MORPHS;
        final private int PRESETSCOUNT;
        final private int[] PRESETS;
    }
}
