/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import net.jpountz.lz4.*;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;
import restringer.LittleEndianInputStream;
import restringer.Timer;
import restringer.ess.papyrus.EID;
import restringer.ess.papyrus.Papyrus;
import restringer.ess.papyrus.PapyrusElement;
import restringer.ess.papyrus.ScriptInstance;
import restringer.gui.FilterTreeModel;
import restringer.gui.FilterTreeModel.Node;
import restringer.gui.ProgressModel;

/**
 * Describes a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 *
 */
final public class ESS implements Element {

    /**
     * Reads a savegame and creates an <code>ESS</code> object to represent it.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param saveFile The file containing the savegame.
     * @param model A progress model, for displaying a progressbar.
     * @return The <code>ESS</code> object.
     * @throws IOException
     *
     */
    static public ESS readESS(File saveFile, ProgressModel model) throws IOException {
        Objects.requireNonNull(saveFile);
        model = (model == null ? new ProgressModel() : model);

        model.setValueIsAdjusting(true);
        model.setMinimum(0);
        model.setMaximum(50682);
        model.setValue(0);
        model.setValueIsAdjusting(false);

        String skseName = saveFile.getName().replaceAll("ess$", "skse");
        File skseFile = new File(saveFile.getParent(), skseName);

        try {
            if (skseFile.exists()) {
                try (LittleEndianInputStream input = LittleEndianInputStream.openD(saveFile)) {
                    byte[] skse = Files.readAllBytes(skseFile.toPath());
                    return new ESS(input, saveFile.getName(), skse, model);
                }
            } else {
                try (LittleEndianInputStream input = LittleEndianInputStream.openD(saveFile)) {
                    return new ESS(input, saveFile.getName(), null, model);
                }
            }
        } finally {
            prevSaveFile = saveFile;
        }
    }

    /**
     * Writes out a savegame.
     *
     * Exceptions are not handled. At all. Not even a little bit.
     *
     * @param ess The <code>ESS</code> object.
     * @param saveFile The file into which to write the savegame.
     * @param model A progress model, for displaying a progressbar.
     * @throws IOException
     *
     */
    static public void writeESS(ESS ess, File saveFile, ProgressModel model) throws IOException {
        Objects.requireNonNull(ess);
        Objects.requireNonNull(saveFile);
        Objects.requireNonNull(model);

        model.setValueIsAdjusting(true);
        model.setMinimum(0);
        model.setMaximum(50682);
        model.setValue(0);
        model.setValueIsAdjusting(false);

        boolean testMode = false;
        String skseName = saveFile.getName().replaceAll("ess$", "skse");
        File skseFile = new File(saveFile.getParent(), skseName);

        if (saveFile.exists()) {
            backupFile(saveFile);
        }
        if (skseFile.exists()) {
            backupFile(skseFile);
        }

        if (ess.SKSE != null) {
            try (LittleEndianDataOutput output = LittleEndianDataOutput.open(saveFile)) {
                Files.write(skseFile.toPath(), ess.SKSE);
                ess.rangeModel = model;
                ess.write(output);
            }

        } else {
            try (LittleEndianDataOutput output = LittleEndianDataOutput.open(saveFile)) {
                ess.rangeModel = model;
                ess.write(output);
            }
        }

        if (testMode) {
            compareESS(prevSaveFile, saveFile);
            System.out.println("STILL IN TESTING MODE");
            assert false : "Write was successfull, but we are still in testing mode!!!";
        }

    }

    /**
     * Creates a new <code>ESS</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream for the savegame.
     * @param filename The filename of the <code>ESS</code>.
     * @param skse The skse co-save.
     * @param model A progress model, for displaying a progressbar.
     * @throws IOException
     */
    private ESS(LittleEndianInputStream input, String filename, byte[] skse, ProgressModel model) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(model);
        this.rangeModel = Objects.requireNonNull(model);

        int i = -1;
        this.TIMER = new Timer("Timer");
        TIMER.start();

        LOG.info("Reading savegame.");

        // Read the header. This includes the magic string.
        this.HEADER = new Header(input);
        Game game = this.HEADER.GAME;
        LOG.fine("Reading savegame: read header.");

        // This is the stream that will be used for the remainder of the 
        // constructor.
        final LittleEndianInputStream INPUT;

        // SkyrimSE uses compression, so enable that.
        if (game.isSSE()) {
            final int ORIGINAL_LEN = input.readInt();
            final int COMPRESSED_LEN = input.readInt();
            final byte[] ORIGINAL = new byte[ORIGINAL_LEN];
            final byte[] COMPRESSED = new byte[COMPRESSED_LEN];
            input.read(COMPRESSED);

            final LZ4Factory LZ4FACTORY = LZ4Factory.safeInstance();
            final LZ4SafeDecompressor LZ4DECOMP = LZ4FACTORY.safeDecompressor();
            LZ4DECOMP.decompress(COMPRESSED, ORIGINAL);
            INPUT = LittleEndianInputStream.wrap(ORIGINAL);
            
        } else {
            INPUT = input;
        }

        // Read the form version.
        this.FORMVERSION = INPUT.readByte();
        switch (game) {
            case SKYRIM:
                assert this.FORMVERSION == 74 || this.FORMVERSION == 73 : "Invalid formVersion: " + this.FORMVERSION;
                this.UNKNOWN_STRING = null;
                break;
            case SKYRIMSE:
                assert this.FORMVERSION == 77 : "Invalid formVersion: " + this.FORMVERSION;
                this.UNKNOWN_STRING = null;
                break;
            case FALLOUT4:
                assert this.FORMVERSION == 67 : "Invalid formVersion: " + this.FORMVERSION;
                this.UNKNOWN_STRING = INPUT.readWString();
                System.out.println(this.UNKNOWN_STRING);
                break;
            default:
                throw new IllegalArgumentException("Unrecognized game.");
        }

        // Make the ESSContext.
        final ESSContext CTX = new ESSContext(this.HEADER, this.FORMVERSION);

        // Read the plugin info section.
        int pluginInfoSize = INPUT.readInt();
        this.PLUGINS = new PluginInfo(INPUT);
        assert pluginInfoSize == this.PLUGINS.calculateSize();
        LOG.fine("Reading savegame: read plugin table.");

        // Read the file location table.
        this.FLT = new FileLocationTable(INPUT);
        this.TABLE1 = new ArrayList<>(this.FLT.table1Count);
        this.TABLE2 = new ArrayList<>(this.FLT.table2Count);
        this.TABLE3 = new ArrayList<>(this.FLT.table3Count);
        this.CHANGEFORMS = new ArrayList<>(this.FLT.changeFormCount);
        this.CHANGEFORMS_MAP = new LinkedHashMap<>(this.FLT.changeFormCount);
        LOG.fine("Reading savegame: read file location table.");

        // Read the first set of data tables.
        try {
            for (i = 0; i < this.FLT.table1Count; i++) {
                GlobalData DATA = new GlobalData(INPUT, CTX);
                assert 0 <= DATA.getType() && DATA.getType() <= 100 : "Invalid type for Table1: " + DATA.getType();
                this.TABLE1.add(DATA);
                LOG.log(Level.FINE, "Reading savegame: \tGlobalData type {0}.", DATA.getType());
                LOG.fine("Reading savegame: read global data table 1.");
            }
            LOG.fine("Reading savegame: read GlobalDataTable #1.");
        } catch (IOException ex) {
            throw new IOException(String.format("Error; read %d/%d GlobalData from table #1.", i, this.FLT.table1Count), ex);
        }

        // Read the second set of data tables.
        try {
            for (i = 0; i < this.FLT.table2Count; i++) {
                GlobalData DATA = new GlobalData(INPUT, CTX);
                assert 100 <= DATA.getType() && DATA.getType() < 1000 : "Invalid type for Table2: " + DATA.getType();
                this.TABLE2.add(DATA);
                LOG.log(Level.FINE, "Reading savegame: \tGlobalData type {0}.", DATA.getType());
            }
            LOG.fine("Reading savegame: read GlobalDataTable #2.");
        } catch (IOException ex) {
            throw new IOException(String.format("Error; read %d/%d GlobalData from table #2.", i, this.FLT.table2Count), ex);
        }

        this.rangeModel.setValue(22);

        // Read the changeforms.
        try {
            for (i = 0; i < this.FLT.changeFormCount; i++) {
                ChangeForm FORM = new ChangeForm(INPUT, CTX);
                this.CHANGEFORMS.add(FORM);
                this.CHANGEFORMS_MAP.put(FORM.getRefID(), FORM);
            }
            LOG.fine("Reading savegame: read changeform table.");
        } catch (IOException ex) {
            throw new IOException(String.format("Error; read %d/%d ChangeForm definitions.", i, this.FLT.changeFormCount), ex);
        }

        this.rangeModel.setValue(820);

        // Read the third set of data tables.
        try {
            for (i = 0; i < this.FLT.table3Count; i++) {
                GlobalData DATA = new GlobalData(INPUT, CTX);
                assert 1000 <= DATA.getType() && DATA.getType() <= 10000 : "Invalid type for Table3: " + DATA.getType();
                this.TABLE3.add(DATA);

                if (DATA.getType() == 1001) {
                    Papyrus p = DATA.getPapyrus();
                    System.out.print(p);
                }

                LOG.log(Level.FINE, "Reading savegame: \tGlobalData type {0}.", DATA.getType());
            }
            LOG.fine("Reading savegame: read GlobalDataTable #3.");
        } catch (Exception | Error ex) {
            throw new IOException(String.format("Error; read %d/%d GlobalData from table #3.", i, this.FLT.table3Count), ex);
        }

        this.rangeModel.setValue(4207);

        Papyrus p = this.TABLE3.get(1).getPapyrus();

        int formIDCount = INPUT.readInt();
        this.FORMIDARRAY = new int[formIDCount];
        for (i = 0; i < formIDCount; i++) {
            this.FORMIDARRAY[i] = INPUT.readInt();
        }
        LOG.fine("Reading savegame: read formid array.");

        int worldspaceIDCount = INPUT.readInt();
        this.VISITEDWORLDSPACEARRAY = new int[worldspaceIDCount];
        for (i = 0; i < worldspaceIDCount; i++) {
            this.VISITEDWORLDSPACEARRAY[i] = INPUT.readInt();
        }
        LOG.fine("Reading savegame: read visited worldspace array.");

        final byte[] BUF = new byte[8192];
        final ByteArrayOutputStream BAOS = new ByteArrayOutputStream();

        while (INPUT.available() > 0) {
            int read = INPUT.read(BUF);
            BAOS.write(BUF, 0, read);
        }

        this.UNKNOWN3 = BAOS.toByteArray();
        LOG.fine("Reading savegame: read unknown block.");

        this.rangeModel.setValue(4326);

        assert this.TABLE3.get(1).getType() == 1001;
        this.PAPYRUS = this.TABLE3.get(1).getPapyrus();
        LOG.fine("Papyrus block is ready.");

        this.resolveRefs();
        LOG.fine("Reading savegame: resolved Papyrus references.");

        this.rangeModel.setValue(15349);

        if (null == skse) {
            this.SKSE = null;
        } else {
            this.SKSE = skse;
        }

        float size = this.calculateSize() / 1048576.0f;

        this.FILENAME = filename;
        this.DIGEST = input.getDigest();

        TIMER.stop();
        LOG.info(String.format("Savegame read: %.1f mb in %s.", size, this.TIMER.getFormattedTime()));
    }

    /**
     * Writes the object to a <code>LittleEndianDataOutput</code>. No error
     * handling is performed.
     *
     * @param output The output stream for the savegame.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        Objects.requireNonNull(rangeModel);
        LOG.info("Writing savegame.");
        TIMER.restart();

        Game game = this.HEADER.GAME;

        // Write the header. This includes the magic string.
        this.HEADER.write(output);
        rangeModel.setValue(rangeModel.getValue() + 16 + this.HEADER.calculateSize());
        LOG.fine("Writing savegame: wrote header.");

        // SkyrimSE uses compression, so enable that.
        if (game == Game.SKYRIMSE) {
            final ByteArrayOutputStream BAOS = new ByteArrayOutputStream(this.calculateSize() / 2);
            try (final LittleEndianDataOutput output2 = LittleEndianDataOutput.wrap(BAOS)) {
                this.writeBody(output2);
            }

            final byte[] ORIGINAL = BAOS.toByteArray();
            final int ORIGINAL_LEN = ORIGINAL.length;

            //final LZ4Factory LZ4FACTORY = LZ4Factory.safeInstance();
            //final LZ4Compressor LZ4COMP = LZ4FACTORY.highCompressor();
            final LZ4Factory LZ4FACTORY = LZ4Factory.fastestInstance();
            final LZ4Compressor LZ4COMP = LZ4FACTORY.highCompressor(20);
            final byte[] COMPRESSED = LZ4COMP.compress(ORIGINAL);
            final int COMPRESSED_LEN = COMPRESSED.length;

            output.writeInt(ORIGINAL_LEN);
            output.writeInt(COMPRESSED_LEN);
            output.write(COMPRESSED);

        } else {
            this.writeBody(output);
        }
    }

    /**
     * Performs the work of writing the savefile's contents, other than the
     * header and screenshot.
     *
     * @param output
     * @throws IOException
     */
    private void writeBody(LittleEndianDataOutput output) throws IOException {
        // Write the form version.
        output.write(this.FORMVERSION);

        // Write the plugin info section.
        output.writeInt(this.PLUGINS.calculateSize());
        this.PLUGINS.write(output);
        rangeModel.setValue(rangeModel.getValue() + 5 + this.PLUGINS.calculateSize());
        LOG.fine("Writing savegame: wrote plugin table.");

        // Rebuild and then write the file location table.
        this.FLT.rebuild(this);
        this.FLT.write(output);
        rangeModel.setValue(rangeModel.getValue() + this.FLT.calculateSize());
        LOG.fine("Writing savegame: rebuilt and wrote file location table.");

        for (GlobalData data : this.TABLE1) {
            try {
                data.write(output);
                rangeModel.setValue(rangeModel.getValue() + data.calculateSize());
                LOG.log(Level.FINE, "Writing savegame: \tGlobalData type {0}.", data.getType());
            } catch (IOException ex) {
                int idx = this.TABLE1.indexOf(data);
                throw new IOException("Error writing GlobalData " + idx + " from table 1.", ex);
            }

        }
        LOG.fine("Writing savegame: wrote GlobalDataTable #1.");

        for (GlobalData data : this.TABLE2) {
            try {
                data.write(output);
                rangeModel.setValue(rangeModel.getValue() + data.calculateSize());
                LOG.log(Level.FINE, "Writing savegame: \tGlobalData type {0}.", data.getType());
            } catch (IOException ex) {
                int idx = this.TABLE2.indexOf(data);
                throw new IOException("Error writing GlobalData " + idx + " from table 2.", ex);
            }
        }
        LOG.fine("Writing savegame: wrote GlobalDataTable #2.");

        for (ChangeForm form : this.CHANGEFORMS) {
            try {
                form.write(output);
                rangeModel.setValue(rangeModel.getValue() + form.calculateSize());
            } catch (IOException ex) {
                int idx = this.CHANGEFORMS.indexOf(form);
                throw new IOException("Error writing ChangeForm " + idx + " / " + this.CHANGEFORMS.size() + ".", ex);
            }
        }
        LOG.fine("Writing savegame: wrote changeform table.");

        for (GlobalData data : this.TABLE3) {
            try {
                data.write(output);
                rangeModel.setValue(rangeModel.getValue() + data.calculateSize());
                LOG.log(Level.FINE, "Writing savegame: \tGlobalData type {0}.", data.getType());
            } catch (IOException ex) {
                int idx = this.TABLE3.indexOf(data);
                throw new IOException("Error writing GlobalData " + idx + " from table 3.", ex);
            }
        }
        LOG.fine("Writing savegame: wrote GlobalDataTable #3.");

        output.writeInt(this.FORMIDARRAY.length);
        for (int formID : this.FORMIDARRAY) {
            output.writeInt(formID);
        }
        rangeModel.setValue(rangeModel.getValue() + 4 + 4 * this.FORMIDARRAY.length);
        LOG.fine("Writing savegame: wrote formid array.");

        output.writeInt(this.VISITEDWORLDSPACEARRAY.length);
        for (int formID : this.VISITEDWORLDSPACEARRAY) {
            output.writeInt(formID);
        }
        rangeModel.setValue(rangeModel.getValue() + 4 + 4 * this.VISITEDWORLDSPACEARRAY.length);
        LOG.fine("Writing savegame: wrote visited worldspace array.");

        output.write(this.UNKNOWN3);
        rangeModel.setValue(rangeModel.getValue() + this.UNKNOWN3.length);
        LOG.fine("Writing savegame: wrote unknown block.");

        float size = this.calculateSize() / 1048576.0f;
        TIMER.stop();
        LOG.info(String.format("Savegame written: %.1f mb in %s.", size, this.TIMER.getFormattedTime()));
    }

    /**
     * @see Element#calculateSize()
     * @return
     */
    @Override
    public int calculateSize() {
        int sum = 0;
        sum += this.HEADER.calculateSize();
        sum += 1;
        sum += this.PLUGINS.calculateSize();
        sum += this.FLT.calculateSize();

        sum += this.TABLE1.parallelStream().mapToInt(v -> v.calculateSize()).sum();
        sum += this.TABLE2.parallelStream().mapToInt(v -> v.calculateSize()).sum();
        sum += this.CHANGEFORMS.stream().mapToInt(v -> v.calculateSize()).sum();
        sum += this.TABLE3.parallelStream().mapToInt(v -> v.calculateSize()).sum();

        sum += 4;
        sum += 4 * this.FORMIDARRAY.length;
        sum += 4;
        sum += 4 * this.VISITEDWORLDSPACEARRAY.length;
        sum += this.UNKNOWN3.length;

        return sum;
    }

    /**
     * @see PapyrusElement#resolveRefs(restringer.ess.papyrus.Papyrus)
     */
    public void resolveRefs() {
        this.CHANGEFORMS.forEach(v -> v.resolveRefs(this, null));
        this.PAPYRUS.resolveRefs(this, null);
        this.PLUGINS.getPlugins().forEach(v -> v.resolveRefs(this, null));
    }

    /**
     * Assign names to changeforms.
     *
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        this.CHANGEFORMS.forEach(v -> v.addNames(names, strings));
        this.PAPYRUS.addNames(names, strings);
    }

    /**
     * @return The papyrus section.
     */
    public Papyrus getPapyrus() {
        return this.PAPYRUS;
    }

    /**
     * @return The digest of the <code>ESS</code> when it was read from the
     * disk.
     */
    public Long getDigest() {
        return this.DIGEST;
    }

    /**
     * @return The filename of the <code>ESS</code> when it was read from the
     * disk.
     */
    public String getFilename() {
        return this.FILENAME;
    }

    /**
     * @return The list of change forms.
     */
    public Map<RefID, ChangeForm> getChangeForms() {
        return this.CHANGEFORMS_MAP;
    }

    /**
     * @return The array of form IDs.
     */
    public int[] getFormIDs() {
        return this.FORMIDARRAY;
    }

    /**
     * @return The list of plugins.
     */
    public PluginInfo getPluginInfo() {
        return this.PLUGINS;
    }

    /**
     * Removes all <code>ChangeForm</code> objects with havok entries.
     *
     * @return The number of forms removed.
     */
    public int resetHavok() {
        for (ChangeForm form : this.CHANGEFORMS) {
            //form.
        }

        return 0;
    }

    /**
     * Removes null entries from form lists.
     *
     * @return An array containing two ints; the first is the number of entries
     * that were removed, and the second is the number of forms that had entries
     * remvoed.
     */
    public int[] cleanseFormLists() {
        int entries = 0;
        int forms = 0;

        for (ChangeForm form : this.CHANGEFORMS) {
            ChangeFormData data = form.getData();
            if (!(data instanceof ChangeFormFLST)) {
                continue;
            }

            ChangeFormFLST flst = (ChangeFormFLST) data;
            int removed = flst.cleanse();

            if (removed > 0) {
                entries += removed;
                forms++;
            }
        }

        return new int[]{entries, forms};
    }

    /**
     * Removes all script instances that are associated with non-existent
     * created forms.
     *
     * @return A count of the number of script instances that were removed.
     */
    public int removeNonexistentCreated() {
        int i = 0;

        LinkedList<ScriptInstance> nonexist = new LinkedList<>();

        this.PAPYRUS.getInstances().values().forEach(instance -> {
            RefID ref = instance.getRefID();
            if (ref.getType() == RefID.Type.CREATED) {
                assert this.CHANGEFORMS_MAP.containsKey(ref) == (ref.getForm() != null);
                if (ref.getForm() == null) {
                    nonexist.add(instance);
                }
            }
        });

        nonexist.forEach(v -> this.removeElement(v));
        return nonexist.size();
    }

    /**
     * Removes every <code>ScriptInstance</code> provided by a specified plugin.
     *
     * @param plugin The plugin whose instances will be removed.
     * @return A count of the number of script instances that were removed.
     */
    public int removePluginInstances(Plugin plugin) {
        Objects.requireNonNull(plugin);
        final Set<ScriptInstance> INSTANCES = plugin.getInstances();
        INSTANCES.stream().forEach(v -> this.PAPYRUS.removeElement(v));
        return INSTANCES.size();
    }

    /**
     * Removes every <code>ChangeForm</code> provided by a specified plugin.
     *
     * @param plugin The plugin whose forms will be removed.
     * @return A count of the number of changeforms that were removed.
     */
    public int removePluginForms(Plugin plugin) {
        Objects.requireNonNull(plugin);
        final Set<ChangeForm> FORMS = plugin.getForms();
        FORMS.stream().forEach(v -> this.removeElement(v));
        return FORMS.size();
    }

    /**
     * Removes an <code>Element</code>.
     *
     * @param element The element to remove.
     * @return A flag indicating that the element was found and removed.
     */
    public boolean removeElement(Element element) {
        Objects.requireNonNull(element);

        if (element instanceof ChangeForm) {
            final ChangeForm FORM = (ChangeForm) element;
            if (!this.CHANGEFORMS.contains(FORM)) {
                return false;
            }

            this.CHANGEFORMS.remove(FORM);
            this.CHANGEFORMS_MAP.remove(FORM.getRefID());
            return true;

        } else if (element instanceof PapyrusElement) {
            return this.PAPYRUS.removeElement((PapyrusElement) element);

        } else {
            return false;
        }
    }

    /**
     * @return A <code>FilterTreeModel</code>.
     */
    public FilterTreeModel createTreeModel() {
        TIMER.restart();
        //this.rangeModel = Objects.requireNonNull(model);

        final FilterTreeModel MODEL = new FilterTreeModel();
        final Papyrus papyrus = this.getPapyrus();

        // Put strings in idividual alphabetized containers.
        Map<Character, ArrayList<Element>> stringDictionary = new TreeMap<>();
        int stringProtoSize = papyrus.getStringTable().size() / 50;

        papyrus.getStringTable().forEach(string -> {
            if (string.length() > 0) {
                char firstChar = Character.toUpperCase(string.charAt(0));

                if (Character.isLetter(firstChar)) {
                    List<Element> entry = stringDictionary.computeIfAbsent(firstChar, ch -> new ArrayList<>(stringProtoSize));
                    entry.add(string);

                } else {
                    List<Element> entry = stringDictionary.computeIfAbsent('0', ch -> new ArrayList<>(stringProtoSize));
                    entry.add(string);
                }
            } else {
                List<Element> entry = stringDictionary.computeIfAbsent('0', ch -> new ArrayList<>(stringProtoSize));
                entry.add(string);
            }
        });

        ArrayList<Node> stringNodes = new ArrayList<>(stringDictionary.size());
        stringDictionary.forEach((ch, list) -> stringNodes.add(MODEL.elementContainer(Character.toString(ch), list)));

        TIMER.stop();
        LOG.info(String.format("Classifying strings took %s.", TIMER.getFormattedTime()));
        TIMER.restart();

        this.rangeModel.setValue(16493);

        // Put script instances in individual alphabetized containers.
        Map<Character, ArrayList<Element>> instanceDictionary = new TreeMap<>();
        int instanceProtoSize = papyrus.getInstances().size() / 30;

        papyrus.getInstances().values().forEach(instance -> {
            char firstChar = Character.toUpperCase(instance.toString().charAt(0));

            if (Character.isLetter(firstChar)) {
                List<Element> entry = instanceDictionary.computeIfAbsent(firstChar, ch -> new ArrayList<>(instanceProtoSize));
                entry.add(instance);

            } else {
                List<Element> entry = instanceDictionary.computeIfAbsent('0', ch -> new ArrayList<>(instanceProtoSize));
                entry.add(instance);
            }
        });

        ArrayList<Node> instanceNodes = new ArrayList<>(instanceDictionary.size());
        instanceDictionary.forEach((ch, list) -> instanceNodes.add(MODEL.elementContainer(Character.toString(ch), list)));

        TIMER.stop();
        LOG.info(String.format("Classifying instances took %s.", TIMER.getFormattedTime()));
        TIMER.restart();

        this.rangeModel.setValue(45821);

        // ActiveScript nodes.
        List<Node> activeNodes = new ArrayList<>(papyrus.getActiveScripts().size());
        papyrus.getActiveScripts().values().forEach(active -> {
            activeNodes.add(MODEL.node(active, active.getData().getStackFrames()));
        });

        TIMER.stop();
        LOG.info(String.format("Making activeescript nodes took %s.", TIMER.getFormattedTime()));
        TIMER.restart();

        // Function message nodes.
        List<Node> funcNodes = papyrus.getFunctionMessages()
                .stream()
                .map(v -> MODEL.node(v, v.getMessage()))
                .collect(Collectors.toList());

        TIMER.stop();
        LOG.info(String.format("Making function message nodes took %s.", TIMER.getFormattedTime()));
        TIMER.restart();

        // Suspended stack nodes.
        List<Node> stackNodes1 = papyrus.getSuspendedStacks1()
                .stream()
                .map(v -> MODEL.node(v, v.getMessage()))
                .collect(Collectors.toList());

        List<Node> stackNodes2 = papyrus.getSuspendedStacks2()
                .stream()
                .map(v -> MODEL.node(v, v.getMessage()))
                .collect(Collectors.toList());

        TIMER.stop();
        LOG.info(String.format("Making suspended stack nodes took %s.", TIMER.getFormattedTime()));
        TIMER.restart();

        this.rangeModel.setValue(45831);

        // Put changeform nodes into Type containers.
        Map<ChangeForm.Type, ArrayList<ChangeForm>> formDictionary = new TreeMap<>();
        int formProtoSize = this.getChangeForms().size() / 40;

        this.getChangeForms().values().forEach(form -> {
            final ChangeForm.Type CODE = form.getType();
            List<ChangeForm> entry = formDictionary.computeIfAbsent(CODE, c -> new ArrayList<>(instanceProtoSize));
            entry.add(form);
        });

        ArrayList<Node> formNodes = new ArrayList<>(formDictionary.size());
        formDictionary.forEach((type, list) -> formNodes.add(MODEL.elementContainer(type.toString(), list)));

        TIMER.stop();
        LOG.info(String.format("Classifying changeforms took %s.", TIMER.getFormattedTime()));
        TIMER.restart();

        TIMER.stop();
        LOG.info(String.format("Making queued unbind nodes took %s.", TIMER.getFormattedTime()));
        TIMER.restart();

        // Make the top-level folders.
        Node pluginsNode = MODEL.elementContainer("Plugins", this.getPluginInfo().getPlugins());
        Node stringsNode = MODEL.nodeContainer("Strings", stringNodes);
        Node scriptsNode = MODEL.elementContainer("Scripts", papyrus.getScripts().values());
        Node instancesNode = MODEL.nodeContainer("Script Instances", instanceNodes);
        Node referencesNode = MODEL.elementContainer("References", papyrus.getReferences().values());
        Node arraysNode = MODEL.elementContainer("Arrays", papyrus.getArrays().values());

        Node activeNode = MODEL.nodeContainer("Active Scripts", activeNodes);
        Node funcNode = MODEL.nodeContainer("Function Messages", funcNodes);
        Node stack1Node = MODEL.nodeContainer("Suspended Stacks 1", stackNodes1);
        Node stack2Node = MODEL.nodeContainer("Suspended Stacks 2", stackNodes2);
        Node formsNode = MODEL.nodeContainer("ChangeForms", formNodes);
        Node unbindsNode = MODEL.elementContainer("QueuedUnbinds", papyrus.getUnbinds());

        TIMER.stop();
        LOG.info(String.format("Making toplevel folder nodes took %s.", TIMER.getFormattedTime()));
        TIMER.restart();

        this.rangeModel.setValue(46670);

        // Do some sorting.
        stringNodes.forEach(n -> n.sort());
        instanceNodes.forEach(n -> n.sort());
        stringsNode.sort();
        instancesNode.sort();
        scriptsNode.sort();
        formsNode.sort();

        TIMER.stop();
        LOG.info(String.format("Sorting took %s.", TIMER.getFormattedTime()));
        TIMER.restart();

        this.rangeModel.setValue(50681);

        // Populate the root elementNode.
        ArrayList<Node> topLevel = new ArrayList<>(14);
        topLevel.add(pluginsNode);
        topLevel.add(stringsNode);
        topLevel.add(scriptsNode);
        topLevel.add(instancesNode);
        topLevel.add(referencesNode);
        topLevel.add(arraysNode);
        topLevel.add(activeNode);
        topLevel.add(funcNode);
        topLevel.add(stack1Node);
        topLevel.add(stack2Node);
        topLevel.add(formsNode);
        topLevel.add(unbindsNode);

        Node root = MODEL.root(this, topLevel);
        MODEL.setRoot(root);

        TIMER.stop();
        LOG.info(String.format("Making the model %s.", TIMER.getFormattedTime()));

        this.rangeModel.setValue(50682);

        return MODEL;
    }

    /**
     * Does a very general search for an ID.
     *
     * @param id The ID to search for.
     * @return Any match of any kind.
     */
    public Element broadSpectrumMatch(int id) {
        RefID ref = new RefID(id);
        if (this.CHANGEFORMS_MAP.containsKey(ref)) {
            return this.CHANGEFORMS_MAP.get(ref);
        }

        return this.PAPYRUS.broadSpectrumMatch(EID.make4byte(id));
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @return
     */
    public String getInfo(restringer.Profile.Analysis analysis) {
        final StringBuilder BUF = new StringBuilder();

        long time = this.HEADER.FILETIME;
        long millis = time / 10000L - 11644473600000L;
        final java.util.Date DATE = new java.util.Date(millis);

        int seconds = (int) (time / 10000000L) % 60;
        int minutes = (int) (time / 600000000L) % 60;
        int hours = (int) (time / 36000000000L) % 24;
        int years = (int) (time / (864000000000L * 365.25));
        int days = (int) (time / 864000000000L);

        BUF.append(this.FILENAME);
        BUF.append("\n\nMD5: ").append(this.DIGEST);
        BUF.append(String.format("\nTimestamp: %s\n", DATE));
        BUF.append("\nSavenumber: ").append(this.HEADER.SAVENUMBER);
        BUF.append("\nVersion: ").append(this.HEADER.VERSION);
        BUF.append("\n\nPlayer name: ").append(this.HEADER.NAME);
        BUF.append("\nGender: ").append(this.HEADER.SEX);
        BUF.append("\nRaceID: ").append(this.HEADER.RACEID);
        BUF.append("\nLevel: ").append(this.HEADER.LEVEL);
        BUF.append("\nCurrent XP: ").append(this.HEADER.CURRENT_XP);
        BUF.append("\nNeeded XP: ").append(this.HEADER.NEEDED_XP);
        BUF.append("\nLocation: ").append(this.HEADER.LOCATION);
        BUF.append("\nGame date: ").append(this.HEADER.GAMEDATE);

        return BUF.toString();
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return this.FILENAME;
    }

    static private void compareESS(File orig, File copy) throws IOException {
        try (LittleEndianInputStream IN1 = LittleEndianInputStream.openD(orig);
                LittleEndianInputStream IN2 = LittleEndianInputStream.openD(copy)) {

            final Header HEADER1 = new Header(IN1);
            final Header HEADER2 = new Header(IN2);
            Game game = HEADER1.GAME;

            final LittleEndianInputStream DEC1;
            final LittleEndianInputStream DEC2;

            // SkyrimSE uses compression, so enable that.
            if (game.isSSE()) {
                DEC1 = LittleEndianInputStream.wrapD(new LZ4BlockInputStream(IN1));
                DEC2 = LittleEndianInputStream.wrapD(new LZ4BlockInputStream(IN2));
            } else {
                DEC1 = IN1;
                DEC2 = IN2;
            }
            assert DEC1.getDigest().equals(DEC2.getDigest());

            // Read the form version.
            final int FORMVERSION1 = DEC1.readByte();
            final int FORMVERSION2 = DEC2.readByte();
            assert FORMVERSION1 == FORMVERSION2;
            assert DEC1.getDigest().equals(DEC2.getDigest());

            switch (game) {
                case SKYRIM:
                    assert FORMVERSION1 == 74 || FORMVERSION1 == 73 : "Invalid formVersion: " + FORMVERSION1;
                    assert FORMVERSION2 == 74 || FORMVERSION2 == 73 : "Invalid formVersion: " + FORMVERSION2;
                    break;
                case SKYRIMSE:
                    assert FORMVERSION1 == 77 : "Invalid formVersion: " + FORMVERSION1;
                    assert FORMVERSION2 == 77 : "Invalid formVersion: " + FORMVERSION2;
                    break;
                case FALLOUT4:
                    assert FORMVERSION1 == 67 : "Invalid formVersion: " + FORMVERSION1;
                    assert FORMVERSION2 == 67 : "Invalid formVersion: " + FORMVERSION2;
                    final String UNKNOWN_STRING1 = DEC1.readWString();
                    final String UNKNOWN_STRING2 = DEC2.readWString();
                    break;
                default:
                    throw new IllegalArgumentException("Unrecognized game.");
            }

            // Make the ESSContexts.
            final ESSContext CTX1 = new ESSContext(HEADER1, FORMVERSION1);
            final ESSContext CTX2 = new ESSContext(HEADER2, FORMVERSION2);

            // Read the plugin info section.
            int pluginInfoSize1 = DEC1.readInt();
            int pluginInfoSize2 = DEC2.readInt();
            assert pluginInfoSize1 == pluginInfoSize2;

            PluginInfo PLUGINS1 = new PluginInfo(DEC1);
            PluginInfo PLUGINS2 = new PluginInfo(DEC2);
            assert Objects.equals(PLUGINS1, PLUGINS2);
            assert pluginInfoSize1 == PLUGINS1.calculateSize();
            assert pluginInfoSize2 == PLUGINS2.calculateSize();
            assert DEC1.getDigest().equals(DEC2.getDigest());

            // Read the file location table.
            final FileLocationTable FLT1 = new FileLocationTable(DEC1);
            final FileLocationTable FLT2 = new FileLocationTable(DEC2);
            assert DEC1.getDigest().equals(DEC2.getDigest());
            //=========================
            //assert Objects.equals(FLT1, FLT2);

            final ArrayList<GlobalData> TABLE11 = new ArrayList<>(FLT1.table1Count);
            final ArrayList<GlobalData> TABLE21 = new ArrayList<>(FLT1.table2Count);
            final ArrayList<GlobalData> TABLE31 = new ArrayList<>(FLT1.table3Count);

            final ArrayList<GlobalData> TABLE12 = new ArrayList<>(FLT2.table1Count);
            final ArrayList<GlobalData> TABLE22 = new ArrayList<>(FLT2.table2Count);
            final ArrayList<GlobalData> TABLE32 = new ArrayList<>(FLT2.table3Count);

            final ArrayList<ChangeForm> CHANGEFORMS1 = new ArrayList<>(FLT1.changeFormCount);
            final ArrayList<ChangeForm> CHANGEFORMS2 = new ArrayList<>(FLT2.changeFormCount);

            LinkedHashMap<RefID, ChangeForm> CHANGEFORMS_MAP1 = new LinkedHashMap<>(FLT1.changeFormCount);
            LinkedHashMap<RefID, ChangeForm> CHANGEFORMS_MAP2 = new LinkedHashMap<>(FLT2.changeFormCount);

            int i = 0;

            // Read the first set of data tables.
            try {
                for (i = 0; i < FLT1.table1Count; i++) {
                    GlobalData DATA1 = new GlobalData(DEC1, CTX1);
                    GlobalData DATA2 = new GlobalData(DEC2, CTX2);
                    TABLE11.add(DATA1);
                    TABLE12.add(DATA2);
                    assert Objects.equals(DATA1, DATA2);
                    assert DEC1.getDigest().equals(DEC2.getDigest());
                }
                assert TABLE11.equals(TABLE12);
                LOG.fine("Reading savegame: read GlobalDataTable #1.");
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d GlobalData from table #1.", i, FLT1.table1Count), ex);
            }

            // Read the second set of data tables.
            try {
                for (i = 0; i < FLT1.table2Count; i++) {
                    GlobalData DATA1 = new GlobalData(DEC1, CTX1);
                    GlobalData DATA2 = new GlobalData(DEC2, CTX2);
                    TABLE21.add(DATA1);
                    TABLE22.add(DATA2);
                    assert Objects.equals(DATA1, DATA2);
                    assert DEC1.getDigest().equals(DEC2.getDigest());
                }
                assert TABLE21.equals(TABLE22);
                LOG.fine("Reading savegame: read GlobalDataTable #2.");
            } catch (IOException ex) {
                throw new IOException(String.format("Error; read %d/%d GlobalData from table #2.", i, FLT1.table2Count), ex);
            }

            // Read the changeforms.
            try {
                for (i = 0; i < FLT1.changeFormCount; i++) {
                    ChangeForm FORM1 = new ChangeForm(DEC1, CTX1);
                    ChangeForm FORM2 = new ChangeForm(DEC2, CTX2);
                    CHANGEFORMS1.add(FORM1);
                    CHANGEFORMS2.add(FORM2);
                    assert FORM1.identical(FORM2);
                    assert DEC1.getDigest().equals(DEC2.getDigest());
                    CHANGEFORMS_MAP1.put(FORM1.getRefID(), FORM1);
                    CHANGEFORMS_MAP2.put(FORM2.getRefID(), FORM2);
                }
                assert CHANGEFORMS1.equals(CHANGEFORMS2);
                LOG.fine("Reading savegame: read changeform table.");
            } catch (IOException | AssertionError ex) {
                throw new IOException(String.format("Error; read %d/%d ChangeForm definitions.", i, FLT1.changeFormCount), ex);
            }

            // Read the third set of data tables.
            try {
                for (i = 0; i < FLT1.table3Count; i++) {
                    GlobalData DATA1 = new GlobalData(DEC1, CTX1);
                    GlobalData DATA2 = new GlobalData(DEC2, CTX2);
                    TABLE31.add(DATA1);
                    TABLE32.add(DATA2);
                    assert Objects.equals(DATA1, DATA2);
                    assert DEC1.getDigest().equals(DEC2.getDigest());
                    LOG.log(Level.FINE, "Reading savegame: \tGlobalData type {0}.", DATA1.getType());
                }
                assert TABLE31.equals(TABLE32);
                LOG.fine("Reading savegame: read GlobalDataTable #3.");
            } catch (Exception | Error ex) {
                throw new IOException(String.format("Error; read %d/%d GlobalData from table #3.", i, FLT1.table3Count), ex);
            }

            final Papyrus P1 = TABLE31.get(1).getPapyrus();
            final Papyrus P2 = TABLE32.get(1).getPapyrus();

            int formIDCount1 = DEC1.readInt();
            int formIDCount2 = DEC2.readInt();
            final int[] FORMIDARRAY1 = new int[formIDCount1];
            final int[] FORMIDARRAY2 = new int[formIDCount2];

            for (i = 0; i < formIDCount1; i++) {
                FORMIDARRAY1[i] = DEC1.readInt();
                FORMIDARRAY2[i] = DEC2.readInt();
            }
            assert Arrays.equals(FORMIDARRAY1, FORMIDARRAY2);
            LOG.fine("Reading savegame: read formid array.");
            assert DEC1.getDigest().equals(DEC2.getDigest());

            int worldspaceIDCount1 = DEC1.readInt();
            int worldspaceIDCount2 = DEC2.readInt();
            final int[] VISITEDWORLDSPACEARRAY1 = new int[worldspaceIDCount1];
            final int[] VISITEDWORLDSPACEARRAY2 = new int[worldspaceIDCount2];

            for (i = 0; i < worldspaceIDCount1; i++) {
                VISITEDWORLDSPACEARRAY1[i] = DEC1.readInt();
                VISITEDWORLDSPACEARRAY2[i] = DEC2.readInt();
            }
            assert Arrays.equals(VISITEDWORLDSPACEARRAY1, VISITEDWORLDSPACEARRAY2);
            LOG.fine("Reading savegame: read visited worldspace array.");
            assert DEC1.getDigest().equals(DEC2.getDigest());

            final byte[] BUF = new byte[8192];
            final ByteArrayOutputStream BAOS = new ByteArrayOutputStream();

            while (DEC1.available() > 0) {
                int read1 = DEC1.read(BUF);
                BAOS.write(BUF, 0, read1);

                int read2 = DEC2.read(BUF);
                BAOS.write(BUF, 0, read2);
            }

            final byte[] UNKNOWN31 = BAOS.toByteArray();
            final byte[] UNKNOWN32 = BAOS.toByteArray();
            assert Arrays.equals(UNKNOWN31, UNKNOWN32);
            LOG.fine("Reading savegame: read unknown block.");

            assert DEC1.getDigest().equals(DEC2.getDigest());
        }
    }

    final private Header HEADER;
    final private byte FORMVERSION;
    final private String UNKNOWN_STRING;
    final private PluginInfo PLUGINS;
    final private FileLocationTable FLT;
    final private List<GlobalData> TABLE1;
    final private List<GlobalData> TABLE2;
    final private List<ChangeForm> CHANGEFORMS;
    final private Map<RefID, ChangeForm> CHANGEFORMS_MAP;
    final private List<GlobalData> TABLE3;
    final private int[] FORMIDARRAY;
    final private int[] VISITEDWORLDSPACEARRAY;
    final private byte[] UNKNOWN3;
    final private byte[] SKSE;
    final String FILENAME;
    final private Long DIGEST;
    final private Papyrus PAPYRUS;
    final private Timer TIMER;
    private ProgressModel rangeModel;
    static final private Logger LOG = Logger.getLogger(ESS.class.getCanonicalName());
    static private File prevSaveFile = null;

    static final public Predicate<Element> THREAD = (Element v)
            -> v instanceof restringer.ess.papyrus.ActiveScript;

    static final public Predicate<Element> OWNABLE = (Element v)
            -> v instanceof restringer.ess.papyrus.ActiveScript
            || v instanceof restringer.ess.papyrus.StackFrame
            || v instanceof restringer.ess.papyrus.ArrayInfo;

    static final public Predicate<Element> DELETABLE = (Element v)
            -> v instanceof restringer.ess.papyrus.Script
            || v instanceof restringer.ess.papyrus.ScriptInstance
            || v instanceof restringer.ess.papyrus.Reference
            || v instanceof restringer.ess.papyrus.ArrayInfo
            || v instanceof restringer.ess.papyrus.ActiveScript
            || v instanceof restringer.ess.ChangeForm;

    /**
     * Describes the table of file locations.
     *
     * @author Mark Fairchild
     * @version 2016/05/08
     */
    static final public class FileLocationTable implements Element {

        /**
         * Creates a new <code>FileLocationTable</code> by reading from a
         * <code>LittleEndianDataOutput</code>. No error handling is performed.
         *
         * @param input The input stream.
         * @throws IOException
         */
        public FileLocationTable(LittleEndianInput input) throws IOException {
            assert null != input;
            this.formIDArrayCountOffset = input.readInt();
            this.unknownTable3Offset = input.readInt();
            this.table1Offset = input.readInt();
            this.table2Offset = input.readInt();
            this.changeFormsOffset = input.readInt();
            this.table3Offset = input.readInt();

            this.table1Count = input.readInt();
            this.table2Count = input.readInt();
            this.table3Count = input.readInt() + 1;
            this.changeFormCount = input.readInt();

            this.UNUSED = new int[15];
            for (int i = 0; i < 15; i++) {
                this.UNUSED[i] = input.readInt();
            }
        }

        /**
         * Rebuilds the file location table for an <code>ESS</code>.
         *
         * @param ess
         */
        public void rebuild(ESS ess) {
            int table1Size = ess.TABLE1.stream().mapToInt(v -> v.calculateSize()).sum();
            int table2Size = ess.TABLE2.stream().mapToInt(v -> v.calculateSize()).sum();
            int table3Size = ess.TABLE3.stream().mapToInt(v -> v.calculateSize()).sum();
            int changeFormsSize = ess.CHANGEFORMS.parallelStream().mapToInt(v -> v.calculateSize()).sum();

            this.table1Offset = 0;
            this.table1Offset += ess.HEADER.calculateSize();
            this.table1Offset += 5;
            this.table1Offset += ess.PLUGINS.calculateSize();
            this.table1Offset += this.calculateSize();

            this.table2Offset = this.table1Offset + table1Size;
            this.changeFormCount = ess.CHANGEFORMS.size();
            this.changeFormsOffset = this.table2Offset + table2Size;
            this.table3Offset = this.changeFormsOffset + changeFormsSize;
            this.formIDArrayCountOffset = this.table3Offset + table3Size;

            this.unknownTable3Offset = 0;
            this.unknownTable3Offset += this.formIDArrayCountOffset;
            this.unknownTable3Offset += 4 + 4 * ess.FORMIDARRAY.length;
            this.unknownTable3Offset += 4 + 4 * ess.VISITEDWORLDSPACEARRAY.length;
        }

        /**
         * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
         * @param output The output stream.
         * @throws IOException
         */
        @Override
        public void write(LittleEndianDataOutput output) throws IOException {
            assert null != output;

            output.writeInt(this.formIDArrayCountOffset);
            output.writeInt(this.unknownTable3Offset);
            output.writeInt(this.table1Offset);
            output.writeInt(this.table2Offset);
            output.writeInt(this.changeFormsOffset);
            output.writeInt(this.table3Offset);
            output.writeInt(this.table1Count);
            output.writeInt(this.table2Count);
            output.writeInt(this.table3Count - 1);
            output.writeInt(this.changeFormCount);

            for (int i = 0; i < 15; i++) {
                output.writeInt(this.UNUSED[i]);
            }
        }

        /**
         * @see restringer.ess.Element#calculateSize()
         * @return The size of the <code>Element</code> in bytes.
         */
        @Override
        public int calculateSize() {
            return 100;
        }

        /**
         * @see Object#hashCode()
         * @return
         */
        @Override
        public int hashCode() {
            int hash = 7;
            hash = 73 * hash + this.formIDArrayCountOffset;
            hash = 73 * hash + this.unknownTable3Offset;
            hash = 73 * hash + this.table1Offset;
            hash = 73 * hash + this.table2Offset;
            hash = 73 * hash + this.changeFormsOffset;
            hash = 73 * hash + this.table3Offset;
            hash = 73 * hash + this.table1Count;
            hash = 73 * hash + this.table2Count;
            hash = 73 * hash + this.table3Count;
            hash = 73 * hash + this.changeFormCount;
            hash = 73 * hash + Arrays.hashCode(this.UNUSED);
            return hash;
        }

        /**
         * @see Object#equals(java.lang.Object)
         * @return
         */
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            } else if (obj == null) {
                return false;
            } else if (getClass() != obj.getClass()) {
                return false;
            }

            final FileLocationTable other = (FileLocationTable) obj;
            if (this.formIDArrayCountOffset != other.formIDArrayCountOffset) {
                return false;
            } else if (this.unknownTable3Offset != other.unknownTable3Offset) {
                return false;
            } else if (this.table1Offset != other.table1Offset) {
                return false;
            } else if (this.table2Offset != other.table2Offset) {
                return false;
            } else if (this.changeFormsOffset != other.changeFormsOffset) {
                return false;
            } else if (this.table3Offset != other.table3Offset) {
                return false;
            } else if (this.table1Count != other.table1Count) {
                return false;
            } else if (this.table2Count != other.table2Count) {
                return false;
            } else if (this.table3Count != other.table3Count) {
                return false;
            } else if (this.changeFormCount != other.changeFormCount) {
                return false;
            } else {
                return Arrays.equals(this.UNUSED, other.UNUSED);
            }
        }

        int formIDArrayCountOffset;
        int unknownTable3Offset;
        int table1Offset;
        int table2Offset;
        int changeFormsOffset;
        int table3Offset;
        int table1Count;
        int table2Count;
        int table3Count;
        int changeFormCount;
        int[] UNUSED;

    }

    /**
     * Creates a backup of a file.
     *
     * @param file
     * @throws IOException
     */
    static private void backupFile(File file) throws IOException {
        final String TIME = new java.text.SimpleDateFormat("yyyy.MM.dd.HH.mm.ss").format(new java.util.Date());
        final String NAME = file.getName();
        final String NEWNAME = NAME + "." + TIME;
        final File newFile = new File(file.getParent(), NEWNAME);

        if (!newFile.exists()) {
            newFile.createNewFile();
        }

        try (final java.nio.channels.FileChannel SRC = new java.io.FileInputStream(file).getChannel();
                final java.nio.channels.FileChannel DEST = new java.io.FileOutputStream(newFile).getChannel()) {
            DEST.transferFrom(SRC, 0, SRC.size());
        }
    }
}
