/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.File;
import java.util.List;
import java.util.regex.Pattern;
import javax.swing.filechooser.FileNameExtensionFilter;

/**
 *
 * @author Mark Fairchild
 */
public enum Game {
    SKYRIM("Skyrim Savefile", "ess"),
    SKYRIMSE("Skyrim SE Savefile", "ess"),
    FALLOUT4("Fallout 4 Savefile", "fos");

    final public String EXT;
    final public Pattern REGEX;
    final public FileNameExtensionFilter SWING_FILTER;
    static public List<Game> VALUES = java.util.Collections.unmodifiableList(java.util.Arrays.asList(Game.values()));

    /**
     * Finds the <code>Game</code> that matches a filename, by extension.
     *
     * @param filename
     * @return
     */
    static public Game matchFilename(String filename) {
        for (Game game : VALUES) {
            if (game.REGEX.asPredicate().test(filename)) {
                return game;
            }
        }
        return null;
    }

    /**
     * Test if a savefile matches.
     *
     * @param f
     * @return
     */
    public boolean testFilename(File f) {
        return REGEX.asPredicate().test(f.getName());
    }

    /**
     * @return Flag indicating whether the game has a 32bit string model.
     */
    public boolean isStr32() {
        return this.isSSE();
    }

    /**
     * @return Flag indicating whether the game is Fallout 4.
     */
    public boolean isFO4() {
        return this == FALLOUT4;
    }

    /**
     * @return Flag indicating whether the game is Skyrim Legendary Edition or
     * Skyrim Special Edition.
     *
     */
    public boolean isSkyrim() {
        return this.isSSE() || this.isSLE();
    }

    /**
     * @return Flag indicating whether the game is Skyrim Special Edition.
     */
    public boolean isSSE() {
        return this == SKYRIMSE;
    }

    /**
     * @return Flag indicating whether the game is Skyrim Legendary Edition.
     */
    public boolean isSLE() {
        return this == SKYRIM;
    }

    /**
     * Creates a new <code>Game</code> for the specified extension.
     *
     * @param name The game's name.
     * @param extension The game's savefile extension.
     */
    private Game(String name, String extension) {
        final String pattern = String.format(".+\\.(%s)", extension);
        this.REGEX = Pattern.compile(pattern, Pattern.CASE_INSENSITIVE);
        this.EXT = extension;
        this.SWING_FILTER = new FileNameExtensionFilter(name, extension);
    }

};
