/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.ess.papyrus.Papyrus;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import restringer.LittleEndianInput;

/**
 * Describes 3-byte formIDs from Skyrim savegames.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class GlobalData implements Element {

    /**
     * Creates a new <code>GlobalData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The ESSContext.
     * @throws IOException
     */
    public GlobalData(LittleEndianInput input, ESSContext ctx) throws IOException {
        assert null != input;
        this.TYPE = input.readInt();

        int size = input.readInt();
        final byte[] DATA = new byte[size];
        int read = input.read(DATA);

        if (read != size) {
            //throw new IOException();
        }

        this.block = new DefaultGlobalDataBlock(DATA);

        if (this.TYPE == 1001) {
            this.executor = Executors.newSingleThreadExecutor();
            this.task = this.executor.submit(() -> {
                try {
                    return new Papyrus(DATA, ctx, false);
                } catch (IOException ex) {
                    if (ctx.GAME.isSLE()) {
                        return new Papyrus(DATA, ctx, true);
                    } else {
                        throw ex;
                    }
                }
            });

        } else {
            this.executor = null;
            this.task = null;
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        output.writeInt(this.TYPE);
        output.writeInt(this.block.calculateSize());
        this.block.write(output);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return 8 + this.block.calculateSize();
    }

    /**
     * @return The value of the type field.
     */
    public int getType() {
        return this.TYPE;
    }

    /**
     * @return The data block.
     */
    public GlobalDataBlock getDataBlock() {
        return this.block;
    }

    /**
     *
     * @return @throws IOException
     */
    public Papyrus getPapyrus() throws IOException {
        if (this.TYPE != 1001) {
            throw new IllegalStateException("Not a papyrus block.");
        }

        if (null != this.task) {
            try {
                this.block = task.get();
                this.executor.shutdown();
                this.executor = null;
                this.task = null;
            } catch (InterruptedException | ExecutionException ex) {
                throw new IOException("Couldn't process Papyrus block.", ex);
            }
        }

        return (Papyrus) this.block;
    }

    /**
     * @see Object#toString() 
     * @return 
     */
    @Override
    public String toString() {
        return super.toString() + ": type " + Integer.toString(this.TYPE);
    }
    
    /**
     * @see Object#hashCode() 
     * @return 
     */
    @Override
    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + Objects.hashCode(this.block);
        return hash;
    }

    /**
     * @see Object#equals(java.lang.Object) 
     * @return 
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final GlobalData other = (GlobalData) obj;
        return Objects.equals(this.block, other.block);
    }

    final private int TYPE;
    private GlobalDataBlock block;

    private ExecutorService executor;
    private Future<Papyrus> task;

}
