/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;

/**
 * Describes header of Skyrim savegames.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class Header implements Element {

    /**
     * Creates a new <code>Header</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @throws IOException
     */
    public Header(LittleEndianInput input) throws IOException {
        Objects.requireNonNull(input);

        // Read the magic string.
        final byte[] magicBuffer = new byte[20];
        input.read(magicBuffer, 0, 3);

        switch (new String(magicBuffer, 0, 3).toUpperCase()) {
            case "TES":
                input.read(magicBuffer, 3, 10);
                this.MAGIC = new String(magicBuffer, 0, 13);
                break;
            case "FO4":
                input.read(magicBuffer, 3, 9);
                this.MAGIC = new String(magicBuffer, 0, 12);
                break;
            default:
                throw new IllegalArgumentException("Unrecognized header.");
        }

        // Read the header size.
        final int HEADERSIZE = input.readInt();
        assert HEADERSIZE < 256;

        // Read the version number.
        this.VERSION = input.readInt();

        // Identify which game produced the savefile.
        if (this.MAGIC.equalsIgnoreCase("TESV_SAVEGAME")) {
            if (8 <= this.VERSION && this.VERSION <= 9) {
                this.GAME = Game.SKYRIM;
            } else if (this.VERSION == 12) {
                this.GAME = Game.SKYRIMSE;
            } else {
                throw new IllegalArgumentException("Unknown version of Skyrim: " + this.VERSION);
            }
        } else if (this.MAGIC.equalsIgnoreCase("FO4_SAVEGAME")) {
            if (11 <= this.VERSION) {
                this.GAME = Game.FALLOUT4;
            } else {
                throw new IllegalArgumentException("Unknown version of Fallout4: " + this.VERSION);
            }
        } else {
            throw new IllegalArgumentException("Unknown game: " + this.MAGIC);
        }

        this.SAVENUMBER = input.readInt();
        this.NAME = WString.read(input);
        this.LEVEL = input.readInt();
        this.LOCATION = WString.read(input);
        this.GAMEDATE = WString.read(input);
        this.RACEID = WString.read(input);
        this.SEX = input.readShort();
        this.CURRENT_XP = input.readFloat();
        this.NEEDED_XP = input.readFloat();
        this.FILETIME = input.readLong();
        this.SCREENSHOT_WIDTH = input.readInt();
        this.SCREENSHOT_HEIGHT = input.readInt();

        if (this.GAME == Game.SKYRIMSE) {
            this.UNKNOWN_SHORT = input.readShort();
        } else {
            this.UNKNOWN_SHORT = 0;
        }

        assert HEADERSIZE == this.partialSize() : String.format("Header size should be %d, found %d.", HEADERSIZE, this.partialSize());

        switch (this.GAME) {
            case SKYRIM:
                this.BYPP = 3;
                break;
            case FALLOUT4:
            case SKYRIMSE:
                this.BYPP = 4;
                break;
            default:
                throw new IllegalArgumentException("Invalid game: " + this.GAME);
        }
        
        this.SCREENSHOT = new byte[this.BYPP * this.SCREENSHOT_WIDTH * this.SCREENSHOT_HEIGHT];
        input.readFully(this.SCREENSHOT);

    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.write(this.MAGIC.getBytes());
        output.writeInt(this.partialSize());
        output.writeInt(this.VERSION);
        output.writeInt(this.SAVENUMBER);
        output.writeESSElement(this.NAME);
        output.writeInt(this.LEVEL);
        output.writeESSElement(this.LOCATION);
        output.writeESSElement(this.GAMEDATE);
        output.writeESSElement(this.RACEID);
        output.writeShort(this.SEX);
        output.writeFloat(this.CURRENT_XP);
        output.writeFloat(this.NEEDED_XP);
        output.writeLong(this.FILETIME);
        output.writeInt(this.SCREENSHOT_WIDTH);
        output.writeInt(this.SCREENSHOT_HEIGHT);

        if (this.GAME == Game.SKYRIMSE) {
            output.writeShort(this.UNKNOWN_SHORT);
        }

        output.write(this.SCREENSHOT);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 4;
        sum += this.partialSize();
        sum += this.MAGIC.length();
        sum += this.SCREENSHOT.length;
        return sum;
    }

    /**
     * The size of the header, not including the magic string, the size
     * itself, or the screenshot.
     *
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    private int partialSize() {
        int sum = 0;
        sum += 4; // version
        sum += 4; // savenumber
        sum += this.NAME.calculateSize();
        sum += 4; // level
        sum += this.LOCATION.calculateSize();
        sum += this.GAMEDATE.calculateSize();
        sum += this.RACEID.calculateSize();
        sum += 2; // sex
        sum += 4; // current xp
        sum += 4; // needed xp
        sum += 8; // filtime
        sum += 8; // screenshot size
        sum += (this.GAME == Game.SKYRIMSE ? 2 : 0);
        return sum;
    }

    /**
     * @return A <code>BufferedImage</code> that can be used to display the
     * screenshot.
     */
    public BufferedImage getImage() {
        assert 0 < this.SCREENSHOT_WIDTH;
        assert 0 < this.SCREENSHOT_HEIGHT;

        final BufferedImage IMAGE = new BufferedImage(this.SCREENSHOT_WIDTH, this.SCREENSHOT_HEIGHT, BufferedImage.TYPE_INT_RGB);
        int x = 0;
        int y = 0;

        for (int i = 0; i < this.SCREENSHOT.length; i += this.BYPP) {
            int rgb = 0;
            for (int k = 0; k < this.BYPP; k++) {
                rgb += this.SCREENSHOT[i + k] << (k * 8);
            }

            IMAGE.setRGB(x, y, rgb);

            x++;
            if (x >= this.SCREENSHOT_WIDTH) {
                x = 0;
                y++;
            }

            assert 0 <= x && x < this.SCREENSHOT_WIDTH;
            assert 0 <= y && y <= this.SCREENSHOT_HEIGHT;
        }

        return IMAGE;
    }

    final public String MAGIC;
    final public int VERSION;
    final public int SAVENUMBER;
    final public WString NAME;
    final public int LEVEL;
    final public WString LOCATION;
    final public WString GAMEDATE;
    final public WString RACEID;
    final public short SEX;
    final public float CURRENT_XP;
    final public float NEEDED_XP;
    final public long FILETIME;
    final public int SCREENSHOT_WIDTH;
    final public int SCREENSHOT_HEIGHT;
    final public int BYPP;
    final public short UNKNOWN_SHORT;
    final public Game GAME;
    final private byte[] SCREENSHOT;

}
