/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import it.unimi.dsi.fastutil.objects.ObjectRBTreeSet;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.SortedSet;
import java.util.function.Predicate;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;
import restringer.Profile;
import restringer.ess.papyrus.ScriptInstance;

/**
 * Abstraction for plugins.
 *
 * @author Mark Fairchild
 */
final public class Plugin implements Element, AnalyzableElement, Linkable, Comparable<Plugin> {

    static public Plugin PROTOTYPE = new Plugin("Unofficial Skyrim Legendary Edition Patch");

    /**
     * Creates a new <code>Plugin</code> by reading from an input stream.
     *
     * @param input The input stream.
     * @param index The index of the plugin.
     * @throws IOException
     */
    public Plugin(LittleEndianInput input, int index) throws IOException {
        Objects.requireNonNull(input);
        this.NAME = input.readWString();
        this.INDEX = index;
        this.INSTANCES = new ObjectRBTreeSet<>((f1, f2) -> f1.getID().compareTo(f2.getID()));
        this.FORMS = new ObjectRBTreeSet<>((f1, f2) -> f1.getRefID().compareTo(f2.getRefID()));
    }

    private Plugin(String name) {
        this.NAME = name;
        this.INDEX = -1;
        this.INSTANCES = null;
        this.FORMS = null;
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.writeWString(this.NAME);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return 2 + this.NAME.getBytes(StandardCharsets.ISO_8859_1).length;
    }

    /**
     * @see restringer.ess.Linkable#toHTML()
     * @return
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"plugin://%s\">%s</a>", this.NAME, this.toString());
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return this.NAME;
    }

    /**
     * @return The <code>ScriptInstance</code> set.
     */
    public Set<ScriptInstance> getInstances() {
        return this.INSTANCES;
    }

    /**
     * @return The <code>ChangeForm</code> set.
     */
    public Set<ChangeForm> getForms() {
        return this.FORMS;
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();
        BUILDER.append("<html><h3>PLUGIN</h3>");

        BUILDER.append("<p>Name: ").append(this.NAME).append("</p>");
        BUILDER.append("<p>Index: ").append(this.INDEX).append("</p>");

        int numForms = this.FORMS.size();
        int numInstances = this.INSTANCES.size();

        if (null != analysis) {
            final List<String> PROVIDERS = new ArrayList<>();

            Predicate<String> espFilter = esp -> esp.equalsIgnoreCase(NAME);
            analysis.ESPS.forEach((mod, esps) -> {
                esps.stream().filter(espFilter).forEach(esp -> PROVIDERS.add(mod));
            });

            if (!PROVIDERS.isEmpty()) {
                String probableProvider = PROVIDERS.get(PROVIDERS.size() - 1);

                Predicate<SortedSet<String>> modFilter = e -> e.contains(probableProvider);
                int numScripts = (int) analysis.SCRIPT_ORIGINS.values().stream().filter(modFilter).count();

                BUILDER.append(String.format("<p>%d scripts.</p>", numScripts));
                BUILDER.append(String.format("<p>The plugin probably came from mod \"%s\".</p>", probableProvider));

                if (PROVIDERS.size() > 1) {
                    BUILDER.append("<p>Full list of providers:</p><ul>");
                    PROVIDERS.forEach(mod -> BUILDER.append(String.format("<li>%s", mod)));
                    BUILDER.append("</ul>");
                }
            }
        }

        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);

        Predicate<String> filter = esp -> esp.equalsIgnoreCase(NAME);
        final List<String> PROVIDERS = new ArrayList<>();

        analysis.ESPS.forEach((m, esps) -> {
            esps.stream().filter(filter).forEach(esp -> PROVIDERS.add(m));
        });

        return PROVIDERS.contains(mod);
    }

    /**
     * The name field.
     */
    final public String NAME;

    /**
     * The index field.
     */
    final public int INDEX;

    final private Set<ScriptInstance> INSTANCES;
    final private Set<ChangeForm> FORMS;

    /**
     * @see Comparable#compareTo(java.lang.Object)
     * @param o
     * @return
     */
    @Override
    public int compareTo(Plugin o) {
        if (null == o) {
            return 1;
        }

        return Objects.compare(this.NAME, o.NAME, String::compareToIgnoreCase);
    }

    /**
     * @see Object#hashCode()
     * @return
     */
    @Override
    public int hashCode() {
        int hash = 3;
        hash = 29 * hash + Objects.hashCode(this.NAME.toLowerCase());
        return hash;
    }

    /**
     * @see Object#equals(java.lang.Object)
     * @return
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        } else if (obj == null) {
            return false;
        } else if (getClass() != obj.getClass()) {
            return false;
        } else {
            final Plugin other = (Plugin) obj;
            return this.NAME.equalsIgnoreCase(other.NAME);
        }
    }

}
