/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;

/**
 * Describes a savegame's list of plugins.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class PluginInfo implements Element {

    /**
     * Creates a new <code>PluginInfo</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @throws IOException
     */
    public PluginInfo(LittleEndianInput input) throws IOException {
        Objects.requireNonNull(input);

        int size = input.readUnsignedByte();
        assert size >= 0 && size < 256;

        this.PLUGINS = new java.util.ArrayList<>(size);

        for (int i = 0; i < size; i++) {
            Plugin p = new Plugin(input, i);
            this.PLUGINS.add(p);
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        output.writeByte(this.PLUGINS.size());

        for (Plugin p : this.PLUGINS) {
            p.write(output);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 1;
        sum += this.PLUGINS.stream().mapToInt(plugin -> plugin.calculateSize()).sum();
        return sum;
    }

    /**
     * @return The list of plugins.
     */
    public List<Plugin> getPlugins() {
        return Collections.unmodifiableList(this.PLUGINS);
    }

    /**
     * @see Object#hashCode() 
     * @return 
     */
    @Override
    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + this.PLUGINS.hashCode();
        return hash;
    }

    /**
     * @see Object#equals(java.lang.Object) 
     * @return 
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final PluginInfo other = (PluginInfo) obj;
        return Objects.equals(this.PLUGINS, other.PLUGINS);
    }

    final private List<Plugin> PLUGINS;

}
