/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import restringer.LittleEndianDataOutput;
import java.awt.image.BufferedImage;
import restringer.LittleEndianInput;

/**
 * Describes 3-byte formIDs from Skyrim savegames.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class Screenshot implements Element {

    /**
     * Creates a new <code>Screenshot</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param header The savegame header.
     * @throws IOException
     */
    public Screenshot(LittleEndianInput input, Header header) throws IOException {
        assert null != header;
        assert null != input;
        this.WIDTH = header.SCREENSHOT_WIDTH;
        this.HEIGHT = header.SCREENSHOT_HEIGHT;

        switch (header.GAME) {
            case SKYRIM:
                this.BYPP = 3;
                break;
            case FALLOUT4:
            case SKYRIMSE:
                this.BYPP = 4;
                break;
            default:
                throw new IllegalArgumentException("Invalid game: " + header.GAME);
        }
        
        this.DATA = new byte[this.BYPP * this.WIDTH * this.HEIGHT];
        input.readFully(this.DATA);
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        output.write(this.DATA);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return this.DATA.length;
    }

    /**
     * @return A <code>BufferedImage</code> that can be used to display the
     * screenshot.
     */
    public BufferedImage getImage() {
        assert 0 < this.WIDTH;
        assert 0 < this.HEIGHT;

        if (null == this.image) {
            this.image = new BufferedImage(this.WIDTH, this.HEIGHT, BufferedImage.TYPE_INT_RGB);
            int x = 0;
            int y = 0;

            for (int i = 0; i < this.DATA.length; i += this.BYPP) {
                int rgb = 0;                
                for (int k = 0; k < this.BYPP; k++) {
                    rgb += this.DATA[i+k] << (k*8);
                }
                
                this.image.setRGB(x, y, rgb);
                
                x++;
                if (x >= this.WIDTH) {
                    x = 0;
                    y++;
                }

                assert 0 <= x && x < this.WIDTH;
                assert 0 <= y && y <= this.HEIGHT;
            }
        }

        return this.image;
    }

    /**
     * @return String representation.
     */
    //public String toString() {
    //}
    final private int WIDTH;
    final private int HEIGHT;
    final private int BYPP;
    final private byte[] DATA;
    transient private BufferedImage image;

}
