/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.logging.Logger;
import org.mozilla.universalchardet.UniversalDetector;
import restringer.IString;
import restringer.LittleEndianDataOutput;
import restringer.LittleEndianInput;

/**
 * Extends <code>IString</code> by handling charsets and storing the original
 * byte sequence.
 *
 * @author Mark Fairchild
 * @version 2016/06/28
 */
public class WString extends IString implements Element {

    /**
     * Creates a new <code>WString</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @return The new <code>WString</code>.
     * @throws IOException
     */
    static public WString read(LittleEndianInput input) throws IOException {
        Objects.requireNonNull(input);
        final int LEN = input.readUnsignedShort();
        final byte[] BYTES = new byte[LEN];

        int bytesRead = input.read(BYTES);
        assert bytesRead == LEN : String.format("String contained %d bytes, expected %d", bytesRead, LEN);

        final UniversalDetector DETECTOR = new UniversalDetector(null);
        DETECTOR.handleData(BYTES, 0, BYTES.length);
        DETECTOR.dataEnd();
        final String ENCODING = DETECTOR.getDetectedCharset();
        DETECTOR.reset();

        final Charset CHARSET = (null == ENCODING ? UTF8 : Charset.forName(ENCODING));
        assert null != CHARSET;

        if (CHARSET_LOG.add(CHARSET)) {
            LOG.info("Detected a new character encoding: " + CHARSET);
        }

        final String STR = new String(BYTES, CHARSET);
        return new WString(STR, BYTES);
    }

    /**
     * Copy constructor.
     *
     * @param other The original <code>WString</code>.
     */
    protected WString(WString other) {
        super(Objects.requireNonNull(other));
        this.BYTES = other.BYTES;
    }

    /**
     * Creates a new <code>WString</code> from a character sequence; the byte
     * array is generated from the string using UTF-8 encoding.
     *
     * @param cs The <code>CharSequence</code>.
     */
    protected WString(CharSequence cs) {
        super(Objects.requireNonNull(cs));
        this.BYTES = cs.toString().getBytes(UTF8);
    }

    /**
     * Creates a new <code>WString</code> from a character sequence and a byte
     * array.
     *
     * @param cs The <code>CharSequence</code>.
     * @param bytes The byte array.
     */
    private WString(CharSequence cs, byte[] bytes) {
        super(Objects.requireNonNull(cs));
        this.BYTES = Objects.requireNonNull(bytes);
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        if (this.BYTES.length > 0xFFFF) {
            output.writeShort(0xFFFF);
            output.write(this.BYTES, 0, 0xFFFF);
        } else {
            output.writeShort(this.BYTES.length);
            output.write(this.BYTES);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        if (this.BYTES.length > 0xFFFF) {
            return 2 + 0xFFFF;
        } else {
            return 2 + this.BYTES.length;
        }
    }

    /**
     * @return A log of all character sets that have been detected.
     */
    static public java.util.Set<Charset> getCharsetLog() {
        return java.util.Collections.unmodifiableSet(CHARSET_LOG);
    }

    final private byte[] BYTES;

    static final private Logger LOG = Logger.getLogger(WString.class.getName());
    static final private Charset UTF8 = StandardCharsets.UTF_8;
    static final private java.util.Set<Charset> CHARSET_LOG = new java.util.HashSet<>();

}
