/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import restringer.ess.AnalyzableElement;
import java.io.IOException;
import java.util.Objects;
import java.util.SortedSet;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.Profile;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.ess.Linkable;
import restringer.ess.Plugin;
import restringer.ess.RefID;

/**
 * Describes an active script in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class ActiveScript implements PapyrusElement, AnalyzableElement, Linkable, HasID {

    /**
     * Creates a new <code>ActiveScript</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public ActiveScript(LittleEndianInput input, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(ctx);

        this.ID = EID.read4byte(input);
        this.TYPE = input.readByte();
        this.owner = null;
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;
        output.writeESSElement(this.ID);
        output.writeByte(this.TYPE);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        return 1 + this.ID.calculateSize();
    }

    /**
     * Replaces the opcodes of each <code>StackFrame</code> with NOPs.
     */
    public void zero() {
        this.data.getStackFrames().forEach(f -> f.zero());
    }

    /**
     * @return The ID of the papyrus element.
     */
    @Override
    public EID getID() {
        return this.ID;
    }

    /**
     * @return The type of the script.
     */
    public byte getType() {
        return this.TYPE;
    }

    /**
     * @return The instance field.
     */
    public AnalyzableElement getInstance() {
        return this.owner;
    }

    /**
     * @return The <code>ActiveScriptData</code> for the script.
     */
    public ActiveScriptData getData() {
        return this.data;
    }

    /**
     * Sets the data field.
     *
     * @param newData The new value for the data field.
     */
    public void setData(ActiveScriptData newData) {
        this.data = newData;
    }

    /**
     * @return A flag indicating if the <code>ActiveScript</code> is terminated.
     *
     */
    public boolean isTerminated() {
        return this.data.getStackFrames().stream().allMatch(f -> f.isZeroed());
    }

    /**
     * @see restringer.ess.Linkable#toHTML()
     * @return
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"thread://%s\">%s</a>", this.ID, this.toString());
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        final StringBuilder BUILDER = new StringBuilder();
        if (null == this.data) {
            return this.ID + "-" + String.format("%02x", this.TYPE);
        }

        if (this.isTerminated()) {
            BUILDER.append("TERMINATED ");
        }

        if (this.isUndefined()) {
            StackFrame topFrame = this.data.getStackFrames().get(0);
            TString scriptName = topFrame.getScriptName();
            BUILDER.append("#").append(scriptName).append("# ");
        } else if (this.data.getStackFrames().size() > 0) {
            StackFrame topFrame = this.data.getStackFrames().get(0);
            TString scriptName = topFrame.getScriptName();
            BUILDER.append(scriptName).append(" ");
        }

        BUILDER.append("(").append(this.ID).append(") ");
        BUILDER.append(this.data.getStackFrames().size()).append(" frames");

        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();

        if (this.isTerminated()) {
            BUILDER.append("<html><h3>ACTIVESCRIPT (TERMINATED)</h3>");
        } else {
            BUILDER.append("<html><h3>ACTIVESCRIPT</h3>");
        }

        if (this.isTerminated()) {
            BUILDER.append("<p><em>WARNING: SCRIPT TERMINATED!</em><br/>This thread has been terminated and all of its instructions erased.</p>");
        } else if (this.isUndefined()) {
            BUILDER.append("<p><em>WARNING: SCRIPT MISSING!</em><br/>Remove Undefined Instances\" will terminate this thread.</p>");
        }

        if (null != analysis && !this.data.getStackFrames().isEmpty()) {
            StackFrame topFrame = this.data.getStackFrames().get(0);
            SortedSet<String> mods = analysis.SCRIPT_ORIGINS.get(topFrame.getScriptName());
            if (null != mods) {
                String mod = mods.last();
                BUILDER.append(String.format("<p>Probably running code from mod %s.</p>", mod));
            }

        }

        if (null == this.owner) {
            BUILDER.append("<p>This thread doesn't seem to be attached to an instance.</p>");

        } else if (this.owner instanceof ScriptInstance) {
            ScriptInstance instance = (ScriptInstance) this.owner;
            final RefID REF = instance.getRefID();
            final Plugin PLUGIN = REF.getPlugin();

            if (PLUGIN != null) {
                BUILDER.append(String.format("<p>This thread is attached to an instance from %s.</p>", PLUGIN.toHTML()));
            } else if (instance.getRefID().getType() == RefID.Type.CREATED) {
                BUILDER.append("<p>This thread is attach to instance that was created in-game.</p>");
            }
        }

        if (null == this.owner) {
            BUILDER.append("<p>No owner was identified.</p>");

        } else if (this.owner instanceof Linkable) {
            Linkable l = (Linkable) this.owner;
            String type = this.owner.getClass().getSimpleName();
            BUILDER.append(String.format("<p>%s: %s</p>", type, l.toHTML()));
        } else {
            String type = this.owner.getClass().getSimpleName();
            BUILDER.append(String.format("<p>%s: %s</p>", type, this.owner));
        }

        BUILDER.append("<p>");
        BUILDER.append(String.format("ID: %s<br/>", this.getID()));
        BUILDER.append(String.format("Type: %02x<br/>", this.TYPE, this.TYPE));
        BUILDER.append(String.format("Version: %d.%d<br/>", this.data.getMajorVersion(), this.data.getMinorVersion()));
        BUILDER.append(String.format("Unknown (var): %s<br/>", this.data.getUnknownVar().toHTML()));
        BUILDER.append(String.format("Flag: %08x<br/>", this.data.getFlag()));
        BUILDER.append(String.format("Unknown1 (byte): %02x<br/>", this.data.getUnknownByte()));
        BUILDER.append(String.format("Unknown2 (int): %08x<br/>", this.data.getUnknown2()));
        BUILDER.append(String.format("Unknown3 (byte): %02x<br/>", this.data.getUnknown3()));
        if (null != this.data.getUnknown4()) {
            BUILDER.append(String.format("Unknown4 (struct): %s<br/>", this.data.getUnknown4().toHTML()));
        } else {
            BUILDER.append(String.format("Unknown4 (struct): %s<br/>", this.data.getUnknown4()));
        }
        BUILDER.append(String.format("Unknown5 (byte): %02x<br/>", this.data.getUnknown5()));

        Element UNKNOWN2 = save.broadSpectrumMatch(this.getData().getUnknown2());
        if (null != UNKNOWN2) {
            BUILDER.append("<p>Potential match for unknown2 found using general search:<br/>");
            if (UNKNOWN2 instanceof Linkable) {
                BUILDER.append(((Linkable) UNKNOWN2).toHTML());
            } else {
                BUILDER.append(UNKNOWN2.toString());
            }
            BUILDER.append("</p>");
        }

        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);
        return this.data.getStackFrames().stream().anyMatch(v -> v.matches(analysis, mod));
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        assert null != this.data;
        this.data.addNames(names, strings);
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        assert null != this.data;
        this.data.resolveRefs(ess, this);

        if (this.data.getStackFrames().size() > 0) {
            Variable.RefID ref = (Variable.RefID) this.data.getStackFrames().get(0).getOwner();
            this.owner = ref.getReferent();
            //assert null != this.instance;
        }
    }

    /**
     * Checks if any of the stackframes in the <code>ActiveScript</code> is
     * running code from the specified <code>Script</code>.
     *
     * @param script The <code>Script</code> the check.
     * @return A flag indicating if any of the stackframes in the script matches
     * the specified <code>Script</code>.
     */
    public boolean hasScript(Script script) {
        if (null == this.data) {
            return false;
        }
        return this.data.getStackFrames().stream().anyMatch(frame -> frame.getScript() == script);
    }

    /**
     * @return A flag indicating if the <code>ActiveScript</code> is undefined.
     * An <code>ActiveScript</code> is undefined if any of its stack frames are
     * undefined.
     *
     */
    public boolean isUndefined() {
        if (null == this.data) {
            return false;
        }
        return this.data.getStackFrames().stream().anyMatch(frame -> frame.isUndefined());
    }

    final private EID ID;
    final private byte TYPE;
    private ActiveScriptData data;
    private AnalyzableElement owner;

}
