/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.Objects;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.ess.ESS;
import restringer.ess.Element;

/**
 * Describes array data in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/19
 */
final public class ArrayData implements PapyrusElement, HasID {

    /**
     * Creates a new <code>ArrayData</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param arrays The ArrayInfo structures.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public ArrayData(LittleEndianInput input, ArrayMap arrays, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(arrays);
        Objects.requireNonNull(ctx);

        if (ctx.GAME.isSSE() || ctx.GAME.isFO4()) {
            this.ID = EID.read8byte(input);
        } else {
            this.ID = EID.read4byte(input);
        }
        assert arrays.containsKey(this.ID);

        int length = arrays.get(this.ID).getLength();
        this.DATA = new ArrayList<>(length);

        int i = 0;
        try {
            for (i = 0; i < length; i++) {
                Variable var = Variable.read(input, ctx);
                this.DATA.add(var);
            }
        } catch (IOException ex) {
            throw new IOException(String.format("Error; read %d/%d members.", i, length), ex);
        }
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        Objects.requireNonNull(output);
        
        output.writeESSElement(this.ID);

        for (Variable var : this.DATA) {
            var.write(output);
        }
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 0;
        sum += this.ID.calculateSize();
        sum += this.DATA.stream().mapToInt(var -> var.calculateSize()).sum();
        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    @Override
    public EID getID() {
        return this.ID;
    }

    /**
     * @return The contents of the array.
     */
    public List<Variable> getMembers() {
        return this.DATA;
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        this.DATA.forEach(v -> v.addNames(names, strings));
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        this.DATA.forEach(var -> var.resolveRefs(ess, owner));
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return this.ID.toString() + this.DATA;
    }

    final private EID ID;
    final private List<Variable> DATA;

}
