/*
 * Copyright 2016 Mark Fairchild.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package restringer.ess.papyrus;

import restringer.ess.AnalyzableElement;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;
import java.util.SortedSet;
import restringer.LittleEndianInput;
import restringer.LittleEndianDataOutput;
import restringer.Profile;
import restringer.ess.ESS;
import restringer.ess.Element;
import restringer.ess.Linkable;

/**
 * Describes an array in a Skyrim savegame.
 *
 * @author Mark Fairchild
 * @version 2016/06/21
 */
final public class ArrayInfo implements PapyrusElement, AnalyzableElement, Linkable, HasID {

    /**
     * Creates a new <code>ArrayInfo</code> by reading from a
     * <code>LittleEndianDataOutput</code>. No error handling is performed.
     *
     * @param input The input stream.
     * @param ctx The PapyrusContext.
     * @throws IOException
     */
    public ArrayInfo(LittleEndianInput input, PapyrusContext ctx) throws IOException {
        Objects.requireNonNull(input);
        Objects.requireNonNull(ctx);

        if (ctx.GAME.isSSE() || ctx.GAME.isFO4()) {
            this.ID = EID.read8byte(input);
        } else {
            this.ID = EID.read4byte(input);
        }

        Type t = Type.read(input);
        if (!Arrays.asList(VALID_TYPES).contains(t)) {
            throw new IOException("Invalid ArrayInfo type: " + t);
        }
        this.TYPE = t;

        if (this.TYPE == Type.REF) {
            this.REFTYPE = ctx.STRINGS.read(input);
        } else {
            this.REFTYPE = null;
        }

        this.LENGTH = input.readInt();
    }

    /**
     * @see restringer.ess.Element#write(restringer.LittleEndianDataOutput)
     * @param output The output stream.
     * @throws IOException
     */
    @Override
    public void write(LittleEndianDataOutput output) throws IOException {
        assert null != output;

        output.writeESSElement(this.ID);
        this.TYPE.write(output);

        if (null != this.REFTYPE) {
            this.REFTYPE.write(output);
        }

        output.writeInt(this.LENGTH);
    }

    /**
     * @see restringer.ess.Element#calculateSize()
     * @return The size of the <code>Element</code> in bytes.
     */
    @Override
    public int calculateSize() {
        int sum = 5;
        sum += this.ID.calculateSize();
        if (null != this.REFTYPE) {
            sum += this.REFTYPE.calculateSize();
        }

        return sum;
    }

    /**
     * @return The ID of the papyrus element.
     */
    @Override
    public EID getID() {
        return this.ID;
    }

    /**
     * @return The type of the array.
     */
    public Type getType() {
        assert Arrays.asList(VALID_TYPES).contains(this.TYPE);
        return this.TYPE;
    }

    /**
     * @return The reference type of the array.
     */
    public TString getRefType() {
        return this.REFTYPE;
    }

    /**
     * @return the length of the array.
     */
    public int getLength() {
        return this.LENGTH;
    }

    /**
     * @return The <code>ArrayData</code> for the array.
     */
    public ArrayData getData() {
        return this.data;
    }

    /**
     * Sets the data field.
     *
     * @param newData The new value for the data field.
     */
    public void setData(ArrayData newData) {
        this.data = newData;
    }

    /**
     * @return Short string representation.
     */
    public String toValueString() {
        if (this.TYPE == Type.REF) {
            return this.REFTYPE + "[" + this.LENGTH + "]";
        } else if (this.TYPE == Type.NULL && 0 < this.LENGTH) {
            Type t = this.getData().getMembers().get(0).getType();
            return t + "[" + this.LENGTH + "]";
        } else {
            return this.TYPE + "[" + this.LENGTH + "]";
        }
    }

    /**
     * @see restringer.ess.Linkable#toHTML()
     * @return
     */
    @Override
    public String toHTML() {
        return String.format("<a href=\"array://%s\">%s</a>", this.ID, this);
    }

    /**
     * @return String representation.
     */
    @Override
    public String toString() {
        return this.toValueString() + " " + this.ID + " " + "(" + this.HOLDERS.size() + " owners)";
    }

    /**
     * @see AnalyzableElement#getInfo(restringer.Profile.Analysis,
     * restringer.ess.ESS)
     * @param analysis
     * @param save
     * @return
     */
    @Override
    public String getInfo(Profile.Analysis analysis, ESS save) {
        final StringBuilder BUILDER = new StringBuilder();

        BUILDER.append("<html><h3>ARRAY</h3>");

        if (this.HOLDERS.isEmpty()) {
            BUILDER.append("<p><em>WARNING: THIS ARRAY HAS NO OWNER.</em></p>");
        } else {
            BUILDER.append("<p>Owners:</p><ul>");

            this.HOLDERS.stream().forEach(owner -> {
                if (owner instanceof Linkable) {
                    BUILDER.append(String.format("<li>%s %s", owner.getClass().getSimpleName(), ((Linkable) owner).toHTML()));
                } else {
                    BUILDER.append(String.format("<li>%s %s", owner.getClass().getSimpleName(), owner));
                }
            });

            BUILDER.append("</ul>");
        }

        if (null != analysis) {
            this.HOLDERS.forEach(owner -> {
                if (owner instanceof ScriptInstance) {
                    ScriptInstance instance = (ScriptInstance) owner;
                    SortedSet<String> mods = analysis.SCRIPT_ORIGINS.get(instance.getScriptName());
                    if (null != mods) {
                        String mod = mods.last();
                        TString type = instance.getScriptName();
                        BUILDER.append(String.format("<p>Probably created by script <a href=\"script://%s\">%s</a> which came from mod \"%s\".</p>", type, type, mod));
                    }
                }
            });
        }

        BUILDER.append("<p>");
        BUILDER.append(String.format("ID: %s</p>", this.getID()));
        BUILDER.append(String.format("Content type: %s</p>", this.TYPE));

        if (this.TYPE == Type.REF) {
            final Script SCRIPT = save.getPapyrus().getScripts().get(this.REFTYPE);
            if (null != SCRIPT) {
                BUILDER.append(String.format("Reference type: %s<br/>", SCRIPT.toHTML()));
            } else {
                BUILDER.append(String.format("Reference type: %s<br/>", this.REFTYPE));
            }
        }

        BUILDER.append(String.format("Length: %d<br/>", this.getLength()));
        BUILDER.append("</p>");

        BUILDER.append("</html>");
        return BUILDER.toString();
    }

    /**
     * @see AnalyzableElement#matches(restringer.Profile.Analysis,
     * restringer.Mod)
     * @param analysis
     * @param mod
     * @return
     */
    @Override
    public boolean matches(Profile.Analysis analysis, String mod) {
        Objects.requireNonNull(analysis);
        Objects.requireNonNull(mod);
        return false;
    }

    /**
     * @see PapyrusElement#addNames(restringer.esp.ESPIDMap,
     * restringer.esp.StringTable)
     * @param names The map of IDs to names.
     * @param strings The stringtable.
     */
    @Override
    public void addNames(restringer.esp.ESPIDMap names, restringer.esp.StringTable strings) {
        assert null != this.data;
        this.data.addNames(names, strings);
    }

    /**
     * @return The holder of the array, if there is exactly one. Null otherwise.
     */
    public PapyrusElement getHolder() {
        if (this.HOLDERS.size() == 1) {
            return this.HOLDERS.iterator().next();
        } else {
            return null;
        }
    }

    /**
     * @see PapyrusElement#resolveRefs(ESS, Element)
     * @param ess The full savegame.
     * @param owner The owner of the element, or null if it is not owned.
     */
    @Override
    public void resolveRefs(ESS ess, Element owner) {
        assert null != this.data;
        this.data.resolveRefs(ess, this);

        if (null != this.REFTYPE) {
            this.REFTYPE.addRefHolder(this);
        }
    }

    /**
     * Adds an element as a reference holder.
     *
     * @param newHolder The new reference holder.
     */
    public void addRefHolder(PapyrusElement newHolder) {
        Objects.requireNonNull(newHolder);
        this.HOLDERS.add(newHolder);
    }

    final private EID ID;
    final private Type TYPE;
    final private TString REFTYPE;
    final private int LENGTH;
    final private Collection<PapyrusElement> HOLDERS = new ArrayList<>(1);
    private ArrayData data;

    static final private Type[] VALID_TYPES = new Type[]{
        Type.NULL,
        Type.REF,
        Type.STRING,
        Type.INTEGER,
        Type.FLOAT,
        Type.BOOLEAN};

}
